#include "sigctl.h"
#include <assert.h>
#include <signal.h>

// This is saved by blockSignals and restored by unblockSignals:
static sigset_t oldsignals;

// (Number of calls to blockSingals) - (Number of calls to unblockSignals)
static int sigsBlocked=0;

void blockSignals() {
  if(sigsBlocked==0) {
    sigset_t all;
    sigfillset(&all);
    sigprocmask(SIG_SETMASK,&all,&oldsignals);  
  }
  sigsBlocked--; // Note that signals are blocked when this is executed
}

void unblockSignals() {
  sigsBlocked++; // Note that signals are blocked when this is executed
  assert(sigsBlocked<=0);
  if(sigsBlocked==0) {
    sigprocmask(SIG_SETMASK,&oldsignals,0);
  }
}

// Define array with user signal handlers:

const int maxSignalHandlers=10;

struct signalHandler {
  void* parameter;
  signal_handler func;
  signalType type;
};    
  
static signalHandler signalHandlers[maxSignalHandlers];
static int signalHandlerc=0;
static int hinstalled=0; // Is the signal handler of this module installed?

void sigHandler(int signal) {  
  // Call user functions user signal handlers:
  int handeld=0;
  for(int a=0; a<signalHandlerc; a++) {  
    if((signalHandlers[a].type==signalHup && signal==SIGHUP) ||
       (signalHandlers[a].type!=signalHup && signal!=SIGHUP)) 
      // If this signal handler will handle this signal:
      signalHandlers[a].func(signalHandlers[a].parameter);
  }
  if(signal==SIGHUP) return; // Don't terminate on this signal
  
  // And call the default handler to terminate the process:
  
  // First install default signal handler:
  struct sigaction sa;
  sa.sa_handler=SIG_DFL;
  sigfillset(&sa.sa_mask);
  sa.sa_flags=0;
  sigaction(signal,&sa,0);
  
  // Second raise the signal again:
  raise(signal);
}  
  
void registerSignalHandler(void* f, signal_handler ef, signalType type) {
  blockSignals();

  assert(signalHandlerc<maxSignalHandlers);
  signalHandlers[signalHandlerc].parameter=f;
  signalHandlers[signalHandlerc].func=ef;
  signalHandlers[signalHandlerc].type=type;
  signalHandlerc++;
  
  if(!hinstalled) {
    hinstalled=1;
    
    struct sigaction sa;
    sa.sa_handler=sigHandler;
    sigfillset(&sa.sa_mask);   // Block all signals while handling a signal
    sa.sa_flags=0;
    
    // Install out siganl handler for appropriate signals:
    char signals[]={SIGHUP,SIGINT,SIGQUIT,SIGABRT,SIGTERM};
    for(int a=0; a<sizeof(signals); a++) {
      sigaction(signals[a],&sa,0);
    }    
  }
  
  unblockSignals();
}

void unregisterSignalHandler(void* f, signal_handler ef) {
  blockSignals();
  for(int a=0; a<signalHandlerc; a++) 
    if(signalHandlers[a].parameter==f && signalHandlers[a].func==ef) {
      signalHandlers[a]=signalHandlers[signalHandlerc-1];
      signalHandlerc--;
      unblockSignals();
      return;
    }
  unblockSignals();
  assert(0);
}
