#include "common.h"
#include "config.h"
#include "logfile.h"
#include "pcapintf.h"

#include <netdb.h>
#include <string.h>
#include <unistd.h>

char* PcapInterface::init(char* strfilter, char* netdev, LogFile* l) {
  lf=l;
  
  // Find network device to use:
  if(!netdev) {
    netdev=pcap_lookupdev(errmsg);
    if(!netdev)  return errmsg;
  }
  snprintf(interface,sizeof(interface),"%s",netdev);
  
  // Open the device:
  ph=pcap_open_live(netdev,78,1,-1,errmsg); 
    // 14  : For MAC header
    // 4*15: Maxmimal size of IP header
    // 2*2 : Used part of transport header
    // ---
    //  78

  if(!ph) { interface[0]='\x0'; return errmsg; }
  
  // Read the datalink type and check if it is supported:
  datalink=pcap_datalink(ph);
  if(datalink!=DLT_EN10MB) { // Currently only 10MB ethernet supported
    pcap_close(ph);
    snprintf(errmsg,sizeof(errmsg),"Pcap datalink layer #%i not support. Please add support for it.",datalink);
    return errmsg;
  }                   
  
  // Set filter if specified:
  if(strfilter) {
    // Compile the filter:
    int i=pcap_compile(ph,&filter,strfilter,1,0xffffffff);
    if(i==-1) { pcap_close(ph); ph=0; return "Syntax error in filter"; }
  
    // An attach it to the device:
    i=pcap_setfilter(ph,&filter);
    if(i==-1) { pcap_close(ph); ph=0; return "Syntax error in filter"; }
  }
  
  return 0; // On succes
}

char* PcapInterface::nextPacket(Packet& p) {
  for(;;) {
    // Read the next packet:
    struct pcap_pkthdr header;
    const unsigned char* data=pcap_next(ph,&header);
    if(!data) continue;
    
    // Set the packet time:
    p.time.tv.tv_sec=header.ts.tv_sec;
    p.time.tv.tv_usec=header.ts.tv_usec;
    
    // And interpret it
    if(datalink==DLT_EN10MB) {
      mac_header* eth=(mac_header*)data;
  
      // Ignore packet if not IP packet
      if(ntohs(eth->protocol)!=MAC_PROTO_IP4V) continue;
      
      // Complain and ignore header has invalid checksum:
      ip_header* ip=(ip_header*)(eth+1);
      u16 header_len=(ip->ver_hdrlen & 0xf)*4;
      u16 c=0;
      u16* ipu=(u16*)ip;
      u16 orgchecksum=ip->checksum; ip->checksum=0;
      for(int a=0; a<(header_len>>1); a++, ipu++) {
        u16 oldc=c;	
        c+=(*ipu); if(c<oldc) c++;	  
      }
      c=~c;
      if(c!=orgchecksum) {
        lf->log(LogFile::badpacket,"Packet from %s has wrong IP checksum and is ignored",p.maca_src.tostr());
	continue;
      }
	
      // Copy addresses:
      p.maca_dst=eth->dst; 
      p.maca_src=eth->src; 
      p.ip_src=ip->src; 
      p.ip_dst=ip->dst;
      p.proto=ip->protocol;
      tcp_header* tcp=(tcp_header*)(((u8*)ip)+header_len);
      p.port_src=ntohs(tcp->src_port);
      p.port_dst=ntohs(tcp->dst_port);

      // CONFIGURABLE: Use MAC instead of IP size of packets.
      //
      // If you want to base the statistics on the length of the 
      // whole packet including the datalink layer information. 
      // you should replace the line below with this line:
      //   p.ip_len=header.len; 
      p.ip_len=ntohs(ip->len);     
    }    
    return 0;
  }    
}

void PcapInterface::printPacket(char* buf, int buflen, const Packet& p) {
  char macbuf[42];

  snprintf(buf,buflen,"%s(%s.%i) -> ",
    p.maca_src.tostr(),p.ip_src.tostr(),p.port_src);

  int plen=strlen(buf);
  buflen-=plen;
  
  snprintf(buf+plen,buflen,"%s(%s.%i)",
    p.maca_dst.tostr(),p.ip_dst.tostr(),p.port_dst);
}

