#include "common.h"
#include "sigctl.h"
#include "config.h"
#include "logfile.h"

#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/param.h>

LogFile::LogFile()  { 
  h=-1; filename=0;
  registerSignalHandler(this,termFunc,signalTerm);
  registerSignalHandler(this,hupFunc,signalHup);
}

LogFile::~LogFile() { 
  unregisterSignalHandler(this,termFunc); 
  unregisterSignalHandler(this,hupFunc); 
  delete[] filename;
  if(h!=-1) close(h);
}

void LogFile::termFunc(void *l) {
  ((LogFile*)l)->log(status,"Server terminated because of received signal");
}

void LogFile::hupFunc(void *lo) {
  LogFile* l=(LogFile*)lo;
  
  // Reopen logfile:
  if(l->h==-1) return;
  l->log(status,"HUP signal received, closing logfile");
  close(l->h);
  l->h=open(l->filename,O_CREAT|O_WRONLY|O_APPEND,
       S_IRUSR|S_IWUSR|
       S_IRGRP|S_IWGRP|
       S_IROTH);
  l->log(status,"HUP signal received, logfile reopened");
}

char* LogFile::init(Config* cnf, int ps) {
  prnstderr=ps;

  blockSignals(); // Don't allow a SIGHUP signal to reopen the logfile now
  h=open(cnf->logfile(),O_CREAT|O_WRONLY|O_APPEND,
    S_IRUSR|S_IWUSR|
    S_IRGRP|S_IWGRP|
    S_IROTH);
    
  if(h==-1) {
    snprintf(errmsg,sizeof(errmsg),"Could not create logfile %s: %s",cnf->logfile(),strerror(errno));
    unblockSignals();
    return errmsg;
  }
  
  filename=new char[strlen(cnf->logfile())+1];
  strcpy(filename,cnf->logfile());

  logaddrchange=cnf->logaddrchange();
  logbadpackets=cnf->logbadpackets();
  
  unblockSignals();
  
  return 0;
}

void LogFile::log(logTypes t, const char* format,...) {
  blockSignals();
  if((h!=-1) &&
     (((t==addrchange) && logaddrchange) ||
      ((t==badpacket) && logbadpackets) ||
      (t==status) || (t==print))) {
     
    char buf[256];
    va_list ls;
    va_start(ls,format);
    vsnprintf(buf,sizeof(buf),format,ls);
    va_end(ls);
    
    char buf1[256];
    Time t; gettimeofday(&t.tv,0);
    sprintf(buf1,"%s: %s\n",t.tostr(),buf);
    write(h,buf1,strlen(buf1));

    if(prnstderr) {
      fprintf(stderr,buf1);
    }
  }
  unblockSignals();
}

void LogFile::closeAllOtherFiles() {
  blockSignals();
  for(int a=0; a<NOFILE; a++) if(a!=h) close(a);
  unblockSignals();
}
 