// This module is used by both the deamon and the client processes.
// It defines common datastructures and the contest for the shared memory
// area.

#include <arpa/inet.h>
#include <sys/time.h>
#include <limits.h>
#include <math.h>
#include <assert.h>
#include <string.h>

#include "netstructs.h"

// Used everywhere to save error messages returned from functions:
extern char errmsg[256];

//-----------------------------------------------------------------------------
// Generel non-applications specific structurs

// This represents a time in GMT format in seconds and microseconds sincde1970.
struct Time {
  timeval tv;
  
  Time() { tv.tv_sec=0; tv.tv_usec=0; }

  int operator==(const Time& t) const { return tv.tv_sec==t.tv.tv_sec &&
                                               tv.tv_usec==t.tv.tv_usec; }
					       
  int operator<=(const Time& t) const { 
    return tv.tv_sec<t.tv.tv_sec || (tv.tv_sec==t.tv.tv_sec && tv.tv_usec<=t.tv.tv_usec);
  }

  int operator>=(time_t t) const { 
    return tv.tv_sec>=t;
  }
  
  // Operators:
  
  // Adds a number of seconds:  
  Time& operator+=(long s) {
    tv.tv_sec+=s;
    return *this;
  }    
  
  Time& operator-=(const Time& t) {
    tv.tv_sec-=t.tv.tv_sec;
    tv.tv_usec-=t.tv.tv_usec;
    if(tv.tv_usec<0) {
      tv.tv_usec+=1000000;
      tv.tv_sec-=1;
    }
  }

  Time operator-(const Time& t) {
    Time t1=*this;
    t1-=t;
    return t1;
  }
  
  // Other functions:

  int isZero() const {
    return tv.tv_sec==0 && tv.tv_usec==0;
  }
    
  double asDouble() {
    return tv.tv_sec+tv.tv_usec/(1000000.0);
  }

  const char* tostr() const;
};

// We are now going to define a 64 bit signed integer type named int64
#ifdef __GNUC__
// In GNU C the type is defined for us already!
#include <sys/types.h>
typedef int64_t int64;

#else  
  // We define it ourself.
  
  // The template below defines a 2 complement integers consisting of two
  // integers. This should work even if the computer does not use
  // 2 complement arithmitic for signed numbers.
  // T should be unsigned integer type
  // SignedT should be the signed type of the same integer type.
  template<class T, class SignedT> class DoubleInt {
    DoubleInt(T a, T b) : ls(a), ms(b) {}
    
    // Sign of number:
    static signed char sign(T t) {
      const T_bits=(CHAR_BIT)*sizeof(T);
      return (1<<(T_bits-1)) & t ? -1 : 1;
    }
    
    // Inverts the sign:
    int negate() {
      ms=~ms;
      ls=~ls;
      (*this)+=1;
    }
    
    // Unsigned operators:
    int unsigned_lesseq(const DoubleInt& b) {
      return ms<b.ms || (ms==b.ms && ls<=b.ls);
    }
    void unsigned_shr() {
      const T_bits=(CHAR_BIT)*sizeof(T);
      ls>>=1;
      if(ms & 1) ls|=(1<<(T_bits-1));
      ms>>=1;
    }
  
    // Number of used bits:
    int NumBit() const {
      const T_bits=(CHAR_BIT)*sizeof(T);
      if(ms==0) {
        int a;
        T b=ls;
        for(a=0; b!=0; a++) b>>=1;
        return a;
      } else {
        int a;
        T b=ms;
        for(a=0; b!=0; a++) b>>=1;
        return a+T_bits;
      }
    }
        
    T ls; // Most significant part
    T ms; // Least significant part

    public:
    
    DoubleInt() {}
    DoubleInt(SignedT a) { (*this)=a; }
    DoubleInt(T a) { (*this)=a; }
    DoubleInt(const T& a) { ls=a.ls; ms=a.ms; }
    
    DoubleInt& operator = (SignedT a) { 
      if(a<0) {
        ls=-a; ms=0; negate();
      } else {
        ls=a; ms=0;
      }
      return *this;
    }

    DoubleInt& operator = (T a) { 
      ls=a; ms=0;
      return *this;
    }

    operator double() {
      // For some reason GNU C++ seems to break with a core dump from wiplc 
      // when I use pow in the code below - so I don't.
      const T_bits=(CHAR_BIT)*sizeof(T);
      DoubleInt a=*this;
      signed char c=sign(a);
      if(c<0) a.negate();
      double res=double(a.ls)+double(a.ms)*(1U<<T_bits-1)*2.0;
      return c<0 ? -res : res;
    }
    
    operator T() {
      return ls;
    }
    
    operator SignedT() {
      return (SignedT)ls;
    }
    
    DoubleInt& operator+=(const DoubleInt& b) {        
      T ols=ls;
      ls+=b.ls;    
      ms+=b.ms;      
      if(ls<ols) ms++;
      return *this;
    }
    
    DoubleInt operator+(const DoubleInt& b) const {
      DoubleInt a=*this;
      return a+=b;
    }
  
    DoubleInt& operator-=(const DoubleInt& b) {
      T ols=ls;
      ls-=b.ls;
      ms-=b.ms;      
      if(ls>ols) ms--;
      return *this;
    }
    
    DoubleInt operator-(const DoubleInt& b) const {
      DoubleInt a=*this;      
      return a-=b;
    }

    int operator==(const DoubleInt& b) const {
      return (ms==b.ms && ls==b.ls);
    }

    int operator!=(const DoubleInt& b) const {
      return (ms!=b.ms || ls!=b.ls);
    }    

    int operator<=(const DoubleInt& b) const {
      signed char sa=sign(ms);
      signed char sb=sign(b.ms);
      if(sa==sb) // Same sign
	return ms<b.ms || (ms==b.ms && ls<=b.ls);
      else 
        return sa<sb;      
    }

    int operator<(const DoubleInt& b) const {
      signed char sa=sign(ms);
      signed char sb=sign(b.ms);
      if(sa==sb) // Same sign
	return ms<b.ms || (ms==b.ms && ls<b.ls);
      else 
        return sa<sb;      
    }
    
    DoubleInt& operator >>= (int i) {
      if(i<=0) return *this;
      const T_bits=(CHAR_BIT)*sizeof(T);
      T neg=ms & (1<<(T_bits-1));
      for(;i>0 ; i--) {
	ls>>=1;
	if(ms & 1) ls|=1<<(T_bits-1);
	ms>>=1;
	ms|=neg;
      }
      return *this;
    }
    
    DoubleInt operator >> (int i) {
      DoubleInt a=*this;
      return a>>=i;
    }

    DoubleInt operator << (int i) const {
      if(i<=0) return *this;
      const T_bits=(CHAR_BIT)*sizeof(T);
      if(i>=T_bits) 
        return DoubleInt(0,ls<<(i-T_bits));
      else
        return DoubleInt(ls<<i, (ms<<i)|((ls>>(T_bits-i)) & ((1<<i)-1) ));
    }
    
    DoubleInt& operator <<= (int i) {
      return *this=(*this)<<i;
    }
    
    DoubleInt& operator*=(DoubleInt b) {
     //  If we were an unsigned integer:
      //   If type T had values in [0;99] we have:
      //      a*b
      //    = (a.ms*100+a.ls)*(b.ms*100+b.ls)
      //   = (a.ms*b.ms)*10000+(a.ms*b.ls+a.ls*b.ms)*100+(a.ls*b.ls)
      //   Since we are calculating modulo 10000 the only problem left is to
      //   calculate a.ls*b.ls.
      //   If we write:
      //     a.ls=a1*10+a2 and b.ls=b1*10+b2
      //   Where a1,a2,b1 and b2 are in interval [0;9], then we have:
      //     a.ls*b.ls=(a1*b1)*100+(a1*b2+a2*b1)*10+(a2*b2)
      //   Where non of the multiplications on the right will overflow.

      const T_bits2=(CHAR_BIT)*sizeof(T)/2;
      DoubleInt a=*this;

      // Handle sign:
      signed char sgn=1;
      if(sign(a.ms)<0) {
        sgn=-1; a.negate();
      }
      if(sign(b.ms)<0) {
        sgn*=-1; b.negate();
      }
    
      // Calculate a.ls*b.ls into this object:
      T a1=a.ls>>T_bits2;
      T a2=a.ls & ((1<<T_bits2)-1);
      T b1=b.ls>>T_bits2;
      T b2=b.ls & ((1<<T_bits2)-1);
    
      ms=0;
      ls=a1*b2;
      *this+=a2*b1;
      *this=*this << T_bits2;
      *this+=a2*b2;
    
      // And make the rest of the result:
      ms+=a.ms*b.ls+a.ls*b.ms+a1*b1;
    
      if(sgn<0) negate();
      return *this;
    }
    
    DoubleInt operator*(const DoubleInt& a) const {
      DoubleInt r=*this;
      return r*=a;
    }
  
    DoubleInt& operator/=(const DoubleInt& B) {
      DoubleInt a=*this;
      DoubleInt b=B;
      const T_bits=(CHAR_BIT)*sizeof(T);
      
      // Handle sign
      signed char sgn=1;
      if(sign(a.ms)<0) {
        sgn=-1; a.negate();
      }
      if(sign(b.ms)<0) {
        sgn*=-1; b.negate();
      }
    
      // We shall calculate a/b. We make a bitwise division:
      int bnmb=b.NumBit();
      assert(bnmb!=0); // Don't allow division by zero
      int l=a.NumBit()-bnmb;
      ms=ls=0; 
      if(l<0) { return *this; } // If b as more bits than a
      b=b<<l;
      for(int i=0; i<=l; i++) {
        if(i!=0) { *this<<=1; b.unsigned_shr(); }
        if(b.unsigned_lesseq(a)) {
          ls|=1;
          a-=b;	
        }
      }
      
      if(sgn<0) negate();
      return *this;
    }
    
    DoubleInt operator/(const DoubleInt& a) const {
      DoubleInt r=*this;
      return r/=a;
    } 
    
    DoubleInt operator &(const DoubleInt& a) const {
      return DoubleInt(ls & a.ls,ms & a.ms);
    }

    DoubleInt& operator &=(DoubleInt& a) {
      ls&=a.ls;
      ms&=a.ms;
      return *this;
    }
  };

  // We use two plain integers to define our 64 bit integer and hope
  // they are at least 32 bits each. If they are 64 bit we get 128 bit counters,
  // but that shouldn't hurt...    

  typedef DoubleInt<unsigned int, signed int> int64;
#endif

// Supporting functions:
extern int int64_compare(const void* a,const void* b);

//-----------------------------------------------------------------------------
// Generel application specific data structurs:

// A MAC address
class mac_addr {
  mac_addr_t ea;
  
  public:    
//  mac_addr() {}
//  mac_addr(const mac_addr_t& e) : ea(e) {}

//  mac_addr& operator = (const mac_addr& m) { memcpy(ea.addr,m.ea.addr,sizeof(ea.addr)); return *this; }
  mac_addr& operator = (const mac_addr_t& m) { memcpy(ea.addr,m.addr,sizeof(ea.addr)); return *this; }
  mac_addr& operator = (const unsigned char* m) { memcpy(ea.addr,m,sizeof(ea.addr)); return *this; }

//  operator const ether_addr*() const { return &ea; }
  int operator == (const mac_addr& m) const { return !memcmp(ea.addr,m.ea.addr,sizeof(ea.addr)); }
  int operator == (const unsigned char* m) const { return !memcmp(ea.addr,m,sizeof(ea.addr)); }
  int operator != (const mac_addr& m) const { return memcmp(ea.addr,m.ea.addr,sizeof(ea.addr)); }
  
  const char* tostr() const;
  static int getmaxlen_tostr(); // maximal length of returned value from tostr  
  int fromstr(const char*); // Returns 0 on succes  

  static inline int compare(const mac_addr& a, const mac_addr& b) { return a!=b; }  
  static inline int compare(const void* a, const void* b) { return compare(*((mac_addr*)a),*((mac_addr*)b)); }
  
  void setZero();
  int isBrdCast();
};

// An IP address:
struct ip_addr {
  in_addr ia;
  
  public:
//  ip_addr() {}
//  ip_addr(const in_addr& i) : ia(i) {}
//  operator in_addr() const { return ia; }
  int operator == (const in_addr& i) const { return ia.s_addr==i.s_addr; }
  int operator == (const ip_addr& i) const { return ia.s_addr==i.ia.s_addr; }
  int operator != (const ip_addr& i) const { return ia.s_addr!=i.ia.s_addr; }
//  ip_addr& operator = (const ip_addr& i) { ia.s_addr=i.ia.s_addr; return *this; }
  ip_addr& operator = (const in_addr& i) { ia.s_addr=i.s_addr; return *this; }
  ip_addr& operator = (unsigned int i) { ia.s_addr=i; return *this; }

  const char* tostr(int comp=4) const;    // comp=Number of components
  static int getmaxlen_tostr(int comp=4); // maximal length of returned value from tostr  
  int fromstr(const char*); // Returns 0 on succes

//  static inline int compare(const ip_addr& a, const ip_addr& b) { return ntohl(a.ia.s_addr)-ntohl(b.ia.s_addr); }
  static inline int compare(const ip_addr& a, const ip_addr& b) { return memcmp(&a.ia.s_addr,&b.ia.s_addr,sizeof(a.ia.s_addr)); }
  static inline int compare(const void* a, const void* b) { return compare(*((ip_addr*)a),*((ip_addr*)b)); }
  
  unsigned asUnsigned() { return ia.s_addr; }
  
  void setZero();
};

// An identification of a card:
union card_id {
  ip_addr  ipaddr; 
  mac_addr macaddr;

  static inline int compare(const card_id& a, const card_id& b) { return memcmp(&a,&b,sizeof(card_id)); }
  static inline int compare(const void* a, const void* b) { return memcmp(&a,&b,sizeof(card_id)); }
}; 
enum card_id_type { idIP,idMAC,idUNKNOWN };

//-----------------------------------------------------------------------------
// Format of the shared memory area:

const int maxcounterc=32;
const int MaxCounterNameLen=10;

// Information about a single network card.
struct CardInfo {
  // If in a mode where indexing after MAC adress the MAC
  // adresse field is always valid - if indexing on IP adress
  // the IP adresse field is always valid. The other one
  // is then the latest used address:
  mac_addr macaddr; 
  ip_addr  ipaddr;
  
  int64 Counters[maxcounterc]; // All the counters.  

  // Subtracts all the counters in ci from ourself. 
  // counterc should indicate the number of used counters.
  CardInfo& sub(CardInfo& ci,int counterc) {
    assert(macaddr==ci.macaddr);    
    for(int a=0; a<counterc; a++) {
      Counters[a]-=ci.Counters[a];
    }
    return *this;
  }

  // Adds the counters:
  CardInfo& add(const CardInfo& ci,int counterc) {     
    for(int a=0; a<counterc; a++) {
      Counters[a]+=ci.Counters[a];
    }
    return *this;
  }

  // Are all the counters 0?
  int isZero() const {
    for(int a=0; a<maxcounterc; a++) if(Counters[a]!=0) return 0;
    return 1;
  }
  
  // Set all the counters to zero:
  void setZero() {
    for(int a=0; a<maxcounterc; a++) Counters[a]=0;
  }
  

  // These functions are used to do sorting:  
  static int reverse; // Reverse sorting? Should be left 0 after sorting
  
  static inline int rev(int a) { return reverse ? -a : a; }
  
  static int compare_mac(const void* a,const void* b) {
    return rev(mac_addr::compare(((CardInfo*)a)->macaddr,((CardInfo*)b)->macaddr));
  }
  
  static int compare_ip(const void* a,const void* b) {
    return rev(ip_addr::compare(((CardInfo*)a)->ipaddr,((CardInfo*)b)->ipaddr));
  }  

  static int counter; // Sort by the given counter
  static int compare_counter(const void* a,const void* b) {
    return rev(int64_compare(&((CardInfo*)a)->Counters[counter],
                             &((CardInfo*)b)->Counters[counter]));
  }
};

// This is the shared memory area:
struct ServerMem {
  // Number of counters updated by this daemon.
  int Counterc;  
  
  // The names of the different counters:
  char CounterNames[maxcounterc][MaxCounterNameLen+1];
  
  Time start;    // The start time of the daemon.

  int maxcardc;  // Maximal number of cards to save information about.
  int curcardc;  // Number of cards we have currently.
  
  card_id_type cit; // How we index the table - is never idUNKNOWN

  int proxymaxcon;      // If this is difference from 0 wipl support proxyremapping
                        // and this is the maximal number of connection.

  CardInfo cards[0]; // An array of maxcards elements with all our cards.
                     // These must be sorted by the MAC address
  
  // If proxyremap is true here follow a memory area with the remapping
  // information. This is maintained by the proxydict.c module	     
};

// Default filenames:
#define DEFAULT_LOGFILE    "/var/log/wipld"
#define DEFAULT_CNFFILE    "/etc/wipld.conf"
#define DEFAULT_PIDFILE    "/var/run/wipld.pid"
#define DEFAULT_DAEMONFILE "/etc/wipld.conf"
