/*
   Perrotcp - TCP logging routines
   --------------------------------------------------------------------
   Perro - The Internet Protocols logger

   Copyright (C) 1998, 1999, 2000 Diego Javier Grigna <diego@grigna.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "common.h"

void usage( void);

int main( int argc, char *argv[])
{
 FILE *fperro;         /* Simple log file pointer            */
 FILE *fpraw;          /* Raw log file poinetr               */
 char buf[     65535]; /* The buffer we use to read a packet */
 struct perro_iphdr  *iph;  /* IP header pointer             */
 struct perro_tcphdr *tcph; /* TCP header pointer            */
 struct in_addr addrs;
 struct in_addr *mask;       /* Mask to apply for IP addresses to ignore */
 struct in_addr *ignoremask; /* Mask to apply for ignored packets        */
 struct tm *ltm;       /* Used by localtime(3)                      */
 time_t t;             /* Used by time(2)                           */
 char *ti;             /* Used by ctime(3)                          */
 int flag_mask;        /* Only log packets that matches tcpflagmask */
 PERRO_U8 tcpflagmask; /* The TCP flag mask (to ignore packets)     */
 int ch;               /* For command line parsing                  */
 int iphlen;           /* IP header length (Used to skip            */
                       /* options)                                  */
 int alloc_ndx;        /* Index of the last allocated mask          */
 int allocated;        /* Number of allocated (struct in_addr)      */
 signed short *ignoreport; /* Array of ignored ports.                 */
 int palloc_ndx;       /* The same that alloc_ndx, but for ignoreport */
 int pallocated;       /* ditto.                                      */
 int i;

 progname = perro_basename( strdup( argv[ 0]));

 if( argc < 2)
     usage();

 perro_init();

 flag_mask   = 0;
 tcpflagmask = 0;
 alloc_ndx   = 0;
 allocated   = 0;
 palloc_ndx  = 0;
 pallocated  = 0;
 ignoreport  = NULL;
 mask        = NULL;
 ignoremask  = NULL;

 /* Parse command line */

 while(( ch = getopt( argc, argv, "qlwrf:i:p:")) != EOF){
         switch( ch) {
                case 'q': flag_quiet = 1;
                          break;
                case 'l': flag_log   = 1;
                          break;
                case 'w': flag_raw   = 1;
                          break;
                case 'r': flag_dont_resolve = 0;
                          break;
                case 'f': flag_mask = 1;
                          tcpflagmask = atoi( optarg);
                          break;
                case 'i': if( alloc_ndx >= allocated) {
                              allocated += 4;
                              mask       = ( struct in_addr *)
                                             realloc( mask, 
                                                      sizeof( struct in_addr)
                                                      * allocated);
                              ignoremask = ( struct in_addr *)
                                             realloc( ignoremask,
                                                      sizeof( struct in_addr)
                                                      * allocated);
                          }
                          process_host_and_mask( optarg
                                               , &mask[ alloc_ndx]
                                               , &ignoremask[ alloc_ndx]);
                          alloc_ndx++;
                          break;
                case 'p': if( palloc_ndx >= pallocated) {
                              pallocated += 4;
                              ignoreport = ( signed short *)
                                             realloc( ignoreport,
                                                      sizeof( signed short)
                                                      * pallocated);
                          }
                          ignoreport[ palloc_ndx] = atoi( optarg); 
                          palloc_ndx++;
                          break;
                default : usage();
         }
 }

 if( !flag_log && !flag_raw)
     usage();

 open_socket( IPPROTO_TCP);

 show_version( "perrotcp (TCP logger)");

 gobackground();
 init_signals();

 if( flag_log) {
     fperro = fopen( PER_TCP_LOG, "a+");

     t  = time( NULL);
     ti = ctime( &t);

     /* Write log headings */
     fprintf( fperro, "%s\n", perro_hyphen);
     fprintf( fperro, "Perro TCP logger - Begins at %-24.24s\n", ti);
     fprintf( fperro, "%s\n", perro_hyphen);
     if( alloc_ndx != 0) {
         fprintf( fperro, "Ignore hostname/mask enabled.\n");
         for( i = 0; i < alloc_ndx; i++) {
              fprintf( fperro, "%-5d: Mask            : %s\n", i, inet_ntoa( mask[       i]));
              fprintf( fperro, "%-5d: Ignore host mask: %s\n", i, inet_ntoa( ignoremask[ i]));
         }
         fprintf( fperro, "%s\n", perro_hyphen);
     }
     fprintf( fperro, "-      Date      -   Source IP   -     Domain Name     -dst port- Service name -\n");
     fprintf( fperro, "%s\n", perro_hyphen);

     fclose( fperro);
 }

 while( 1) {
again:
        if( read( socket_fd, buf, sizeof( buf)) <= 0)
            continue;

        /* Cast the IP header */
        iph = ( struct perro_iphdr *) buf;

        /* Do we ignore some IP addresses? */
        if( alloc_ndx != 0) {
            /*
             * Take the IP source address
             * 'and' it with the mask
             * and if it is equal to the ignore mask
             * ignore the packet
             */

            addrs.s_addr = iph->saddr;

            for( i = 0; i < alloc_ndx; i++)
                 if( (addrs.s_addr & mask[ i].s_addr) == ignoremask[ i].s_addr)
                      goto again;
        }

        /* Skip IP options */
        iphlen  = ( iph->hlv & 0x0F) << 2;
        iphlen -= 20;

        /* Cast the TCP header */
        tcph = ( struct perro_tcphdr *) &buf[ sizeof( struct perro_iphdr) + iphlen];

        if( flag_mask && ((tcph->th_flags | tcpflagmask) != tcpflagmask))
            continue;

        /* Do we ignore some TCP destination port? */
        if( palloc_ndx != 0) {
            for( i = 0; i < palloc_ndx; i++)
                 if( htons( tcph->th_dport) == ignoreport[ i])
                     goto again;
        }

        t = time( NULL);

        /* Make a simple human readeable log */
        if( flag_log) {

            fperro = fopen( PER_TCP_LOG, "a+");

            ltm = localtime( &t);

            addrs.s_addr = iph->saddr;

            fprintf( fperro, "%02d:%02d:%02d %02d/%02d/%02d %-15.15s  %-19.19s    %5d   %-12.12s\n"
                             , ltm->tm_hour     , ltm->tm_min , ltm->tm_sec
                             , ltm->tm_mon + 1  , ltm->tm_mday
                             , ltm->tm_year >= 100 ?
                               ltm->tm_year - 100 : ltm->tm_year
                             , inet_ntoa( addrs), resolve_host_name( iph->saddr)
                             , htons( tcph->th_dport), get_serv_name( tcph->th_dport, "tcp"));

            fclose( fperro);
        }

        /* Dump time + raw data */

        if( flag_raw) {
            fpraw = fopen( PER_TCP_RAW, "a+");
            fwrite(  &t,           sizeof( t), 1, fpraw);
            fwrite( buf, htons( iph->tot_len), 1, fpraw);
            fclose( fpraw);
        }

 } /* end while( 1) */

 close( socket_fd);

 return 0;
}

void usage( void)
{
 fprintf( stderr, "\nPerro Release %s - perrotcp (TCP logger)\n\n"
                  "         by  Diego J. Grigna (diego@grigna.com)\n\n"
                  "Usage:\n%s [-q] [-lwr] [-f flagmask] [-i hostname[/mask]] "
                  "[-p port]\n"
                  "\t-q\t Quiet mode, don't send output to stdout.\n"
                  "\t-l\t Make a simple human readeable log\n"
                  "\t-w\t Log time + raw data\n"
                  "\t-w\t Log time + raw data\n"
                  "\t-r\t Resolve domain names\n"
                  "\t-f\t Only log packets that matches flagmask\n"
                  "\t-i\t Ignore packets from hostname/mask\n"
                  "\t-p\t Ignore packets with TCP destination port = port\n\n"
                  , PER_VERSION, progname);

 exit( -1);
}

