/*==============================================================================
| netl compiler
|   by Graham THE Ollis <ollisg@netl.org>
|
|   Copyright (C) 1997 Graham THE Ollis <ollisg@netl.org>
|
|   This program is free software; you can redistribute it and/or modify
|   it under the terms of the GNU General Public License as published by
|   the Free Software Foundation; either version 2 of the License, or
|   (at your option) any later version.
|
|   This program is distributed in the hope that it will be useful,
|   but WITHOUT ANY WARRANTY; without even the implied warranty of
|   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|   GNU General Public License for more details.
|
|   You should have received a copy of the GNU General Public License
|   along with this program; if not, write to the Free Software
|   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
|
|  Date       Name	Revision
|  ---------  --------  --------
|  18 Jul 99  G. Ollis	created module
|  03 Jul 99  G. Ollis	added support for /x bitmasks on dstip= type rules
|=============================================================================*/

#include <stdio.h>
#include <netinet/in.h>

#include "netl/global.h"

#include "netl/action.h"
#include "netl/filter.h"
#include "netl/ether.h"
#include "netl/ip.h"
#include "netl/config.h"
#include "netl/filter.h"
#include "netl/resolve.h"

/*==============================================================================
| generate_c
| + given the current configuration, attempt to generate c code for a single
|   netl module which will do the equivalent thing, but faster.
| + one exception is the input module.  that will always be specified run time
|   and dynamically.
|=============================================================================*/

char *int2prot[4] = { "TCP", "UDP", "ICMP", "IGNP" };
char *int2act[7] = { "NONE", "LOG", "DUMP", "IGNORE", "DL", "OTHER", "NULL" };
char *prot2logtype[6] = { "ip_with_port", "ip_with_port", 
			"ip_no_port", "ip_no_port",
			"unknown", "ip_unknown" };

void
generate_c(FILE *fp)
{
	int i, n, prot = -1;


	/* #####################################################################
	| GENERATED CODE
	######################################################################*/

	fprintf(fp, "

/* this file generated by netl */

#include <unistd.h>
#include <stdio.h>
#include <time.h>

#include \"netl/global.h\"
#include \"netl/ether.h\"
#include \"netl/ip.h\"
#include \"netl/action.h\"
#include \"netl/filter.h\"
#include \"netl/config.h\"
#include \"netl/io.h\"
#include \"netl/resolve.h\"

struct configlist req;

void log_ip_with_port(u8 *dg, size_t len, char *extra_string, char* logname);
void log_ip_no_port(u8 *dg, size_t len, char *extra_string, char* logname);
void log_non_ip(u8 *dg, size_t len, char *extra_string, char* logname);
void log_ip_unknown(u8 *dg, size_t len, char *extra_string, char* logname);
void log_unknown(u8 *dg, size_t len, char *extra_string, char* logname);
char *dump(u8 *dg, char *logname, size_t len);

");
	for(i=0; i<num_acts; i++) {
		if(acts[i].action_code == ACTION_USER) {
			fprintf(fp, "action_mod *%s_am;\n", acts[i].name);
		}
	}
	for(i=0; i<num_filters; i++) {
		if(filters[i].filter_code == PROT_USER) {
			fprintf(fp, "filt_mod *%s_fm;\n", filters[i].name);
		}
	}
	fprintf(fp,"

/* construct() */

void
construct(void)
{
");

	/* #####################################################################
	| END GENERATED CODE
	######################################################################*/

	/*======================================================================
	| the constructor must load up any action/filter modules.
	|=====================================================================*/

	for(i=0; i<num_acts; i++) {
		if(acts[i].action_code == ACTION_USER) {
			fprintf(fp, "\t%s_am = lookup_act(\"%s\", ACTION_USER);\n",
				acts[i].name, acts[i].name);
		}
	}
	for(i=0; i<num_filters; i++) {
		if(filters[i].filter_code == PROT_USER) {
			fprintf(fp, "\t%s_fm = lookup_filter(\"%s\", PROT_USER);\n", 
				filters[i].name, filters[i].name);
		}
	}
	alias_dump(fp);	

	/* #####################################################################
	| GENERATED CODE
	######################################################################*/

	fprintf(fp, "}

/* check() */

void
check(u8 *dg, size_t len)
{
	iphdr *ip = (iphdr *) &dg[14];
	tcphdr *h = (tcphdr *) &dg[(ip->ihl << 2) + 14];
	u8 flags=*(((char *) h) + 13);
	icmphdr *ih = (icmphdr *) &dg[(ip->ihl <<2) + 14];

	/* flags */
	int done_log = 0;
	int done_dump = 0;

");

	/* #####################################################################
	| END GENERATED CODE
	######################################################################*/

	/*======================================================================
	| now we do all the specific protocol checking.
	|=====================================================================*/

	if(num_filters == 0) {	/* this is a pretty silly thing to do, but hey */
		fprintf(fp, "\tif(1) {\n");
	}

	for(i=0; i<num_filters; i++) {
		struct configlist *cf = filters[i].cf;
		for(n=0; n<cf->index; n++) {

			struct configitem *ci = &cf->c[n];

			/*======================================================
			| check for basic protocol types first
			======================================================*/

			if(prot != ci->protocol) {

				if(prot != -1) {
					fprintf(fp, "\t}\n");
				}
				prot = ci->protocol;

				if(prot == PROT_USER) {
					fprintf(fp, "\t/* PROT_USER */\n");
					fprintf(fp, "\tif(1) {\n");
					fprintf(fp, "\t\t%s_fm->check(dg, len);\n",
						ci->filtermod->name);
				} else if(prot == PROT_TCP  ||
					  prot == PROT_UDP  ||
					  prot == PROT_ICMP ||
					  prot == PROT_IGNP) {
					fprintf(fp, "\t/* PROT_%s */
	if(
		((machdr*)dg)->type == MACTYPE_IPDG	&&
		ip->version == IP_VERSION		&&
		ip->protocol == PROTOCOL_%s
	) {\n", int2prot[prot], int2prot[prot]);
				} else if(prot == PROT_RAW) {
					fprintf(fp, "\t/* PROT_RAW */
	if(
		1		/* raw packets go through no initial filter */
	) {\n");
				} else if(prot == PROT_IP) {
					fprintf(fp, "\t/* PROT_IP */
	if(
		((machdr*)dg)->type == MACTYPE_IPDG	&&
		ip->version == IP_VERSION		&&
	) {\n");
				}
			}


			/*======================================================
			| then print out the rule checking code
			======================================================*/

#define FULL 0xffffffff

			fprintf(fp, "\t\tif(\t1 &&\n");

			if(ci->check_src_ip && prot != PROT_RAW) {
				if(ci->src_ip_mask == FULL) 
					fprintf(fp, "\t\t\tntohl(ip->saddr) == 0x%08x\t&& /* \"%s\" */\n",
						ntohl(ci->src_ip), ip2string(ci->src_ip));
				else
					fprintf(fp, "\t\t\t(ntohl(ip->saddr) & 0x%08x) == 0x%08x\t&& /* \"%s/%s\" */\n",
						ntohl(ci->src_ip_mask), 
						ntohl(ci->src_ip),
						ip2string(ci->src_ip),
						ip2string(ci->src_ip_mask));
			}
			if(ci->check_dst_ip && prot != PROT_RAW) {
				if(ci->dst_ip_mask == FULL)
					fprintf(fp, "\t\t\tntohl(ip->daddr) == 0x%08x\t&& /* \"%s\" */\n",
						ntohl(ci->dst_ip), ip2string(ci->dst_ip));
				else
					fprintf(fp, "\t\t\t(ntohl(ip->daddr) & 0x%08x) == 0x%08x\t&& /* \"%s/%s\" */\n",
						ntohl(ci->dst_ip_mask), 
						ntohl(ci->dst_ip),
						ip2string(ci->dst_ip),
						ip2string(ci->dst_ip_mask));
			}
			if(ci->check_src_prt && (prot == PROT_TCP || 
						 prot == PROT_UDP)) {
				if(ci->src_prt1 == ci->src_prt2) {
					fprintf(fp, "\t\t\tntohs(h->source) == 0x%04x\t&&\n",
							ntohs(ci->src_prt1));
				} else {
					fprintf(fp, "\t\t\tntohs(h->source) > 0x%04x\t&&\n",
							ntohs(ci->src_prt2));
					fprintf(fp, "\t\t\tntohs(h->source) < 0x%04x\t&&\n",
							ntohs(ci->src_prt1));
				}
			}
			if(ci->check_dst_prt && (prot == PROT_TCP || 
						 prot == PROT_UDP)) {
				if(ci->dst_prt1 == ci->dst_prt2) {
					fprintf(fp, "\t\t\tntohs(h->dest) == 0x%04x\t&&\n",
							ntohs(ci->dst_prt1));
				} else {
					fprintf(fp, "\t\t\tntohs(h->dest) > 0x%04x\t&&\n",
							ntohs(ci->dst_prt2));
					fprintf(fp, "\t\t\tntohs(h->dest) < 0x%04x\t&&\n",
							ntohs(ci->dst_prt1));
				}
			}
			if(ci->check_src_ip_not && prot != PROT_RAW) {
				if(ci->src_ip_not_mask == FULL)
					fprintf(fp, "\t\t\tntohs(ip->saddr) != 0x%08x\t&& /* \"%s\" */\n",
						ntohl(ci->src_ip_not), ip2string(ci->src_ip_not));
				else
					fprintf(fp, "\t\t\t(ntohl(ip->saddr) & 0x%08x) == 0x%08x\t&& /* \"%s/%s\" */\n",
						ntohl(ci->src_ip_not_mask), 
						ntohl(ci->src_ip_not),
						ip2string(ci->src_ip_not),
						ip2string(ci->src_ip_not_mask));
			}
			if(ci->check_dst_ip_not && prot != PROT_RAW) {
				if(ci->dst_ip_not_mask == FULL)
					fprintf(fp, "\t\t\tntohl(ip->daddr) != 0x%08x\t&& /* \"%s\" */\n",
						ntohl(ci->dst_ip_not), ip2string(ci->dst_ip_not));
				else
					fprintf(fp, "\t\t\t(ntohl(ip->daddr) & 0x%08x) == 0x%08x\t&& /* \"%s/%s\" */\n",
						ntohl(ci->dst_ip_not_mask), 
						ntohl(ci->dst_ip_not),
						ip2string(ci->dst_ip_not),
						ip2string(ci->dst_ip_not_mask));
			}
			if(ci->check_src_prt_not && (prot == PROT_TCP ||
						     prot == PROT_UDP)) {
				fprintf(fp, "\t\t\tntohs(h->source) != 0x%04x\t&&\n",
					ntohs(ci->src_prt_not));
			}
			if(ci->check_dst_prt_not && (prot == PROT_TCP ||
						     prot == PROT_UDP)) {
				fprintf(fp, "\t\t\tntohs(h->dest) != 0x%04x\t&&\n",
					ntohs(ci->dst_prt_not));
			}
			if(ci->check_icmp_type && prot == PROT_ICMP) {
				fprintf(fp, "\t\t\tih->type == 0x%02x\t&&\n",
					ci->icmp_type);
			}
			if(ci->check_icmp_code && prot == PROT_ICMP) {
				fprintf(fp, "\t\t\tih->code == 0x%02x\t&&\n",
					ci->icmp_code);
			}
			if(ci->check_tcp_flags_on && prot == PROT_TCP) {
				fprintf(fp, "\t\t\t(flags & 0x%02x) == 0x%02x\t&&\n",
					ci->tcp_flags_on, ci->tcp_flags_on);
			}
			if(ci->check_tcp_flags_off && prot == PROT_TCP) {
				fprintf(fp, "\t\t\t(flags & 0x%02x) != 0x%02x\t&&\n",
					ci->tcp_flags_off, ci->tcp_flags_off);
			}
			if(ci->check_src_hw) {
				int n;
				fprintf(fp, "\t\t\t");
				for(n=0; n<6; n++) {
					fprintf(fp, "dg[%d] == 0x%02x && ", 
						n+6,
						ci->src_hw[n]);
				}
				fprintf(fp, "\n");
			}
			if(ci->check_dst_hw) {
				int n;
				fprintf(fp, "\t\t\t");
				for(n=0; n<6; n++) {
					fprintf(fp, "dg[%d] == 0x%02x && ", 
						n,
						ci->dst_hw[n]);
				}
				fprintf(fp, "\n");
			}
			if(ci->check_src_hw_not) {
				int n;
				fprintf(fp, "\t\t\t");
				for(n=0; n<6; n++) {
					fprintf(fp, "dg[%d] != 0x%02x && ", 
						n+6,
						ci->src_hw_not[n]);
				}
				fprintf(fp, "\n");
			}
			if(ci->check_dst_hw_not) {
				int n;
				fprintf(fp, "\t\t\t");
				for(n=0; n<6; n++) {
					fprintf(fp, "dg[%d] != 0x%02x && ", 
						n,
						ci->dst_hw_not[n]);
				}
				fprintf(fp, "\n");
			}
			/* === GNR === */

			fprintf(fp, "\t\t\t1\n\t\t) {\n");

			/*======================================================
			| finally, print out the action code as necessary
			======================================================*/

			if(ci->action != ACTION_USER)
				fprintf(fp, "\t\t\t/* ACTION_%s */\n", int2act[ci->action]);
			else
				fprintf(fp, "\t\t\t/* ACTION_USER */\n");

			switch(ci->action) {

				case ACTION_USER :
					fprintf(fp, "
\t\t\t{
\t\t\t\tstruct configitem ci;
\t\t\t\tci.logname = \"%s\";
\t\t\t\t%s_am->action(dg, &ci, len);
\t\t\t}\n", ci->logname, ci->actionmod->name);
					break;

				case ACTION_LOG :
					if(prot == PROT_USER)
						fprintf(fp, 
"\t\t\tif(done_log == 0) {
\t\t\t\tlog_unknown(dg, len, \"\", \"%s\");
\t\t\t\tdone_log = 1;
\t\t\t}\n",		ci->logname);
					else
						fprintf(fp, 
"\t\t\tif(done_log == 0) {
\t\t\t\tlog_%s(dg, len, \"\", \"%s\");
\t\t\t\tdone_log = 1;
\t\t\t}\n",		prot2logtype[prot], ci->logname);
					break;

				case ACTION_DUMP :
					fprintf(fp, 
"\t\t\tif(done_dump == 0) {
\t\t\t\tdump(dg, \"%s\", len);
\t\t\t\tdone_dump = 1;
\t\t\t}\n", ci->logname);
					break;

				case ACTION_IGNORE :
					fprintf(fp, "\t\t\treturn;\n");
					break;

				case ACTION_DL :
					if(prot == PROT_USER)
						fprintf(fp, 
"\t\t\tif(done_log == 0) {
\t\t\t\tlog_unknown(dg, len, dump(dg, \"%s\", len), \"%s\");
\t\t\t\tdone_log = 1;
\t\t\t}\n", ci->logname, ci->logname);
					else
						fprintf(fp, 
"\t\t\tif(done_log == 0) {
\t\t\t\tlog_%s(dg, len, dump(dg, \"%s\", len), \"%s\");
\t\t\t\tdone_log = 1;
\t\t\t}\n",		prot2logtype[prot], ci->logname, ci->logname);
					break;
			}

			fprintf(fp, "\t\t}\n");

		}

	}


	/* #####################################################################
	| GENERATED CODE
	######################################################################*/

	fputs("\t}\n}


void
log_ip_with_port(u8 *dg, size_t len, char *extra_string, char* logname)
{
	iphdr *ip = (iphdr *) &dg[14];
	tcphdr *t = (tcphdr *) &dg[(ip->ihl << 2) + 14];
	log(\"%s %s:%d => %s:%d (%s)\", logname, 
		ip2string(ip->saddr), ntohs(t->source), 
		ip2string(ip->daddr), ntohs(t->dest), extra_string);
}

void
log_ip_no_port(u8 *dg, size_t len, char *extra_string, char* logname)
{
	iphdr *ip = (iphdr *) &dg[14];
	log(\"%s %s => %s (%s)\", logname, ip2string(ip->saddr), 
			ip2string(ip->daddr), extra_string);
}

void
log_non_ip(u8 *dg, size_t len, char *extra_string, char* logname)
{
	log(\"%s %02x:%02x:%02x:%02x:%02x:%02x => %02x:%02x:%02x:%02x:%02x:%02x (%s)\",
		logname, dg[6], dg[7], dg[8], dg[9], dg[10], dg[11], dg[0], dg[1], dg[2], dg[3], dg[4], dg[5],
                        extra_string);

}

void
log_ip_unknown(u8 *dg, size_t len, char *extra_string, char* logname)
{
	iphdr *ip = (iphdr *) &dg[14];
	if(ip->protocol == PROTOCOL_TCP || ip->protocol == PROTOCOL_UDP)
		log_ip_with_port(dg, len, extra_string, logname);
	else
		log_ip_no_port(dg, len, extra_string, logname);
}

void
log_unknown(u8 *dg, size_t len, char *extra_string, char* logname)
{
	if(((machdr*)dg)->type == MACTYPE_IPDG)
		log_ip_unknown(dg, len, extra_string, logname);
	else
		log_non_ip(dg, len, extra_string, logname);
}

char *
dump(u8 *dg, char *logname, size_t len)
{
	static char	fn[1024];
	static int	sequence=0;
	FILE		*fp;

	if(logname[0] != 0) {
		snprintf(fn, 1024, \"/tmp/netl/%s-%d-%d-%d.dg\", logname, getpid(), (unsigned) time(NULL), sequence++);
	} else {
		snprintf(fn, 1024, \"/tmp/netl/%d-%d-%d.dg\", getpid(), (unsigned) time(NULL), sequence++);
	}
	if((fp=fopen(fn, \"w\"))==NULL) {
		err(\"unable to open dump file %s\", fn);
		return NULL;
	}
	if(fwrite(dg, 1, len, fp) != len)
		err(\"error writing to dump file %s\", fn);
	fclose(fp);

	return fn;
}

", fp);

	/* #####################################################################
	| END GENERATED CODE
	######################################################################*/

}
