/**
 * $Id: kimon.cpp,v 1.2 1998/05/13 09:37:32 dahlweid Exp $
 *
 * Markus Dahlweid (dahlweid@informatik.uni-bremen.de)
 *
 */

#include <kapp.h>
#include <kwm.h>

#include <qwidget.h>
#include <qevent.h>
#include <qpainter.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <asm/param.h>
#include <sys/mman.h>
#include <linux/isdn.h>

#include "kimon.h"
#include "kconfig.h"

#include "kimon.moc"
#include <X11/Xlib.h>
#include <qtooltip.h>
#include <qframe.h>
#include <qmsgbox.h>
#include <qtabdlg.h>
#include <qlayout.h>
#include <qcombo.h>

#include <iostream.h>

KApplication *app;
KConfig *kimonConf;

KIsdnMon::KIsdnMon(QWidget *parent, const char *name, WFlags f)
    : QWidget (parent, name, f)
{
    IInfo = NULL;

    parseConfig();
    
    memset(idmap,0,sizeof(idmap));
    memset(phone,0,sizeof(phone));
    timer_interval = 1000;     /* this is default (1 sec) */
    tid = startTimer(timer_interval);

    readInfo();
    IInfo = new KIsdnInfo(num_isdn_lines());
    copyUsage();
    updateValues();

    _menu = new KPopupMenu("KDE-ISDN-Monitor");
    _menu->insertItem("Open", this, SLOT(open()));
    _menu->insertItem("Preferences", this, SLOT(prefs()));
    _menu->insertSeparator();
    _menu->insertItem("Quit", this, SLOT(quit()));
}

KIsdnMon::~KIsdnMon()
{
    int i;
    for (i=0; i<(int)(sizeof(idmap)/sizeof(char*)); i++)
	if (idmap[i]) free(idmap[i]);
    for (i=0; i<(int)(sizeof(phone)/sizeof(char*)); i++)
	if(phone[i])free(phone[i]);
    killTimer(tid);
}

void KIsdnMon::parseConfig() {
    QColor *temp1 = new QColor("black");
    QColor *temp2 = new QColor(0,225,0);
    QColor *temp3 = new QColor(220,0,0);
    QColor *temp4 = new QColor(0,0,200);
    
    black = kimonConf->readColorEntry("BackgroundColor", temp1);
    green = kimonConf->readColorEntry("InactiveColor", temp2);
    red   = kimonConf->readColorEntry("OutgoingColor", temp3);
    blue  = kimonConf->readColorEntry("IncomingColor", temp4);

    brush[0] = QBrush(black, SolidPattern);
    brush[1] = QBrush(green, SolidPattern);
    brush[2] = QBrush(red,   SolidPattern);
    brush[3] = QBrush(blue,  SolidPattern);

    ISDNinfo = kimonConf->readEntry("IsdninfoPath", "/dev/isdninfo");
    MiddleClick = kimonConf->readEntry("MiddleClickPath", "xisdnload");

    openOnChange = kimonConf->readNumEntry("OpenOnChange",2);

    delete temp1;
    delete temp2;
    delete temp3;
    delete temp4;
}

void scan_str(char *buffer, char *field[], int max) {
    char *buf = buffer;
    int i;
    char *token = strtok(buf, "\t");

    for(i=0; i < max; i++) {
        token = strtok(NULL, " ");
	if (field[i])
	    free(field[i]);
	field[i]=strdup(token);
    }
}
 
void scan_int(char *buffer, int (*field)[ISDN_MAX_CHANNELS], int max) {
    char * buf = buffer;
    int i;
    char * token = strtok(buf, "\t");

    for(i=0; i < max; i++) {
        token = strtok(NULL, " ");
        (*field)[i] = atoi(token);
    }
}

void scan_int(char *buffer, int (*field)[ISDN_MAX_DRIVERS], int max) {
    char * buf = buffer;
    int i;
    char * token = strtok(buf, "\t");

    for(i=0; i < max; i++) {
        token = strtok(NULL, " ");
        (*field)[i] = atoi(token);
    }
}

void KIsdnMon::paintEvent(QPaintEvent *)
{
    QPainter p;

    p.begin(this);
    
    p.setWindow(0, 0, 19, 19);
    p.setPen(black);
    // p.fillRect(10, 0, 80, 100, brush[0]); // black

    for(i=0; i < 2; i++)
        if (chmap[i] >= 0 && drmap[i] >= 0) {
	    int brushi=1;		// green
	    if ((flags[0] >> i)&1){
		brushi= (usage[i] & ISDN_USAGE_OUTGOING) ? 2 // red
		    : 3;		// blue
	    }
	    
	    p.fillRect(3, i*10+1, 14, 8, brush[brushi]);
	    p.drawRect(3, i*10+1, 14, 8);
	}
    
    p.end();
}

int KIsdnMon::num_isdn_lines() {
    int lines = 0;
    for(i=0; i < ISDN_MAX_CHANNELS; i++)
        if (chmap[i] >= 0 && drmap[i] >= 0) {
	    lines++;
	}
    return lines;
}

/**
 * this function reads the necessary information from ISDNINFO
 */

void KIsdnMon::readInfo() {
    if (!(isdninfo = fopen((const char*)ISDNinfo, "r"))) {
        fprintf(stderr, "ERROR: Can't open '%s'\n", (const char*)ISDNinfo);
        exit(1);
    }
    
    fgets(buf, IM_BUFSIZE, isdninfo);
    scan_str(buf, idmap, ISDN_MAX_CHANNELS);
    
    fgets(buf, IM_BUFSIZE, isdninfo);
    scan_int(buf, &chmap, ISDN_MAX_CHANNELS);
    
    fgets(buf, IM_BUFSIZE, isdninfo);
    scan_int(buf, &drmap, ISDN_MAX_CHANNELS);
    
    fgets(buf, IM_BUFSIZE, isdninfo);
    scan_int(buf, &usage, ISDN_MAX_CHANNELS);

    fgets(buf, IM_BUFSIZE, isdninfo);
    scan_int(buf, &flags, ISDN_MAX_DRIVERS);

    fgets(buf, IM_BUFSIZE, isdninfo);
    scan_str(buf, phone, ISDN_MAX_CHANNELS);

    fclose(isdninfo);

    if (IInfo!=NULL) {
	updateValues();
	int erg = compareUsage();
	if (erg && openOnChange) {
	    open();
	}
    }
    
    repaint();
}
    
void KIsdnMon::mouseReleaseEvent (QMouseEvent *e) {
    if (e->button()==RightButton) {
	_menu->popup(QPoint(this->x(), this->y()),6);
	_menu->exec();
    }
    if (e->button()==MidButton) {
	QString loadcmd = MiddleClick+"&";
	system((char const *)loadcmd);
    }
}

void KIsdnMon::mouseDoubleClickEvent (QMouseEvent *) {
    IInfo->show();
    IInfo->raise();
}

void KIsdnMon::timerEvent(QTimerEvent *) {
    readInfo();
}

void KIsdnMon::copyUsage() {
    for(int i=0; i < ISDN_MAX_CHANNELS; i++)
	compare[i]=usage[i];
}

int KIsdnMon::compareUsage() {
    /* results:
       0 - no changes
       1 - outgoing call
       2 - incoming call
    */
    for(int i=0; i < ISDN_MAX_CHANNELS; i++)
        if (chmap[i] >= 0 && drmap[i] >= 0) 
	    if ((flags[0] >> i) & 1) {
		int retval = (usage[i] & ISDN_USAGE_OUTGOING) ? 1  // outgoing
		    : 2; // incoming
		if (usage[i] != compare[i]) {
		    IInfo->reset_time(i);
		    copyUsage();
		    return retval;
		}
	    } else {
		copyUsage();
	    }
        
    return 0;
}

void KIsdnMon::updateValues() {
    int num=0;
    
    for(int i=0; i < ISDN_MAX_CHANNELS; i++)
        if (chmap[i] >= 0 && drmap[i] >= 0) {
	    IInfo->set_id(num, num);
	    IInfo->set_channel(num, idmap[i]);
	    
            if (usage[i] & ISDN_USAGE_OUTGOING) 
		IInfo->set_status(num, "Out");
	    else if (usage[i] == ISDN_USAGE_EXCLUSIVE) 
		IInfo->set_status(num, "Excl.");
	    else if (usage[i]) 
		IInfo->set_status(num, "In");
	    else 
		IInfo->set_status(num, "Off");
	    
            switch(usage[i] & ISDN_USAGE_MASK) {
		case ISDN_USAGE_RAW:
		    IInfo->set_usage(num, "Raw");
		    break;
		case ISDN_USAGE_MODEM:
		    IInfo->set_usage(num, "Modem");
		    break;
		case ISDN_USAGE_NET:
		    IInfo->set_usage(num, "Net");
		    break;
		case ISDN_USAGE_VOICE:
		    IInfo->set_usage(num, "Voice");
		    break;
		case ISDN_USAGE_FAX:
		    IInfo->set_usage(num, "Fax");
		    break;
		default:
		    IInfo->set_usage(num, "None");
		    break;
	    }

	    if(usage[i] & ISDN_USAGE_MASK) {
		IInfo->set_phone(num, phone[i]);
	    }
	    else
		IInfo->set_phone(num, "");
	    
	    IInfo->set_online(num, ((flags[0] >> i) & 1)+
			      (usage[i] & ISDN_USAGE_OUTGOING)/
			      ISDN_USAGE_OUTGOING*2);
	    
	    num++;
	}
}

void KIsdnMon::open() {
    IInfo->show();
    IInfo->raise();
}

void KIsdnMon::quit() {
    kimonConf->sync();
    app->quit();
}

void KIsdnMon::prefs() {
    QTabDialog *tab = new QTabDialog();
    
    QWidget *inf = new QWidget(tab);
    QLabel *l = new QLabel(inf);
    l->setMinimumSize(100,25);
    l->setText("Kimon");
    l->move(25,20);
    l->setFont(QFont("Helvetica", 20, QFont::Bold));
    l = new QLabel(inf);
    l->setAutoResize(true);
    l->setText(KIMON_VERSION);
    l->move(50,70);
    l = new QLabel(inf);
    l->setAutoResize(true);
    l->setText("a ISDN-monitor for the K Desktop Environment");
    l->move(50,110);
    l = new QLabel(inf);
    l->setAutoResize(true);
    l->setText("by Markus Dahlweid (dahlweid@tzi.org)");
    l->move(50,130);
    tab->addTab(inf, "Info");
    
    KIsdnPathPrefs *paths = new KIsdnPathPrefs(tab);
    tab->addTab(paths, "Behaviour");
    KIsdnColorPrefs *col = new KIsdnColorPrefs(tab);
    tab->addTab(col, "Colors");

    tab->setOkButton();
    tab->setCancelButton();

    connect(tab, SIGNAL(applyButtonPressed()), paths, SLOT(saveValues()));
    connect(tab, SIGNAL(defaultButtonPressed()), paths, SLOT(readValues()));
    connect(tab, SIGNAL(applyButtonPressed()), col, SLOT(saveValues()));
    connect(tab, SIGNAL(defaultButtonPressed()), col, SLOT(readValues()));
    connect(tab, SIGNAL(applyButtonPressed()), SLOT(parseConfig()));
    connect(tab, SIGNAL(defaultButtonPressed()), SLOT(parseConfig()));
    
    tab->setMaximumSize(400,300);
    tab->show();
}

/**
 * A small light in each status-window line.
 */

KIsdnInfoLineStatus::KIsdnInfoLineStatus(QWidget *parent = 0) :QFrame(parent){
    number=0;
}

KIsdnInfoLineStatus::~KIsdnInfoLineStatus(){
    if (number) free(number);
}

void KIsdnInfoLineStatus::set_phone(const char *phone){
    if (number) free(number);
    number=strdup(phone);
};

void KIsdnInfoLineStatus::mouseReleaseEvent (QMouseEvent *) {
    QString command = kimonConf->readEntry("IsdnctrlPath", "/sbin/isdnctrl");
    char cmd[500];
    if (number&&number[0]>='0'&&number[0]<='9'){
	sprintf(cmd,
		"%s hangup `%s list all|"
		"awk '/interface/{cif=$5}"
		"/ %s/{ifc=cif}END{print substr(ifc,2,length(ifc)-3)}'`",
		(const char*)command, (const char*)command, number);
	int erg = system(cmd);
	if (erg!=0) {
	    QString msg = "Error while executing the following command:\n";
	    msg += command;
	    
	    QMessageBox::warning(this, "kimon error",
				      (const char*)msg);
	}
	
    }
}


/**
 * A line in the status-window.
 */

KIsdnInfoLine::KIsdnInfoLine(QWidget *parent = 0, bool panel=true, const char *name = 0) 
    : QWidget(parent, name) 
{
    _upsince=QDateTime::currentDateTime();
    
    green = new QColor(0,225,0);
    red   = new QColor(220,0,0);
    blue  = new QColor(0,0,200);

   _panel = panel;
   _onl = false;
   
    _id = new QLabel(this);
    _id->setFixedSize(20, 20);
    _id->setAlignment(AlignHCenter|AlignVCenter);
    _id->setText("");
    _id->move(5,3);
    _id->show();
    
    _channel = new QLabel(this);
    _channel->setFixedSize(95,20);
    if (_panel) _channel->setFrameStyle(QFrame::Panel|QFrame::Sunken);
    _channel->setText("");
    _channel->move(30,3);
    _channel->show();
    
    _status = new QLabel(this);
    _status->setFixedSize(40,20);
    if (_panel) _status->setFrameStyle(QFrame::Panel|QFrame::Sunken);
    _status->setAlignment(AlignHCenter|AlignVCenter);
    _status->setText("Off");
    _status->move(130,3);
    _status->show();
    
    _time = new QLabel(this);
    _time->setFixedSize(60,20);
    if (_panel) _time->setFrameStyle(QFrame::Panel|QFrame::Sunken);
    _time->setAlignment(AlignHCenter|AlignVCenter);
    _time->setText("0:00:00");
    _time->move(175,3);
    _time->show();
    
    _phone = new QLabel(this);
    _phone->setFixedSize(200,20);
    if (_panel) _phone->setFrameStyle(QFrame::Panel|QFrame::Sunken);
    _phone->setText("");
    _phone->move(240,3);
    _phone->show();
    
    _usage = new QLabel(this);
    _usage->setFixedSize(40,20);
    if (_panel) _usage->setFrameStyle(QFrame::Panel|QFrame::Sunken);
    _usage->setAlignment(AlignHCenter|AlignVCenter);
    _usage->setText("None");
    _usage->move(435,3);
    _usage->show();
    
    if (_panel) {
	_online = new KIsdnInfoLineStatus(this);
	_online->setFixedSize(18,18);
	_online->setLineWidth(2);
	_online->setMidLineWidth(1);
	_online->setBackgroundColor(kimonConf->readColorEntry("InactiveColor",green));
	_online->setFrameStyle(QFrame::Box | QFrame::Raised);
	_online->move(480, 4);
	_online->show();
    } else {
	_online = NULL;
    }
    
    setFixedSize(505,23);
}

KIsdnInfoLine::~KIsdnInfoLine() {
}

void KIsdnInfoLine::set_id(int id) {
    _id->setNum(id);
}

void KIsdnInfoLine::set_phone(char* phone) {
  if(_online)
    _online->set_phone(phone);
    _phone->setText(phone);
}

void KIsdnInfoLine::set_usage(char* usage) {
    _usage->setText(usage);
}

void KIsdnInfoLine::set_status(char* status) {
    _status->setText(status);
}

void KIsdnInfoLine::set_channel(char* channel) {
    _channel->setText(channel);
}

void KIsdnInfoLine::set_online(int dir) {
    if (dir != _onl) {
	switch(dir){
	    case 3:
		_online->setBackgroundColor(kimonConf->readColorEntry("OutgoingColor", 
								      red));
		break;
	    case 1:
		_online->setBackgroundColor(kimonConf->readColorEntry("IncomingColor",
								      blue));
		break;
	    default:
		_online->setBackgroundColor(kimonConf->readColorEntry("InactiveColor",
								      green));
		break;
	}
	_onl = dir;
    }
    recalc_time();
}

void KIsdnInfoLine::set_time(char *time) {
    _time->setText(time);
}

void KIsdnInfoLine::reset_time() {
    _upsince=QDateTime::currentDateTime();
    sprintf(timestr, "0:00:00");
    set_time(timestr);
}

void KIsdnInfoLine::recalc_time() {
    if ((_onl==1) || (_onl==3)) {
	dur = _upsince.secsTo (QDateTime::currentDateTime());
	sec = dur % 60;
	min = dur / 60;
	hou = min / 60;
	min = min % 60;
	sprintf(timestr, "%d:%02d:%02d", hou, min, sec);
    } else {
	sprintf(timestr, "0:00:00");
    }
    
    set_time(timestr);
}

/**
 * The status-window
 */

KIsdnInfo::KIsdnInfo(int num, QWidget *parent = 0, const char *name = 0) 
    : QWidget(parent, name) 
{
    KIsdnInfoLine *iil = new KIsdnInfoLine(this, false);
    iil->set_channel("Channel");
    iil->set_phone("Remote-number");
    iil->set_usage("Usage");
    iil->set_status("Status");
    iil->set_time("Time");
    iil->show();
    
    lines = num;
    for(int i=0; i < lines; i++) {
	info[i] = new KIsdnInfoLine(this);
	(info[i])->set_id(i);
	(info[i])->move(0,(i+1)*(info[i])->height());
	(info[i])->show();
    }
    if (lines==0) {
	setFixedSize(0,0);
    } else {
	setFixedSize((info[0])->width(), (lines+1)*(info[0])->height());
    }
}

KIsdnInfo::~KIsdnInfo() {
}

void KIsdnInfo::set_id(int chan, int id) {
    (info[chan])->set_id(id);
}

void KIsdnInfo::set_phone(int chan, char* phone) {
    (info[chan])->set_phone(phone);
}

void KIsdnInfo::set_usage(int chan, char* usage) {
    (info[chan])->set_usage(usage);
}

void KIsdnInfo::set_status(int chan, char* status) {
    (info[chan])->set_status(status);
}

void KIsdnInfo::set_channel(int chan, char* channel) {
    (info[chan])->set_channel(channel);
}

void KIsdnInfo::set_online(int chan, int online) {
    (info[chan])->set_online(online);
}

void KIsdnInfo::reset_time(int chan) {
    (info[chan])->reset_time();
}

void KIsdnInfo::recalc_time(int chan) {
    (info[chan])->recalc_time();
}


/**
 * The preferences-dialog
 */

KIsdnPathPrefs::KIsdnPathPrefs(QWidget *parent = 0, const char *name = 0)
    : QWidget(parent, name)
{
    QGridLayout * g = new QGridLayout( this, 6, 2, 10);
    g->setColStretch( 1, 1 );
    g->setColStretch( 0, 0 );
    g->setRowStretch( 0, 0);
    g->setRowStretch( 5, 1);

    ISDNinfo = new QLineEdit(this);
    ISDNcontrol = new QLineEdit(this);
    MiddleClick = new QLineEdit(this);
    OpenOnCall = new QComboBox(this);
   
    ISDNinfo->setMinimumSize(100,30);
    ISDNcontrol->setMinimumSize(100,30);
    MiddleClick->setMinimumSize(100,30);
    OpenOnCall->setMinimumSize(100,30);

    OpenOnCall->insertItem("never",0);
    OpenOnCall->insertItem("on outgoing calls",1);
    OpenOnCall->insertItem("on incoming calls",2);
    OpenOnCall->insertItem("on incoming and outgoing calls",3);
    
    g->addWidget(ISDNinfo, 1,1);
    g->addWidget(ISDNcontrol, 2,1);
    g->addWidget(MiddleClick, 3,1);
    g->addWidget(OpenOnCall, 4,1);
    
    QLabel *l = new QLabel(this);
    g->addWidget(l,1,0);
    l->setText("Path to isdn&info:");
    l->setBuddy(ISDNinfo);
    l->setMinimumSize( l->sizeHint() );
    
    l = new QLabel(this);
    g->addWidget(l,2,0);
    l->setText("Path to isdn&ctrl:");
    l->setBuddy(ISDNcontrol);
    l->setMinimumSize( l->sizeHint() );
    
    l = new QLabel(this);
    g->addWidget(l,3,0);
    l->setText("&Execute on middle-click:");
    l->setBuddy(MiddleClick);
    l->setMinimumSize( l->sizeHint() );
    
    l = new QLabel(this);
    g->addWidget(l,4,0);
    l->setText("Open &monitor-window:");
    l->setBuddy(OpenOnCall);
    l->setMinimumSize( l->sizeHint() );
    
    g->activate();
    
    readValues();
    
    //setFixedSize(400, 250);
}

KIsdnPathPrefs::~KIsdnPathPrefs() {}

void KIsdnPathPrefs::readValues() {
    ISDNinfo->setText((const char*)kimonConf->readEntry("IsdninfoPath", 
							"/dev/isdninfo"));
    ISDNcontrol->setText((const char*)kimonConf->readEntry("IsdnctrlPath", 
							   "/sbin/isdnctrl"));
    MiddleClick->setText((const char*)kimonConf->readEntry("MiddleClickPath", 
							   "xisdnload"));
    OpenOnCall->setCurrentItem(kimonConf->readNumEntry("OpenOnCall", 2));

    repaint();
}

void KIsdnPathPrefs::saveValues() {
    kimonConf->writeEntry("IsdninfoPath", ISDNinfo->text());
    kimonConf->writeEntry("IsdnctrlPath", ISDNcontrol->text());
    kimonConf->writeEntry("MiddleClickPath", MiddleClick->text());
    kimonConf->writeEntry("OpenOnCall", OpenOnCall->currentItem());
}

/**
 * The color-preferences-dialog
 */

KIsdnColorPrefs::KIsdnColorPrefs(QWidget *parent = 0, const char *name = 0)
    : QWidget(parent, name)
{
    QGridLayout * g = new QGridLayout( this, 6, 3, 10);
    g->setColStretch( 1, 0 );
    g->setColStretch( 0, 0 );
    g->setColStretch( 2, 1 );
    g->setRowStretch( 0, 0);
    g->setRowStretch( 5, 1);

    black = new KColorButton(this);
    red = new KColorButton(this);
    blue = new KColorButton(this);
    green = new KColorButton(this);
       
    black->setMinimumSize(100,30);
    red->setMinimumSize(100,30);
    blue->setMinimumSize(100,30);
    green->setMinimumSize(100,30);

    g->addWidget(black, 1,1);
    g->addWidget(green, 2,1);
    g->addWidget(red, 3,1);
    g->addWidget(blue, 4,1);
    
    QLabel *l = new QLabel(this);
    g->addWidget(l,1,0);
    l->setText("&Backgroundcolor:");
    l->setBuddy(black);
    l->setMinimumSize( l->sizeHint() );
    
    l = new QLabel(this);
    g->addWidget(l,2,0);
    l->setText("&Inactivity");
    l->setBuddy(green);
    l->setMinimumSize( l->sizeHint() );
    
    l = new QLabel(this);
    g->addWidget(l,3,0);
    l->setText("&Outgoing call:");
    l->setBuddy(red);
    l->setMinimumSize( l->sizeHint() );
    
    l = new QLabel(this);
    g->addWidget(l,4,0);
    l->setText("&Incoming call:");
    l->setBuddy(blue);
    l->setMinimumSize( l->sizeHint() );
    
    g->activate();
    
    readValues();
}

KIsdnColorPrefs::~KIsdnColorPrefs() {}

void KIsdnColorPrefs::readValues() {
    QColor *temp1 = new QColor("black");
    QColor *temp2 = new QColor(0,225,0);
    QColor *temp3 = new QColor(220,0,0);
    QColor *temp4 = new QColor(0,0,200);
    
    black->setColor(kimonConf->readColorEntry("BackgroundColor",temp1));
    green->setColor(kimonConf->readColorEntry("InactiveColor",temp2));
    red->setColor(kimonConf->readColorEntry("OutgoingColor",temp3));
    blue->setColor(kimonConf->readColorEntry("IncomingColor",temp4));

    delete temp1;
    delete temp2;
    delete temp3;
    delete temp4;
}

void KIsdnColorPrefs::saveValues() {
    kimonConf->writeEntry("BackgroundColor",black->color());
    kimonConf->writeEntry("InactiveColor",green->color());
    kimonConf->writeEntry("OutgoingColor",red->color());
    kimonConf->writeEntry("IncomingColor",blue->color());
}

/**
 * Create and display the IsdnMonitor.
 */

int main( int argc, char **argv )
{
    app = new KApplication( argc, argv, "kimon" );
    kimonConf = app->getConfig();
    kimonConf->setGroup("KimonISDNMonitor");
    
    KIsdnMon* IMon = new KIsdnMon(0, 0, WStyle_NormalBorder);
    app->setMainWidget(IMon);
    KWM::setDockWindow(IMon->winId());
    IMon->show();
    
    return app->exec();
}

