/***

itrafmon.c - the IP traffic monitor module
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <sys/time.h>
#include <sys/wait.h>
#include <errno.h>
#include <signal.h>
#include <ctype.h>
#include "tcptable.h"
#include "othptab.h"
#include "isdntab.h"
#include "packet.h"
#include "ifaces.h"
#include "ipcsum.h"
#include "promisc.h"
#include "utfdefs.h"
#include "utfilter.h"
#include "othfilter.h"
#include "deskman.h"
#include "error.h"
#include "attrs.h"
#include "log.h"
#include "revname.h"
#include "rvnamed.h"
#include "dirs.h"
#include "timer.h"
#include "ipfrag.h"
#include "options.h"
#include "instances.h"
#include "logvars.h"

#define SCROLLUP 0
#define SCROLLDOWN 1

extern int exitloop;
extern int daemonized;

void writetcplog(int logging, FILE * fd, struct tcptableent *entry,
		 unsigned int pktlen, int mac, char *message);
void write_tcp_unclosed(int logging, FILE * fd, struct tcptable *table);
void write_timeout_log(int logging, FILE *logfile, struct tcptableent *tcpnode);
void show_sort_statwin(WINDOW **statwin, PANEL **statpanel);

void rotate_ipmon_log()
{
    rotate_flag = 1;
    strcpy(target_logname, current_logfile);
    signal(SIGUSR1, rotate_ipmon_log);
}

/* Hot key indicators for the bottom line */

void ipmonhelp()
{
    move(LINES - 1, 1);
    printkeyhelp("Up/Dn/PgUp/PgDn", "-scroll  ", stdscr, HIGHATTR, STATUSBARATTR);
    move(LINES - 1, 43);
    printkeyhelp("W", "-chg actv win  ", stdscr, HIGHATTR, STATUSBARATTR);
    printkeyhelp("S", "-sort TCP  ", stdscr, HIGHATTR, STATUSBARATTR);
    stdexitkeyhelp();
};

void uniq_help(int what)
{
    move(LINES - 1, 25);
    if (!what)
	printkeyhelp("M", "-more TCP info   ", stdscr, HIGHATTR, STATUSBARATTR);
    else
	printkeyhelp("Lft/Rt", "-vtcl scrl  ", stdscr, HIGHATTR, STATUSBARATTR);
}

/* Mark general packet count indicators */

void prepare_ip_statwin(WINDOW * win)
{
    wattrset(win, IPSTATLABELATTR);
    wmove(win, 0, 1);
    wprintw(win, "IP:");
    wmove(win, 0, 16 * COLS / 80);
    wprintw(win, "TCP:");
    wmove(win, 0, 31 * COLS / 80);
    wprintw(win, "UDP:");
    wmove(win, 0, 46 * COLS / 80);
    wprintw(win, "ICMP:");
    wmove(win, 0, 62 * COLS / 80);
    wprintw(win, "Non-IP:");
}

void markactive(int curwin, WINDOW * tw, WINDOW * ow)
{
    WINDOW *win1;
    WINDOW *win2;
    int x1, y1, x2, y2;

    if (!curwin) {
	win1 = tw;
	win2 = ow;
    } else {
	win1 = ow;
	win2 = tw;
    }

    getmaxyx(win1, y1, x1);
    getmaxyx(win2, y2, x2);

    wmove(win1, --y1, COLS - 10);
    wattrset(win1, ACTIVEATTR);
    wprintw(win1, " Active ");
    wattrset(win1, BOXATTR);
    wmove(win2, --y2, COLS - 10);
    whline(win2, ACS_HLINE, 8);
}

void show_ip_stats(WINDOW * win, unsigned long long iptotal,
		   unsigned long long tcptotal,
		   unsigned long long udptotal,
		   unsigned long long icmptotal,
		   unsigned long long noniptotal)
{
    wattrset(win, IPSTATATTR);
    wmove(win, 0, 6 * COLS / 80);
    printlargenum(iptotal, win);
    wmove(win, 0, 21 * COLS / 80);
    printlargenum(tcptotal, win);
    wmove(win, 0, 36 * COLS / 80);
    printlargenum(udptotal, win);
    wmove(win, 0, 52 * COLS / 80);
    printlargenum(icmptotal, win);
    wmove(win, 0, 70 * COLS / 80);
    printlargenum(noniptotal, win);
}


/* 
 * Scrolling and paging routines for the upper (TCP) window
 */

void scrollupperwin(struct tcptable *table, int direction,
		    unsigned long *idx, int mode)
{
    char sp_buf[10];
    sprintf(sp_buf, "%%%dc", COLS - 2);
    wattrset(table->tcpscreen, STDATTR);
    if (direction == SCROLLUP) {
	if (table->lastvisible != table->tail) {
	    wscrl(table->tcpscreen, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;
	    (*idx)++;
	    wmove(table->tcpscreen, table->imaxy - 1, 0);
	    scrollok(table->tcpscreen, 0);
	    wprintw(table->tcpscreen, sp_buf, ' ');
	    scrollok(table->tcpscreen, 1);
	    printentry(table, table->lastvisible, *idx, mode);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->tcpscreen, -1);
	    table->firstvisible = table->firstvisible->prev_entry;
	    table->lastvisible = table->lastvisible->prev_entry;
	    (*idx)--;
	    wmove(table->tcpscreen, 0, 0);
	    wprintw(table->tcpscreen, sp_buf, ' ');
	    printentry(table, table->firstvisible, *idx, mode);
	}
    }
}

void pageupperwin(struct tcptable *table, int direction,
		  unsigned long *idx, int mode)
{
    int i = 1;

    wattrset(table->tcpscreen, STDATTR);
    if (direction == SCROLLUP) {
	while ((i <= table->imaxy - 3)
	       && (table->lastvisible != table->tail)) {
	    i++;
	    scrollupperwin(table, direction, idx, mode);
	}
    } else {
	while ((i <= table->imaxy - 3)
	       && (table->firstvisible != table->head)) {
	    i++;
	    scrollupperwin(table, direction, idx, mode);
	}
    }
}

/*
 * Scrolling and paging routines for the lower (non-TCP) window.
 */

void scrolllowerwin(struct othptable *table, int direction)
{
    if (direction == SCROLLUP) {
	if (table->lastvisible != table->tail) {
	    wscrl(table->othpwin, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;

	    if (table->htstat == HIND) {	/* Head indicator on? */
		wmove(table->borderwin, table->obmaxy - 1, 1);
		whline(table->borderwin, ACS_HLINE, 8);
		table->htstat = NOHTIND;
	    }
	    printothpentry(table, table->lastvisible, table->oimaxy - 1, 
	    		   0, (FILE *) NULL);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->othpwin, -1);
	    table->firstvisible = table->firstvisible->prev_entry;
	    table->lastvisible = table->lastvisible->prev_entry;

	    if (table->htstat == TIND) {	/* Tail indicator on? */
		wmove(table->borderwin, table->obmaxy - 1, 1);
		whline(table->borderwin, ACS_HLINE, 8);
		table->htstat = NOHTIND;
	    }
	    printothpentry(table, table->firstvisible, 0,
			   0, (FILE *) NULL);
	}
    }
}

void pagelowerwin(struct othptable *table, int direction)
{
    int i = 1;

    if (direction == SCROLLUP) {
	while ((i <= table->oimaxy - 2)
	       && (table->lastvisible != table->tail)) {
	    i++;
	    scrolllowerwin(table, direction);
	}
    } else {
	while ((i <= table->oimaxy - 2)
	       && (table->firstvisible != table->head)) {
	    i++;
	    scrolllowerwin(table, direction);
	}
    }
}

/*
 * Pop up sorting key window
 */

void show_tcpsort_win(WINDOW ** win, PANEL ** panel)
{
    *win = newwin(9, 35, (LINES - 8) / 2, COLS - 40);
    *panel = new_panel(*win);

    wattrset(*win, DLGBOXATTR);
    colorwin(*win);
    box(*win, ACS_VLINE, ACS_HLINE);
    wattrset(*win, DLGTEXTATTR);
    mvwprintw(*win, 2, 2, "Select sort criterion");
    wmove(*win, 4, 2);
    printkeyhelp("P", " - sort by packet count", *win, DLGHIGHATTR, DLGTEXTATTR);
    wmove(*win, 5, 2);
    printkeyhelp("B", " - sort by byte count", *win, DLGHIGHATTR, DLGTEXTATTR);
    wmove(*win, 6, 2);
    printkeyhelp("Any other key", " - cancel sort", *win, DLGHIGHATTR, DLGTEXTATTR);
    update_panels();
    doupdate();
}

/*
 * Routine to swap two TCP entries.  p1 and p2 are pointers to TCP entries,
 * but p1 must be ahead than p2.  It's a linked list thing.
 */
void swap_tcp_entries(struct tcptable *table,
		      struct tcptableent *p1, struct tcptableent *p2)
{
    struct tcptableent *p2nextsaved;
    struct tcptableent *p1prevsaved;
    unsigned int tmp;

    if (p1 == p2)
	return;

    tmp = p1->index;
    p1->index = p2->index;
    p2->index = tmp;

    p1->next_entry->index = p1->index + 1;
    p2->next_entry->index = p2->index + 1;

    if (p1->prev_entry != NULL)
	p1->prev_entry->next_entry = p2;
    else
	table->head = p2;

    if (p2->next_entry->next_entry != NULL)
	p2->next_entry->next_entry->prev_entry = p1->next_entry;
    else
	table->tail = p1->next_entry;

    p2nextsaved = p2->next_entry->next_entry;
    p1prevsaved = p1->prev_entry;

    if (p1->next_entry->next_entry == p2) {	/* swapping adjacent entries */
	p2->next_entry->next_entry = p1;
	p1->prev_entry = p2->next_entry;
    } else {
	p2->next_entry->next_entry = p1->next_entry->next_entry;
	p1->prev_entry = p2->prev_entry;
	p2->prev_entry->next_entry = p1;
	p1->next_entry->next_entry->prev_entry = p2->next_entry;
    }

    p2->prev_entry = p1prevsaved;
    p1->next_entry->next_entry = p2nextsaved;
}

/* 
 * Sort TCP window entries.  Contributed by Gal Laszlo.  Edited and
 * modified, GPJ <riker@seul.org>
 * Contributor's note: by LG - Gal Laszlo ( SlowTCP@hotmail.com ) 2000.07.19.
 * Author's note: Special (accented) characters removed from contributed
 *                source.
 */
void sortipents(struct tcptable *table, unsigned long *idx, int ch,
		int mode, int logging, FILE *logfile,
		time_t timeout, int *nomem)
{
    struct tcptableent *tcptmp1, *tcptmp2, *tcp_next2;
    unsigned long long val1 = 0, val2 = 0;
    unsigned long lastidx;
    unsigned int idxtmp;
    time_t now;
    WINDOW *statwin;
    PANEL *panel;
    
    if ((table->head == NULL)
	|| (table->head->next_entry->next_entry == NULL))
        return;

    ch = toupper(ch);
    
    if ((ch != 'P') && (ch != 'B'))
        return;
            
    show_sort_statwin(&statwin, &panel);
    
    lastidx = table->tail->index;
    
    tcptmp1 = table->head;

    now = time(NULL);
    
    while (tcptmp1 != NULL) {
	tcptmp2 = tcptmp1->next_entry->next_entry;
	/*
	 * While we're examining, might as well mark any timed
	 * out entries too.
	 */
	 
        if ((timeout > 0) && (tcptmp1->lastupdate / 60 > timeout) && !(tcptmp1->inclosed)) {
            tcptmp1->timedout = tcptmp1->oth_connection->timedout = 1;
            addtoclosedlist(table, tcptmp1, nomem);
            if (logging)
                write_timeout_log(logging, logfile, tcptmp1);
        }
    
        mvwprintw(statwin, 2, 13, "%u%%", tcptmp1->next_entry->index * 100 / lastidx);
        update_panels();
        doupdate();
             
        /*
         * Inner loop
         */
	while (tcptmp2 != NULL) {
	    tcp_next2 = tcptmp2->next_entry->next_entry;
	    switch (ch) {
	    case 'B':
		val1 =
		    max(tcptmp1->bcount, tcptmp1->oth_connection->bcount);
		val2 =
		    max(tcptmp2->bcount, tcptmp2->oth_connection->bcount);
		break;
	    case 'P':
		val1 =
		    max(tcptmp1->pcount, tcptmp1->oth_connection->pcount);
		val2 =
		    max(tcptmp2->pcount, tcptmp2->oth_connection->pcount);
		break;
	    }
	    if (val1 < val2) {	/* descending sort */
		swap_tcp_entries(table, tcptmp1, tcptmp2);
		tcptmp1 = tcptmp2;
	    }

	    tcptmp2 = tcp_next2;
	}

	tcptmp1 = tcptmp1->next_entry->next_entry;
    }
     
    del_panel(panel);
    delwin(statwin);
    update_panels();
    doupdate();
    colorwin(table->tcpscreen);
    tcptmp1 = table->firstvisible = table->head;
    *idx = 1;
    idxtmp = 0;
    while ((tcptmp1 != NULL) && (idxtmp <= table->imaxy - 1)) {
	printentry(table, tcptmp1, *idx, mode);
	if (idxtmp++ <= table->imaxy - 1)
	    table->lastvisible = tcptmp1;
	tcptmp1 = tcptmp1->next_entry;
    }
}

/*
 * Attempt to communicate with rvnamed, and if it doesn't respond, try
 * to start it.
 */

int checkrvnamed(void)
{
    int execstat = 0;
    pid_t cpid = 0;
    int cstat;
    extern int errno;

    indicate("Trying to communicate with reverse lookup server");
    if (!rvnamedactive()) {
	indicate("Starting reverse lookup server");

	if ((cpid = fork()) == 0) {
	    execstat = execl(RVNDFILE, NULL);

	    /* 
	     * execl() never returns, so if we reach this point, we have
	     * a problem.
	     */

	    _exit(1);
	} else if (cpid == -1) {
	    write_error("Can't spawn new process; lookups will block",
			daemonized);
	    return 0;
	} else {
	    while (waitpid(cpid, &cstat, 0) < 0)
		if (errno != EINTR)
		    break;

	    if (WEXITSTATUS(cstat) == 1) {
		write_error("Can't start rvnamed; lookups will block",
			    daemonized);
		return 0;
	    } else {
		sleep(1);
		return 1;
	    }
	}
    }
    return 1;
}

/* 
 * The IP Traffic Monitor
 */

void ipmon(const struct OPTIONS *options, int filtered,
	   struct filterlist *fl, struct othpoptions *ofilter,
	   unsigned int tcptm, int facilitytime, char *ifptr)
{
    int logging = options->logging;
    struct sockaddr_ll fromaddr;	/* iface info */
    unsigned short linktype;	/* data link type */
    int fd;			/* raw socket */
    int isdn_fd = -1;		/* file descriptor for isdnctrl if any */
    char tpacket[MAX_PACKET_SIZE];	/* raw packet data */
    char aligned_buf[ALIGNED_BUF_LEN];	/* aligned buffer */
    char *packet = NULL;	/* network packet ptr */
    struct iphdr *ippacket;
    struct tcphdr *transpacket;	/* IP-encapsulated packet */
    unsigned int sport = 0, dport = 0;	/* TCP/UDP port values */
    int firstin = 0;
    char sp_buf[10];

    unsigned long screen_idx = 1;

    struct timeval tv;
    unsigned long starttime = 0;
    unsigned long now = 0;
    unsigned long timeint = 0;
    unsigned long updtime = 0;
    unsigned long long updtime_usec = 0;
    unsigned long long unow = 0;
    unsigned long closedint = 0;

    WINDOW *statwin;
    PANEL *statpanel;

    WINDOW *sortwin;
    PANEL *sortpanel;

    FILE *logfile = NULL;

    int curwin = 0;

    int readlen;
    char ifname[8];

    unsigned long long iptotal = 0;
    unsigned long long noniptotal = 0;
    unsigned long long tcptotal = 0;
    unsigned long long udptotal = 0;
    unsigned long long icmptotal = 0;
    unsigned long long othptotal = 0;

    unsigned int nbplen;
    unsigned int br;

    int hdrchk;			/* IP checksum values */
    int hdrcsum;

    unsigned int iphlen;
    unsigned int totalhlen;

    struct tcptable table;
    struct tcptableent *tcpentry;
    int mode = 0;

    struct othptable othptbl;
    struct othptabent *othpent;

    struct isdntab isdntable;

    int p_sstat = 0, p_dstat = 0;	/* Reverse lookup statuses prior to */
    					/* reattempt in updateentry(). */
    int nifltok = 0;		/* Non-IP filter ok */

    int fragment = 0;		/* Set to 1 if not first fragment */

    int ch;
    int keymode = 0;
    char msgstring[80];
    int nomem = 0;

    struct promisc_states *promisc_list;

    int rvnfd = 0;

    int instance_id;
    
    /* 
     * Mark this instance of the traffic monitor
     */

    if (!facility_active(IPMONIDFILE, ifptr))
	mark_facility(IPMONIDFILE, "IP traffic monitor", ifptr);
    else {
        snprintf(msgstring, 80, "IP Traffic Monitor already listening on %s", gen_iface_msg(ifptr));
	write_error(msgstring, daemonized);
	return;
    }

    if (ifptr != NULL) {
	if (!iface_supported(ifptr)) {
	    err_iface_unsupported();
	    unmark_facility(IPMONIDFILE, ifptr);
	    return;
	}
	if (!iface_up(ifptr)) {
	    err_iface_down();
	    unmark_facility(IPMONIDFILE, ifptr);
	    return;
	}
    }

    open_socket(&fd);

    if (fd < 0) {
	unmark_facility(IPMONIDFILE, ifptr);
	return;
    }

    if (options->promisc) {
	if (first_active_facility()) {
	    init_promisc_list(&promisc_list);
	    save_promisc_list(promisc_list);
	    srpromisc(1, promisc_list);
	}
    }

    /*
     * Adjust instance counters
     */
     
    adjust_instance_count(PROCCOUNTFILE, 1);
    instance_id = adjust_instance_count(ITRAFMONCOUNTFILE, 1);
    strncpy(active_facility_countfile, ITRAFMONCOUNTFILE, 64);

    init_tcp_table(&table);
    init_othp_table(&othptbl, options->mac);

    if (options->revlook) {
	if (checkrvnamed())
	    open_rvn_socket(&rvnfd);
    } else
	rvnfd = 0;

    if (options->servnames)
	setservent(1);

    ipmonhelp();
    uniq_help(0);
    statwin = newwin(1, COLS, LINES - 2, 0);
    statpanel = new_panel(statwin);
    wattrset(statwin, IPSTATLABELATTR);
    wmove(statwin, 0, 0);
    sprintf(sp_buf, "%%%dc", COLS);
    wprintw(statwin, sp_buf, ' ');
    prepare_ip_statwin(statwin);
    show_ip_stats(statwin, 0, 0, 0, 0, 0);
    markactive(curwin, table.borderwin, othptbl.borderwin);
    update_panels();
    doupdate();

    /*
     * Try to open log file if logging activated.  Turn off logging
     * (for this session only) if an error was discovered in opening
     * the log file.  Configuration setting is kept.  Who knows, the
     * situation may be corrected later.
     */

    bzero(&isdntable, sizeof(struct isdntab));
    
    if (logging) {
        if (strcmp(current_logfile, "") == 0)
            strncpy(current_logfile, gen_instance_logname(IPMONLOG, instance_id), 80);
        
        if (!daemonized)
            input_logfile(current_logfile, &logging);
    }
    
    if (logging)
	opentlog(&logfile, current_logfile);

    if (logfile == NULL)
	logging = 0;

    if (logging)
	signal(SIGUSR1, rotate_ipmon_log);

    rotate_flag = 0;
    writelog(logging, logfile, "******** IP traffic monitor started ********");
   
    exitloop = 0;
    gettimeofday(&tv, NULL);
    starttime = timeint = closedint = tv.tv_sec;

    while (!exitloop) {
	getpacket(fd, tpacket, &fromaddr, &ch, &readlen, ifname,
		  table.tcpscreen);

	if (ch != ERR) {
	    if (keymode == 0) {
		switch (ch) {
		case KEY_UP:
		    if (!curwin)
			scrollupperwin(&table, SCROLLDOWN, &screen_idx,
				       mode);
		    else
			scrolllowerwin(&othptbl, SCROLLDOWN);
		    break;
		case KEY_DOWN:
		    if (!curwin)
			scrollupperwin(&table, SCROLLUP, &screen_idx,
				       mode);
		    else
			scrolllowerwin(&othptbl, SCROLLUP);
		    break;
		case KEY_RIGHT:
		    if (curwin) {
			if (othptbl.strindex != VSCRL_OFFSET)
			    othptbl.strindex = VSCRL_OFFSET;

			refresh_othwindow(&othptbl);
		    }
		    break;
		case KEY_LEFT:
		    if (curwin) {
			if (othptbl.strindex != 0)
			    othptbl.strindex = 0;

			refresh_othwindow(&othptbl);
		    }
		    break;
		case KEY_PPAGE:
		case '-':
		    if (!curwin)
			pageupperwin(&table, SCROLLDOWN, &screen_idx,
				     mode);
		    else
			pagelowerwin(&othptbl, SCROLLDOWN);
		    break;
		case KEY_NPAGE:
		case ' ':
		    if (!curwin)
			pageupperwin(&table, SCROLLUP, &screen_idx, mode);
		    else
			pagelowerwin(&othptbl, SCROLLUP);
		    break;
		case KEY_F(6):
		case 'w':
		case 'W':
		case 9:
		    curwin = ~curwin;
		    markactive(curwin, table.borderwin, othptbl.borderwin);
		    uniq_help(curwin);
		    break;
		case 'm':
		case 'M':
		    if (!curwin) {
			mode = (++mode) % 3;
			if ((mode == 1) && (!options->mac))
			    mode = 2;
			refreshtcpwin(&table, screen_idx, mode);
		    }
		    break;
		case 12:
		case 'l':
		case 'L':
		    refresh_screen();
		    break;

		case 'F':
		case 'f':
		case 'c':
		case 'C':
		    flushclosedentries(&table, tcptm, &screen_idx,
				       logging, logfile);
		    refreshtcpwin(&table, screen_idx, mode);
		    break;
		case 's':
		case 'S':
		    keymode = 1;
		    show_tcpsort_win(&sortwin, &sortpanel);
		    break;
		case 'Q':
		case 'q':
		case 'X':
		case 'x':
		case 24:
		case 27:
		    exitloop = 1;
		}
	    } else if (keymode == 1) {
		keymode = 0;
		del_panel(sortpanel);
		delwin(sortwin);
		update_panels();
		doupdate();
		sortipents(&table, &screen_idx, ch, mode, logging, logfile,
		           options->timeout, &nomem);
	    }
	}
	if (readlen > 0) {
	    if (!iface_supported(ifname))
		continue;

	    /* Is is for any particular interface only? */

	    if (ifptr != NULL) {
		if (strcmp(ifname, ifptr) != 0)
		    continue;
	    }
	    isdn_iface_check(&isdn_fd, ifname);
	    linktype = getlinktype(fromaddr.sll_hatype, ifname,
				   isdn_fd, &isdntable);

	    fromaddr.sll_protocol = htons(fromaddr.sll_protocol);
	    adjustpacket(tpacket, linktype, &packet, aligned_buf,
			 &readlen);

	    if (packet == NULL)
		continue;

	    if (fromaddr.sll_protocol != ETH_P_IP) {
		if ((fromaddr.sll_protocol == ETH_P_ARP) ||
		    (fromaddr.sll_protocol == ETH_P_RARP))
		    nifltok = othfilterok(ofilter, fromaddr.sll_protocol,
					  0, 0, 0, 0);
		else
		    nifltok = othfilterok(ofilter, 0, 0, 0, 0, 0);


		/*
		 * Display non-IP entry in lower window if filter ok,
		 * and nomem not set
		 */


		if ((nifltok) && (!nomem))
		    othpent = add_othp_entry(&othptbl, &table,
					     0, 0, NOT_IP,
					     fromaddr.sll_protocol,
					     linktype, (char *) tpacket,
					     (char *) packet, readlen, ifname, 0, 0,
					     0, logging, logfile,
					     options->servnames, 0,
					     &nomem);

		noniptotal += readlen;
		continue;
	    }
	    /* 
	     * move past data link header, if present
	     */

	    ippacket = (struct iphdr *) packet;
	    iphlen = ippacket->ihl * 4;
	    transpacket = (struct tcphdr *) (packet + iphlen);

	    /*
	     * Compute and verify IP header checksum
	     */

	    hdrcsum = ippacket->check;
	    ippacket->check = 0;
	    hdrchk = in_cksum((u_short *) ippacket, iphlen);

	    if (hdrcsum != hdrchk)
		continue;

	    nbplen = ntohs(ippacket->tot_len);

	    iptotal += nbplen;
	    if (ippacket->protocol == IPPROTO_TCP) {
		tcptotal += ntohs(ippacket->tot_len);

		/*
		 * TCP fragment handling policy:
		 *    IP fragment offset == 0 && !(more frags); continue
		 *    IP fragment offset != 0 {
		 *        pass fragment to fragment processor
		 *        if first fragment already in {
		 *            pass sport, dport, and bytes
		 *        }
		 *        if first frag not in; get next frame
		 *    }
		 *       DM
		 *      011 0 0000 0000 0000 
		 *      001 1 1111 1111 1111
		 *          3    F    F    F
		 */

		if ((ntohs(ippacket->frag_off) & 0x3fff) != 0) {
		    br =
			processfragment(ippacket, &sport, &dport,
					&firstin);
		    if (!firstin)
			continue;
		} else {
		    sport = transpacket->source;
		    dport = transpacket->dest;
		    br = nbplen;
		}

		/*
		 * Apply TCP filter, if active.
		 */

		if ((filtered) &&
		    (!(utfilter(fl, ippacket->saddr, ippacket->daddr,
				ntohs(sport), ntohs(dport), IPPROTO_TCP)))) {
		    show_ip_stats(statwin, iptotal, tcptotal, udptotal,
				  icmptotal, noniptotal);
		    goto cont;
		}
		tcpentry =
		    in_table(&table, ippacket->saddr, ippacket->daddr,
			     ntohs(sport), ntohs(dport), ifname, tcptm,
			     logging, logfile, &nomem);

		/* 
		 * Add a new entry if it doesn't exist, and if it is a 
		 * SYN packet or any packet with more than just headers,
		 * and, to reduce the chances of stales, not a FIN.
		 */

		if ((ntohs(ippacket->frag_off) & 0x3fff) == 0) {	/* first frag only */
		    totalhlen = iphlen + transpacket->doff * 4;

		    if ((tcpentry == NULL) && (!(transpacket->fin)) &&
			((totalhlen < ntohs(ippacket->tot_len)) ||
			 ((totalhlen == ntohs(ippacket->tot_len)) &&
			  (transpacket->syn)))) {

			/*
			 * Ok, so we have a packet.  Add it if this connection
			 * is not yet closed, or if it is a SYN packet.
			 */

			if (!nomem) {
			    tcpentry = addentry(&table,
						(unsigned long)
						ippacket->saddr,
						(unsigned long)
						ippacket->daddr, sport,
						dport, ippacket->protocol,
						ifname, options->revlook,
						rvnfd, options->servnames,
						&nomem);

			    if (tcpentry != NULL)
				printentry(&table,
					   tcpentry->oth_connection,
					   screen_idx, mode);
			}
		    }
		}
		/* 
		 * If we had an addentry() success, we should have no
		 * problem here.  Same thing if we had a table lookup
		 * success.
		 */

		if (tcpentry != NULL) {
		    /* 
		     * Don't bother updating the entry if the connection
		     * has been previously reset.  (Does this really
		     * happen in practice?)
		     */

		    if (!(tcpentry->stat & FLAG_RST)) {
			if (options->revlook) {
			    p_sstat = tcpentry->s_fstat;
			    p_dstat = tcpentry->d_fstat;
			}
			updateentry(&table, tcpentry, transpacket, tpacket,
			            linktype, nbplen,
				    br, ippacket->frag_off, logging,
				    options->mac, options->revlook, rvnfd, logfile,
				    &nomem);

			/*
			 * Log first packet of a TCP connection except if
			 * it's a RST, which was already logged earlier in
			 * updateentry()
			 */
			 
		        if ((tcpentry->pcount == 1) &&
		           (!(tcpentry->stat & FLAG_RST)) && (logging)) {
			    strcpy(msgstring, "first packet");
			    if (transpacket->syn)
			        strcat(msgstring, " (SYN)");

			    writetcplog(logging, logfile,
				    tcpentry, nbplen, options->mac, msgstring);
		        }
			
			if ((options->revlook)
			    && (((p_sstat != RESOLVED)
				 && (tcpentry->s_fstat == RESOLVED))
				|| ((p_dstat != RESOLVED)
				    && (tcpentry->d_fstat == RESOLVED)))) {
			    clearaddr(&table, tcpentry, screen_idx);
			    clearaddr(&table, tcpentry->oth_connection,
				      screen_idx);
			}
			printentry(&table, tcpentry, screen_idx, mode);

			/*
			 * Special cases: Update other direction if it's
			 * an ACK in response to a FIN. 
			 *
			 *         -- or --
			 *
			 * Addresses were just resolved for the other
			 * direction, so we should also do so here.
			 */

			if (((tcpentry->oth_connection->finsent == 2) &&	/* FINed and ACKed */
			     (ntohl(transpacket->seq) ==
			      tcpentry->oth_connection->finack))
			    || ((options->revlook)
				&&
				(((p_sstat
				   != RESOLVED)
				  && (tcpentry->s_fstat == RESOLVED))
				 || ((p_dstat != RESOLVED)
				     && (tcpentry->d_fstat == RESOLVED)))))
			    printentry(&table, tcpentry->oth_connection,
				       screen_idx, mode);
		    }
		}
	    } else {		/* now for the other IP protocols */
		fragment = ((ntohs(ippacket->frag_off) & 0x1fff) != 0);

		if (ippacket->protocol == IPPROTO_UDP)
		    udptotal += nbplen;
		else if (ippacket->protocol == IPPROTO_ICMP) {

		    /*
		     * Cancel the corresponding TCP entry if an ICMP
		     * Destination Unreachable or TTL Exceeded message
		     * is received.
		     */

		    if (((struct icmphdr *) transpacket)->type ==
			ICMP_DEST_UNREACH) process_dest_unreach(&table,
								(char *)
								transpacket,
								ifname,
								&nomem);

		    icmptotal += nbplen;
		} else
		    othptotal += nbplen;

		/*
		 * Add and display non-TCP packet entry if memory is ok,
		 * and if filter permits
		 */

		if ((!nomem) && (othfilterok(ofilter, ippacket->protocol,
					     ippacket->saddr,
					     ippacket->daddr,
					     ntohs(((struct udphdr *)
						    transpacket)->source),
					     ntohs(((struct udphdr *)
						    transpacket)->dest)))) {
		    othpent =
			add_othp_entry(&othptbl, &table, ippacket->saddr,
				       ippacket->daddr, IS_IP,
				       ippacket->protocol, linktype,
				       (char *) tpacket, (char *) transpacket,
				       nbplen, ifname, options->revlook,
				       rvnfd, tcptm, logging, logfile,
				       options->servnames, fragment,
				       &nomem);
		}
	    }

	  cont:
	    show_ip_stats(statwin, iptotal, tcptotal, udptotal, icmptotal,
			  noniptotal);
	}
	gettimeofday(&tv, NULL);
	now = tv.tv_sec;
	unow = tv.tv_sec * 1e+06 + tv.tv_usec;
        
        /* 
         * Print timer at bottom of screen
         */
         
	if (now - timeint >= 5) {
	    printelapsedtime(starttime, now, othptbl.obmaxy - 1, 15,
			     othptbl.borderwin);
	    timeint = now;
	}

	/*
	 * Automatically clear closed/timed out entries
	 */

	if ((options->closedint != 0) &&
	    ((now - closedint) / 60 >= options->closedint)) {
	    flushclosedentries(&table, tcptm, &screen_idx,
			       logging, logfile);
	    refreshtcpwin(&table, screen_idx, mode);
	    closedint = now;
	}

	/*
	 * Update screen at configured intervals.
	 */

	if (((options->updrate != 0)
	     && (now - updtime >= options->updrate))
	    || ((options->updrate == 0)
		&& (unow - updtime_usec >= DEFAULT_UPDATE_DELAY))) {
	    update_panels();
	    doupdate();
	    updtime = now;
	    updtime_usec = unow;
	}
	
	/*
	 * Terminate facility should a lifetime be specified at the
	 * command line
	 */
	if ((facilitytime != 0)
	    && (((now - starttime) / 60) >= facilitytime))
	    exitloop = 1;

        /*
         * Close and rotate log file if signal was received
         */
	if ((rotate_flag == 1) && (logging)) {
	    announce_rotate_prepare(logfile);
	    write_tcp_unclosed(logging, logfile, &table);
	    rotate_logfile(&logfile, target_logname);
	    announce_rotate_complete(logfile);
	    rotate_flag = 0;
	}
    }

    if (get_instance_count(ITRAFMONCOUNTFILE) <= 1)
        killrvnamed();

    if (options->servnames)
	endservent();

    close_rvn_socket(rvnfd);
    close(isdn_fd);

    if ((options->promisc) && (is_last_instance())) {
	load_promisc_list(&promisc_list);
	srpromisc(0, promisc_list);
	destroy_promisc_list(&promisc_list);
    }
    adjust_instance_count(PROCCOUNTFILE, -1);
    adjust_instance_count(ITRAFMONCOUNTFILE, -1);

    del_panel(table.tcppanel);
    del_panel(table.borderpanel);
    del_panel(othptbl.othppanel);
    del_panel(othptbl.borderpanel);
    del_panel(statpanel);
    update_panels();
    doupdate();
    delwin(table.tcpscreen);
    delwin(table.borderwin);
    delwin(othptbl.othpwin);
    delwin(othptbl.borderwin);
    delwin(statwin);
    close(fd);
    destroytcptable(&table);
    destroyothptable(&othptbl);
    destroyfraglist();
    destroy_isdn_table(&isdntable);
    writelog(logging, logfile,
	     "******** IP traffic monitor stopped ********\n");
    unmark_facility(IPMONIDFILE, ifptr);
    if (logfile != NULL)
	fclose(logfile);
	
    strcpy(current_logfile, "");

    signal(SIGUSR1, SIG_DFL);

    return;
}
