
/***

utfilter.c  - UDP/TCP display filter module
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <curses.h>
#include <panel.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
#include <netdb.h>
#include <ctype.h>
#include "dirs.h"
#include "getpath.h"
#include "deskman.h"
#include "attrs.h"
#include "menurt.h"
#include "utfdefs.h"
#include "utfilter.h"
#include "fltedit.h"
#include "input.h"
#include "error.h"
#include "instances.h"

extern int daemonized;

/*
 * Generate a string representation of a number to be used as a name.
 */

void genname(unsigned long n, char *m)
{
    sprintf(m, "%lu", n);
}

void listfileerr(int code)
{
    if (code == 1)
	write_error("Error loading TCP/UDP filter list file", daemonized);
    else
	write_error("Error writing TCP/UDP filter list file", daemonized);
}

void gethostparams(struct hostparams *hp, int *aborted,
		   char *initsrc, char *initsmask, char *initsport,
		   char *initdest, char *initdmask, char *initdport,
		   char *initinex)
{
    struct FIELDLIST fieldlist;
    struct FIELD *dlist;
    WINDOW *dlgwin;
    PANEL *dlgpanel;

    dlgwin = newwin(12, 80, (LINES - 12) / 2, (COLS - 80) / 2);
    dlgpanel = new_panel(dlgwin);

    wattrset(dlgwin, BOXATTR);
    colorwin(dlgwin);
    box(dlgwin, ACS_VLINE, ACS_HLINE);
    wmove(dlgwin, 0, 26);
    wprintw(dlgwin, " First ");

    wmove(dlgwin, 0, 52);
    wprintw(dlgwin, " Second ");

    wattrset(dlgwin, STDATTR);
    wmove(dlgwin, 2, 2);
    wprintw(dlgwin, "Host name/IP address:");
    wmove(dlgwin, 4, 2);
    wprintw(dlgwin, "Wildcard mask:");
    wmove(dlgwin, 6, 2);
    wprintw(dlgwin, "Port:");
    wmove(dlgwin, 8, 2);
    wprintw(dlgwin, "Include/Exclude (I/E):");
    wmove(dlgwin, 10, 2);
    tabkeyhelp(dlgwin);
    wmove(dlgwin, 10, 20);
    stdkeyhelp(dlgwin);
    update_panels();
    doupdate();

    initfields(&fieldlist, 8, 52, (LINES - 12) / 2 + 1,
	       (COLS - 80) / 2 + 27);

    addfield(&fieldlist, 25, 1, 0, initsrc);
    addfield(&fieldlist, 25, 3, 0, initsmask);
    addfield(&fieldlist, 5, 5, 0, initsport);
    addfield(&fieldlist, 25, 1, 26, initdest);
    addfield(&fieldlist, 25, 3, 26, initdmask);
    addfield(&fieldlist, 5, 5, 26, initdport);
    addfield(&fieldlist, 1, 7, 0, initinex);

    dlist = fieldlist.list->nextfield->nextfield->nextfield;

    fillfields(&fieldlist, aborted);
    strcpy(hp->s_fqdn, fieldlist.list->buf);
    strcpy(hp->s_mask, fieldlist.list->nextfield->buf);
    hp->sport = atoi(fieldlist.list->nextfield->nextfield->buf);

    strcpy(hp->d_fqdn, dlist->buf);
    strcpy(hp->d_mask, dlist->nextfield->buf);
    hp->dport = atoi(dlist->nextfield->nextfield->buf);
    hp->reverse = toupper(dlist->nextfield->nextfield->nextfield->buf[0]);

    destroyfields(&fieldlist);
    del_panel(dlgpanel);
    delwin(dlgwin);
    update_panels();
    doupdate();

    if (*aborted)
	return;

}

int mark_filter_change(int protocol)
{
    int resp;

    if (protocol == F_TCP) {
	if (!facility_active(TCPFLTIDFILE))
	    mark_facility(TCPFLTIDFILE, "TCP filter change");
	else {
	    errbox("TCP filter data file in use; try again later",
		   ANYKEY_MSG, &resp);
	    return 0;
	}
    } else {
	if (!facility_active(UDPFLTIDFILE))
	    mark_facility(UDPFLTIDFILE, "UDP filter change");
	else {
	    errbox("UDP filter data file in use; try again later",
		   ANYKEY_MSG, &resp);
	    return 0;
	}
    }
    return 1;
}

void clear_flt_tag(int protocol)
{
    if (protocol == F_TCP)
	unlink(TCPFLTIDFILE);
    else
	unlink(UDPFLTIDFILE);
}

void definefilter(int protocol, int *aborted)
{
    struct filterfileent ffile;
    struct hostparams hp;
    char fntemp[14];

    int pfd;
    int bw;
    int resp;

    /*
     * Lock facility
     */

    if (!mark_filter_change(protocol))
	return;

    get_filter_description(ffile.desc, aborted, "");

    if (*aborted) {
	clear_flt_tag(protocol);
	return;
    }
    genname(time((time_t *) NULL), fntemp);

    pfd =
	open(get_path(T_WORKDIR, fntemp), O_CREAT | O_WRONLY | O_TRUNC,
	     S_IRUSR | S_IWUSR);
    if (pfd < 0) {
	errbox("Cannot create TCP/UDP filter record file", ANYKEY_MSG,
	       &resp);
	*aborted = 1;
	clear_flt_tag(protocol);
	return;
    }
    do {
	gethostparams(&hp, aborted, "", "", "", "0.0.0.0", "0.0.0.0", "0",
		      "I");
	if (!(*aborted)) {
	    bw = write(pfd, &hp, sizeof(struct hostparams));

	    if (bw < 0) {
		errbox("Unable to write TCP/UDP filter record", ANYKEY_MSG,
		       &resp);
		close(pfd);
		clear_flt_tag(protocol);
		return;
	    }
	}
    } while (!(*aborted));

    close(pfd);

    if (protocol == F_TCP)
	pfd =
	    open(TCPFLNAME, O_CREAT | O_WRONLY | O_APPEND,
		 S_IRUSR | S_IWUSR);
    else
	pfd =
	    open(UDPFLNAME, O_CREAT | O_WRONLY | O_APPEND,
		 S_IRUSR | S_IWUSR);

    if (pfd < 0) {
	listfileerr(1);
	clear_flt_tag(protocol);
	return;
    }
    strcpy(ffile.filename, fntemp);
    bw = write(pfd, &ffile, sizeof(struct filterfileent));
    if (bw < 0)
	listfileerr(2);

    close(pfd);

    clear_flt_tag(protocol);
}

void init_filter_table(struct filterlist *fl)
{
    fl->head = fl->tail = NULL;
}

void displayfilters(struct ffnode *ffiles, WINDOW * win)
{
    unsigned int row = 0;
    struct ffnode *pptr;

    pptr = ffiles;

    wattrset(win, STDATTR);
    do {
	wmove(win, row, 2);
	wprintw(win, pptr->ffe.desc);
	row++;
	pptr = pptr->next_entry;
    } while ((row < 8) && (pptr != NULL));

    update_panels();
    doupdate();
}

void operate_select(struct ffnode *ffiles, WINDOW * win,
		    struct ffnode **item, int *aborted)
{
    unsigned int row = 0;
    struct ffnode *pptr;
    int ch;
    int exitloop = 0;

    listkeyhelp();
    update_panels();
    doupdate();
    pptr = ffiles;
    do {
	wattrset(win, PTRATTR);
	wmove(win, row, 1);
	waddch(win, ACS_RARROW);
	ch = wgetch(win);
	wmove(win, row, 1);
	wprintw(win, " ");
	wattrset(win, STDATTR);

	switch (ch) {
	case KEY_UP:
	    if (pptr->prev_entry != NULL) {
		if (row > 0)
		    row--;
		else {
		    wscrl(win, -1);
		    wmove(win, 0, 0);
		    wprintw(win, "%58c", ' ');
		    wmove(win, 0, 2);
		    wprintw(win, pptr->prev_entry->ffe.desc);
		}
		pptr = pptr->prev_entry;
	    }
	    break;
	case KEY_DOWN:
	    if (pptr->next_entry != NULL) {
		if (row < 7)
		    row++;
		else {
		    wscrl(win, 1);
		    scrollok(win, 0);
		    wmove(win, 7, 0);
		    wprintw(win, "%58c", ' ');
		    scrollok(win, 1);
		    wmove(win, 7, 2);
		    wprintw(win, pptr->next_entry->ffe.desc);
		}
		pptr = pptr->next_entry;
	    }
	    break;
	case 12:
	case 'l':
	case 'L':
	    refresh_screen();
	    break;
	case 13:
	    exitloop = 1;
	    *aborted = 0;
	    break;
	case 27:
	case 24:
	case 'X':
	case 'x':
	case 'Q':
	case 'q':
	    exitloop = 1;
	    *aborted = 1;
	    break;
	}
	update_panels();
	doupdate();
    } while (!exitloop);

    *item = pptr;
}

void pickafilter(struct ffnode *ffiles,
		 struct ffnode **fltfile, int *aborted)
{
    WINDOW *borderwin;
    PANEL *borderpanel;
    WINDOW *selectwin;
    PANEL *selectpanel;

    borderwin = newwin(10, 60, (LINES - 10) / 2 - 2, (COLS - 60) / 2 - 2);
    borderpanel = new_panel(borderwin);
    wattrset(borderwin, BOXATTR);
    box(borderwin, ACS_VLINE, ACS_HLINE);

    selectwin = newwin(8, 58, (LINES - 8) / 2 - 2, (COLS - 58) / 2 - 2);
    selectpanel = new_panel(selectwin);
    wattrset(selectwin, STDATTR);
    colorwin(selectwin);
    stdwinset(selectwin);
    wtimeout(selectwin, -1);

    displayfilters(ffiles, selectwin);
    operate_select(ffiles, selectwin, fltfile, aborted);

    del_panel(borderpanel);
    delwin(borderwin);
    del_panel(selectpanel);
    delwin(selectwin);
    update_panels();
    doupdate();
}


void selectfilter(unsigned int protocol,
		  struct filterfileent *ffe, int *aborted)
{
    struct ffnode *fltfile;
    struct ffnode *ffiles;

    if (loadfilterlist(protocol, &ffiles) == 1) {
	listfileerr(1);
	*aborted = 1;
	destroyfilterlist(ffiles);
	return;
    }
    pickafilter(ffiles, &fltfile, aborted);

    if (!(*aborted))
	*ffe = fltfile->ffe;

    destroyfilterlist(ffiles);
}

unsigned long int nametoaddr(char *ascname, int *err)
{
    unsigned long int result;
    struct hostent *he;
    char imsg[45];
    struct in_addr inp;
    int resolv_err = 0;

    resolv_err = inet_aton(ascname, &inp);
    if (resolv_err == 0) {
	snprintf(imsg, 45, "Resolving %s", ascname);
	indicate(imsg);
	he = gethostbyname(ascname);
	if (he != NULL)
	    bcopy((he->h_addr_list)[0], &result, he->h_length);
	else {
	    snprintf(imsg, 45, "Unable to resolve %s", ascname);
	    write_error(imsg, daemonized);
	    *err = 1;
	    return (-1);
	}
    } else
	result = inp.s_addr;

    return (result);
    *err = 0;
}

/* 
 * Loads the filter from the filter file
 */

int loadfilter(char *filename, struct filterlist *fl, int resolve)
{
    struct filterent *fe;
    int pfd;
    unsigned int idx = 0;
    int br;
    int resolv_err = 0;

    init_filter_table(fl);

    pfd = open(filename, O_RDONLY);

    if (pfd < 0) {
	write_error("Error opening TCP/UDP filter file", daemonized);
	fl->head = NULL;
	return 1;
    }
    do {
	fe = malloc(sizeof(struct filterent));
	br = read(pfd, &(fe->hp), sizeof(struct hostparams));

	if (br > 0) {
	    fe->index = idx;
	    if (resolve) {
		fe->saddr = nametoaddr(fe->hp.s_fqdn, &resolv_err);
		fe->daddr = nametoaddr(fe->hp.d_fqdn, &resolv_err);

		if (resolv_err) {
		    free(fe);
		    continue;
		}

		fe->smask = inet_addr(fe->hp.s_mask);
		fe->dmask = inet_addr(fe->hp.d_mask);
	    }
	    if (fl->head == NULL) {
		fl->head = fe;
		fe->prev_entry = NULL;
	    } else {
		fl->tail->next_entry = fe;
		fe->prev_entry = fl->tail;
	    }
	    fe->next_entry = NULL;
	    fl->tail = fe;
	    idx++;
	} else {
	    free(fe);
	}
    } while (br > 0);

    if (br == 0)
	close(pfd);

    return 0;
}

void savefilter(unsigned int protocol, char *filename,
		struct filterlist *fl)
{
    struct filterent *fe = fl->head;
    int pfd;
    int bw;
    int resp;

    pfd = open(filename, O_CREAT | O_TRUNC | O_WRONLY, S_IRUSR | S_IWUSR);

    while (fe != NULL) {
	bw = write(pfd, &(fe->hp), sizeof(struct hostparams));
	if (bw < 0) {
	    errbox("Unable to save filter changes", ANYKEY_MSG, &resp);
	    clear_flt_tag(protocol);
	    return;
	}
	fe = fe->next_entry;
    }

    close(pfd);
}

void marktcpfilter(char *filename)
{
    int fd;
    int resp;

    fd =
	open(TCPFILTERSAVE, O_WRONLY | O_CREAT | O_TRUNC,
	     S_IRUSR | S_IWUSR);

    if (write(fd, filename, FLT_FILENAME_MAX) < 0) {
	errbox("Unable to save TCP/UDP filter name", ANYKEY_MSG, &resp);

	if (fd > 0)
	    close(fd);

	return;
    }
    close(fd);
}

void loadsavedtcpfilter(struct filterlist *fl, int *filtered)
{
    int fd;
    char filename[FLT_FILENAME_MAX];
    int resp;

    fd = open(TCPFILTERSAVE, O_RDONLY);

    if (fd < 0)
	return;

    bzero(filename, FLT_FILENAME_MAX);
    if (read(fd, filename, FLT_FILENAME_MAX) < 0) {
	errbox("Unable to retrieve saved TCP/UDP filter", ANYKEY_MSG,
	       &resp);
	if (fd > 0)
	    close(fd);

	*filtered = 0;
	return;
    }
    close(fd);

    loadfilter(filename, fl, FLT_RESOLVE);

    if (fl->head == NULL)
	*filtered = 0;
    else
	*filtered = 1;
}

/*
 * The TCP/UDP display filter
 */

int utfilter(struct filterlist *fl,
	     unsigned long source, unsigned long dest,
	     unsigned int sport, unsigned int dport, unsigned int protocol)
{
    struct filterent *fe;
    unsigned long fsaddr, fdaddr;
    unsigned long csaddr, cdaddr;
    unsigned long crsaddr, crdaddr;
    int flt_expr1 = 0;
    int flt_expr2 = 0;

    fe = fl->head;

    while (fe != NULL) {
	fsaddr = fe->saddr & fe->smask;
	fdaddr = fe->daddr & fe->dmask;
	csaddr = source & fe->smask;
	cdaddr = dest & fe->dmask;
	crsaddr = source & fe->dmask;
	crdaddr = dest & fe->smask;

	/*
	 * Just using two variables to make it easier to read
	 */

	flt_expr1 = ((csaddr == fsaddr)
		     && ((fe->hp.sport == sport) || (fe->hp.sport == 0)))
	    && ((cdaddr == fdaddr)
		&& ((fe->hp.dport == dport) || (fe->hp.dport == 0)));

	flt_expr2 = ((crsaddr == fdaddr)
		     && ((fe->hp.dport == sport) || (fe->hp.dport == 0)))
	    && ((crdaddr == fsaddr)
		&& ((fe->hp.sport == dport) || (fe->hp.sport == 0)));

	if (flt_expr1 || flt_expr2) {
	    if (toupper(fe->hp.reverse) == 'E')	/* Exclusive */
		return 0;

	    return 1;
	}
	fe = fe->next_entry;
    }

    return 0;
}

/* 
 * Remove a currently applied filter from memory
 */

void destroyfilter(struct filterlist *fl)
{
    struct filterent *fe;
    struct filterent *cfe;

    if (fl->head != NULL) {
	fe = fl->head;
	cfe = fl->head->next_entry;

	do {
	    free(fe);
	    fe = cfe;
	    if (cfe != NULL)
		cfe = cfe->next_entry;
	} while (fe != NULL);

	fl->head = fl->tail = NULL;
    }
}


/*
 * Delete a filter record from the disk
 */

void delfilter(unsigned int protocol, int *aborted)
{
    struct ffnode *fltfile;
    struct ffnode *fltlist;

    if (!mark_filter_change(protocol))
	return;

    if (loadfilterlist(protocol, &fltlist) == 1) {
	*aborted = 1;
	listfileerr(1);
	destroyfilterlist(fltlist);
	clear_flt_tag(protocol);
	return;
    }
    pickafilter(fltlist, &fltfile, aborted);

    if (*aborted) {
	clear_flt_tag(protocol);
	return;
    }
    unlink(get_path(T_WORKDIR, fltfile->ffe.filename));

    if (fltfile->prev_entry == NULL) {
	fltlist = fltlist->next_entry;
	if (fltlist != NULL)
	    fltlist->prev_entry = NULL;
    } else {
	fltfile->prev_entry->next_entry = fltfile->next_entry;

	if (fltfile->next_entry != NULL)
	    fltfile->next_entry->prev_entry = fltfile->prev_entry;
    }

    free(fltfile);

    save_filterlist(protocol, fltlist);
    clear_flt_tag(protocol);
    *aborted = 0;
}

void makefiltermenu(struct MENU *menu)
{
    initmenu(menu, 8, 31, (LINES - 7) / 2 + 3, (COLS - 31) / 2 + 15);
    additem(menu, " ^D^efine new filter...",
	    "Defines a new set of filter parameters");
    additem(menu, " ^A^pply filter...",
	    "Applies filter to be used for TCP display");
    additem(menu, " Detac^h^ filter",
	    "Removes the currently applied filter");
    additem(menu, " ^E^dit filter...",
	    "Modifies existing TCP filter data");
    additem(menu, " Dele^t^e filter...",
	    "Removes a TCP filter from the filter list");
    additem(menu, " E^x^it menu", "Returns to the main menu");
}

void makeudpfiltermenu(struct MENU *menu)
{
    initmenu(menu, 9, 31, (LINES - 11) / 2, (COLS - 30) / 2 - 21);
    additem(menu, " Show ^a^ll UDP packets",
	    "Permits display of all UDP packets");
    additem(menu, " Show ^n^o UDP packets",
	    "Omits all UDP packets from display");
    additem(menu, " ^D^efine custom UDP filter...",
	    "Defines a custom UDP filter");
    additem(menu, " Apply ^c^ustom UDP filter...",
	    "Applies a custom UDP filter");
    additem(menu, " ^E^dit custom UDP filter...",
	    "Modifies a custom UDP filter");
    additem(menu, " Dele^t^e custom UDP filter...",
	    "Removes a custom UDP filter");
    additem(menu, " E^x^it menu", "Returns to the main menu");
}


/* display a menu and perform appropriate filter action */

void udpfilterselect(struct filterlist *fl,
		     unsigned int *filtercode,
		     char *filename, int *faborted)
{
    struct MENU fmenu;
    struct filterfileent ffe;
    unsigned int frow;

    makeudpfiltermenu(&fmenu);

    frow = 1;

    do {
	showmenu(&fmenu);
	operatemenu(&fmenu, &frow, faborted);

	switch (frow) {
	case 1:
	    *filtercode = 1;
	    infobox("All UDP packets will be displayed", ANYKEY_MSG);
	    break;
	case 2:
	    if (*filtercode == 2)
		destroyfilter(fl);

	    *filtercode = 0;
	    infobox("No UDP packets will be displayed", ANYKEY_MSG);
	    break;
	case 3:
	    definefilter(F_UDP, faborted);
	    break;
	case 4:
	    selectfilter(F_UDP, &ffe, faborted);
	    if (!(*faborted)) {
		strncpy(filename, get_path(T_WORKDIR, ffe.filename),
			FLT_FILENAME_MAX - 1);
		loadfilter(filename, fl, FLT_RESOLVE);
		*filtercode = 2;
	    }
	    break;
	case 5:
	    editfilter(F_UDP, faborted);
	    break;
	case 6:
	    delfilter(F_UDP, faborted);
	    if (!(*faborted))
	        infobox("Custom UDP filter deleted", ANYKEY_MSG);
	    break;
	}
    } while (frow != 7);

    destroymenu(&fmenu);
    update_panels();
    doupdate();
}


/* 
 * Display a menu and perform appropriate filter operation
 */

void tcpfilterselect(struct filterlist *fl,
		     unsigned int *filtered, int *faborted)
{
    struct MENU fmenu;
    unsigned int frow;
    struct filterfileent ffe;
    char filename[FLT_FILENAME_MAX];

    makefiltermenu(&fmenu);

    frow = 1;
    do {
	showmenu(&fmenu);
	operatemenu(&fmenu, &frow, faborted);

	switch (frow) {
	case 1:
	    definefilter(F_TCP, faborted);
	    break;
	case 2:
	    selectfilter(F_TCP, &ffe, faborted);
	    if (!(*faborted)) {
		strncpy(filename, get_path(T_WORKDIR, ffe.filename),
			FLT_FILENAME_MAX - 1);
		if (!loadfilter(filename, fl, FLT_RESOLVE)) {
		    *filtered = 1;
		    marktcpfilter(filename);
		}
	    }
	    break;
	case 3:
	    if (*filtered) {
		destroyfilter(fl);
		*filtered = 0;
	    }
	    unlink(TCPFILTERSAVE);
	    infobox("TCP filter deactivated", ANYKEY_MSG);
	    break;
	case 4:
	    editfilter(F_TCP, faborted);
	    break;
	case 5:
	    delfilter(F_TCP, faborted);
	    if (!(*faborted))
	        infobox("TCP filter deleted", ANYKEY_MSG);
	    break;
	}
    } while (frow != 6);

    destroymenu(&fmenu);
    update_panels();
    doupdate();
}
