
/***

landesc.c	- LAN host description management module
		  Currently includes support for Ethernet, PLIP,
		  and FDDI

Copyright (c) Gerard Paul Java 1998

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <curses.h>
#include <panel.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <net/if_arp.h>
#include "deskman.h"
#include "stdwinset.h"
#include "menurt.h"
#include "attrs.h"
#include "dirs.h"
#include "input.h"
#include "error.h"
#include "landesc.h"
#include "links.h"

void etherr(void)
{
    int resp;

    errbox("Unable to open host description file", ANYKEY_MSG, &resp);
}

void loaddesclist(struct desclist *list, unsigned int linktype)
{
    struct desclistent *ptmp = NULL;
    FILE *fd = NULL;
    char descline[140];
    char *desctoken;

    bzero(list, sizeof(list));

    if (linktype == LINK_ETHERNET)
	fd = fopen(ETHFILE, "r");
    else if (linktype == LINK_FDDI)
	fd = fopen(FDDIFILE, "r");

    if (fd == NULL) {
	return;
    }
    while (!feof(fd)) {
	ptmp = malloc(sizeof(struct desclistent));
	if (ptmp == NULL) {
	    printnomem();
	    return;
	}
	bzero(ptmp, sizeof(struct desclistent));
	bzero(descline, 140);
	fgets(descline, 140, fd);

	if (strcmp(descline, "") == 0) {
	    free(ptmp);
	    continue;
	}
	strncpy(ptmp->rec.address, strtok(descline, ":"), 12);
	desctoken = strtok(NULL, "\n");

	if (desctoken != NULL)
	    strncpy(ptmp->rec.desc, desctoken, 64);
	else
	    strcpy(ptmp->rec.desc, "");

	if (list->head == NULL) {
	    list->head = ptmp;
	    ptmp->prev_entry = NULL;
	} else {
	    list->tail->next_entry = ptmp;
	    ptmp->prev_entry = list->tail;
	}

	ptmp->next_entry = NULL;
	list->tail = ptmp;
    }

    fclose(fd);
}

void savedesclist(struct desclist *list, unsigned int linktype)
{
    FILE *fd = NULL;

    struct desclistent *ptmp = list->head;

    if (linktype == LINK_ETHERNET)
	fd = fopen(ETHFILE, "w");
    else if (linktype == LINK_FDDI)
	fd = fopen(FDDIFILE, "w");

    if (fd < 0) {
	etherr();
	return;
    }
    while (ptmp != NULL) {
	fprintf(fd, "%s:%s\n", ptmp->rec.address, ptmp->rec.desc);
	ptmp = ptmp->next_entry;
    }

    fclose(fd);
}

void displayethdescs(struct desclist *list, WINDOW * win)
{
    struct desclistent *ptmp = list->head;
    short i = 0;

    do {
	wmove(win, i, 2);
	wprintw(win, "%s    %s", ptmp->rec.address, ptmp->rec.desc);
	i++;
	ptmp = ptmp->next_entry;
    } while ((i < 18) && (ptmp != NULL));

    update_panels();
    doupdate();
}

void destroydesclist(struct desclist *list)
{
    struct desclistent *ptmp = list->head;
    struct desclistent *ctmp = NULL;

    if (list->head != NULL)
	ctmp = ptmp->next_entry;

    while (ptmp != NULL) {
	free(ptmp);
	ptmp = ctmp;

	if (ctmp != NULL)
	    ctmp = ctmp->next_entry;
    }
}

void operate_descselect(struct desclist *list, struct desclistent **node,
			WINDOW * win, int *aborted)
{
    int ch = 0;
    int i = 0;
    char sp_buf[10];

    int exitloop = 0;

    *node = list->head;

    sprintf(sp_buf, "%%%dc", COLS - 2);

    do {
	wattrset(win, PTRATTR);
	wmove(win, i, 1);
	waddch(win, ACS_RARROW);
	ch = wgetch(win);
	wmove(win, i, 1);
	waddch(win, ' ');
	wattrset(win, STDATTR);

	switch (ch) {
	case KEY_DOWN:
	    if ((*node)->next_entry != NULL) {
		*node = (*node)->next_entry;

		if (i < 17)
		    i++;
		else {
		    wscrl(win, 1);
		    scrollok(win, 0);
		    wmove(win, 17, 0);
		    wprintw(win, sp_buf, ' ');
		    scrollok(win, 1);
		    wmove(win, 17, 2);
		    wprintw(win, "%s    %s", (*node)->rec.address,
			    (*node)->rec.desc);
		}
	    }
	    break;
	case KEY_UP:
	    if ((*node)->prev_entry != NULL) {
		*node = (*node)->prev_entry;

		if (i > 0)
		    i--;
		else {
		    wscrl(win, -1);
		    wmove(win, 0, 0);
		    wprintw(win, sp_buf, ' ');
		    wmove(win, 0, 2);
		    wprintw(win, "%s    %s", (*node)->rec.address,
			    (*node)->rec.desc);
		}
	    }
	    break;
	case 13:
	    exitloop = 1;
	    *aborted = 0;
	    break;
	case 27:
	case 24:
	case 'x':
	case 'X':
	case 'q':
	case 'Q':
	    exitloop = 1;
	    *aborted = 1;
	    break;
	}
    } while (!exitloop);
}

void selectdesc(struct desclist *list, struct desclistent **node,
		int *aborted)
{
    WINDOW *bw;
    PANEL *bp;
    WINDOW *win;
    PANEL *panel;
    int resp;

    if (list->head == NULL) {
	errbox("No descriptions", ANYKEY_MSG, &resp);
	return;
    }
    bw = newwin(20, COLS, 2, 0);
    bp = new_panel(bw);
    wattrset(bw, BOXATTR);
    box(bw, ACS_VLINE, ACS_HLINE);

    win = newwin(18, COLS - 2, 3, 1);
    panel = new_panel(win);
    stdwinset(win);
    wattrset(win, STDATTR);
    colorwin(win);

    listkeyhelp();

    wmove(bw, 0, 2);
    wprintw(bw, " MAC Address ");
    wmove(bw, 0, 18);
    wprintw(bw, " Address Description ");

    displayethdescs(list, win);
    operate_descselect(list, node, win, aborted);

    del_panel(panel);
    delwin(win);
    del_panel(bp);
    delwin(bw);

    update_panels();
    doupdate();
}

void descdlg(struct descrec *rec, char *initaddr, char *initdesc,
	     int *aborted)
{
    WINDOW *win;
    PANEL *panel;
    struct FIELDLIST fieldlist;

    win = newwin(8, 70, 8, (COLS - 70) / 2);
    panel = new_panel(win);

    wattrset(win, BOXATTR);
    colorwin(win);
    box(win, ACS_VLINE, ACS_HLINE);
    wmove(win, 6, 2 * COLS / 80);
    tabkeyhelp(win);
    wmove(win, 6, 20 * COLS / 80);
    stdkeyhelp(win);

    wattrset(win, STDATTR);
    wmove(win, 2, 2 * COLS / 80);
    wprintw(win, "MAC Address:");
    wmove(win, 4, 2 * COLS / 80);
    wprintw(win, "Description:");
    initfields(&fieldlist, 3, 52, 10, (COLS - 52) / 2 + 6 * COLS / 80);
    addfield(&fieldlist, 12, 0, 0, initaddr);
    addfield(&fieldlist, 50, 2, 0, initdesc);

    fillfields(&fieldlist, aborted);

    if (!(*aborted)) {
	strcpy(rec->address, fieldlist.list->buf);
	strcpy(rec->desc, fieldlist.list->nextfield->buf);
    }
    destroyfields(&fieldlist);
    del_panel(panel);
    delwin(win);
}

void addethdesc(struct desclist *list)
{
    struct descrec rec;
    int aborted;
    struct desclistent *ptmp;

    descdlg(&rec, "", "", &aborted);

    if (!aborted) {
	ptmp = malloc(sizeof(struct desclistent));
	if (list->head == NULL) {
	    list->head = ptmp;
	    ptmp->prev_entry = NULL;
	} else {
	    ptmp->prev_entry = list->tail;
	    list->tail->next_entry = ptmp;
	}

	list->tail = ptmp;
	ptmp->next_entry = NULL;
	memcpy(&(ptmp->rec), &rec, sizeof(struct descrec));
    }
    update_panels();
    doupdate();
}

void editethdesc(struct desclist *list)
{
    struct desclistent *ptmp;
    int aborted;

    selectdesc(list, &ptmp, &aborted);

    if (!aborted)
	descdlg(&(ptmp->rec), ptmp->rec.address, ptmp->rec.desc, &aborted);
}

void delethdesc(struct desclist *list)
{
    struct desclistent *ptmp;
    int aborted;

    selectdesc(list, &ptmp, &aborted);

    if (!aborted) {
	if (ptmp->prev_entry != NULL)
	    ptmp->prev_entry->next_entry = ptmp->next_entry;
	else
	    list->head = ptmp->next_entry;

	if (ptmp->next_entry != NULL)
	    ptmp->next_entry->prev_entry = ptmp->prev_entry;
	else
	    list->tail = ptmp->prev_entry;

	free(ptmp);
    }
}


void ethdescmgr(unsigned int linktype)
{
    struct MENU menu;
    int row = 1;
    int aborted;
    struct desclist list;

    loaddesclist(&list, linktype);

    initmenu(&menu, 6, 31, (LINES - 6) / 2, (COLS - 31) / 2);
    additem(&menu, " ^A^dd description...",
	    "Adds a description for an Ethernet address");
    additem(&menu, " ^E^dit description...",
	    "Modifies an existing Ethernet address description");
    additem(&menu, " ^D^elete description...",
	    "Deletes an existing Ethernet address description");
    additem(&menu, " E^x^it menu", "Returns to the main menu");

    do {
	showmenu(&menu);
	operatemenu(&menu, &row, &aborted);

	switch (row) {
	case 1:
	    addethdesc(&list);
	    break;
	case 2:
	    editethdesc(&list);
	    break;
	case 3:
	    delethdesc(&list);
	    break;
	}
    } while (row != 4);

    destroymenu(&menu);
    update_panels();
    doupdate();
    savedesclist(&list, linktype);
}
