/***

fltedit.c	- TCP/UDP Filter Editing Facility

Copyright (c) Gerard Paul Java 1999

This software is open-source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <curses.h>
#include <panel.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include "stdwinset.h"
#include "utfdefs.h"
#include "utfilter.h"
#include "dirs.h"
#include "getpath.h"
#include "attrs.h"
#include "deskman.h"
#include "error.h"

void print_hostparam_line(struct filterent *fe, int idx, WINDOW * win)
{
    wattrset(win, STDATTR);
    mvwprintw(win, idx, 2, "%76c", ' ');
    mvwaddnstr(win, idx, 2, fe->hp.s_fqdn, 15);
    wprintw(win, "/");
    waddnstr(win, fe->hp.s_mask, 12);
    wprintw(win, ":%u", fe->hp.sport);
    mvwaddnstr(win, idx, 38, fe->hp.d_fqdn, 15);
    wprintw(win, "/");
    waddnstr(win, fe->hp.d_mask, 12);
    wprintw(win, ":%u", fe->hp.dport);
    mvwprintw(win, idx, 76, "%c", toupper(fe->hp.reverse));
}

void update_hp_screen(struct filterlist *fl,
		      struct filterent *firstvisible, WINDOW * win)
{
    struct filterent *ftmp = firstvisible;
    int i;

    scrollok(win, 0);
    wattrset(win, STDATTR);
    if (firstvisible == NULL) {
	mvwprintw(win, 0, 0, "%78c", ' ');
	return;
    }

    for (i = 0; i <= 12; i++) {
	if (ftmp != NULL) {
	    print_hostparam_line(ftmp, i, win);
	    ftmp = ftmp->next_entry;
	} else {
	    mvwprintw(win, i, 0, "%78c", ' ');
	}
    }
    scrollok(win, 1);
}

int new_hp_entry(struct filterent **ftemp)
{
    int resp;
    *ftemp = malloc(sizeof(struct filterent));
    if (*ftemp == NULL) {
	errbox("No memory for new filter entry", ANYKEY_MSG, &resp);
	return 0;
    }
    return 1;
}

void modify_host_parameters(unsigned int protocol, struct filterlist *fl)
{
    WINDOW *bwin;
    PANEL *bpanel;
    WINDOW *win;
    PANEL *panel;
    struct filterent *fe;
    struct filterent *ftemp;

    struct filterent *firstvisible = NULL;
    unsigned int idx = 0;
    int endloop_local = 0;
    int ch;
    int gh_aborted = 0;
    char s_portstr[8];
    char d_portstr[8];
    char inexstr[2];
    struct hostparams hptemp;

    bwin = newwin(15, 80, (LINES - 15) / 2, (COLS - 80) / 2);
    bpanel = new_panel(bwin);
    win = newwin(13, 78, (LINES - 13) / 2, (COLS - 78) / 2);
    panel = new_panel(win);

    wattrset(bwin, BOXATTR);
    box(bwin, ACS_VLINE, ACS_HLINE);
    mvwprintw(bwin, 0, 2, " First Host/Mask:Port ");
    mvwprintw(bwin, 0, 38, " Second Host/Mask:Port ");
    mvwprintw(bwin, 0, 74, " I/E ");
    wmove(bwin, 14, 1);

    if (protocol == F_TCP)
	wprintw(bwin, " TCP");
    else
	wprintw(bwin, " UDP");

    wprintw(bwin, " Filter Data ");
    stdwinset(win);
    wattrset(win, STDATTR);
    colorwin(win);

    move(LINES - 1, 1);
    printkeyhelp("Up/Down", "-move ptr ", stdscr);
    printkeyhelp("I", "-insert entry ", stdscr);
    printkeyhelp("A", "-append to list ", stdscr);
    printkeyhelp("D", "-delete entry ", stdscr);
    printkeyhelp("X/Ctrl+X", "-exit", stdscr);

    update_panels();
    doupdate();

    firstvisible = fl->head;
    update_hp_screen(fl, fl->head, win);

    idx = 0;
    fe = fl->head;

    do {
	wattrset(win, PTRATTR);
	if (fe != NULL)
	    mvwaddch(win, idx, 1, ACS_RARROW);
	ch = wgetch(win);
	mvwprintw(win, idx, 1, " ");

	switch (ch) {
	case KEY_UP:
	    if (fl->head != NULL) {
		if (fe->prev_entry != NULL) {
		    if (idx > 0)
			idx--;
		    else {
			wscrl(win, -1);
			scrollok(win, 0);
			mvwprintw(win, 0, 0, "%78c", ' ');
			print_hostparam_line(fe->prev_entry, idx, win);
			scrollok(win, 1);
			firstvisible = firstvisible->prev_entry;
		    }
		    fe = fe->prev_entry;
		}
	    }
	    break;
	case KEY_DOWN:
	    if (fl->head != NULL) {
		if (fe->next_entry != NULL) {
		    if (idx < 12)
			idx++;
		    else {
			wscrl(win, 1);
			scrollok(win, 0);
			mvwprintw(win, 12, 0, "%78c", ' ');
			print_hostparam_line(fe->next_entry, idx, win);
			scrollok(win, 1);
			firstvisible = firstvisible->next_entry;
		    }
		    fe = fe->next_entry;
		}
	    }
	    break;
	case 'i':
	case 'I':
	case KEY_IC:
	    if (!new_hp_entry(&ftemp))
		break;

	    gethostparams(&(ftemp->hp), &gh_aborted, "", "", "", "0.0.0.0",
			  "0.0.0.0", "0", "I");

	    if (gh_aborted) {
		free(ftemp);
		continue;
	    }

	    if (fl->head == NULL) {
		ftemp->next_entry = ftemp->prev_entry = NULL;
		fl->head = fl->tail = ftemp;
		firstvisible = fl->head;
		idx = 0;
	    } else {
		ftemp->next_entry = fe;
		ftemp->prev_entry = fe->prev_entry;

		/*
		 * Point firstvisible at new entry if we inserted at the
		 * top of the list.
		 */

		if (ftemp->prev_entry == NULL) {
		    fl->head = ftemp;
		    firstvisible = ftemp;
		} else
		    fe->prev_entry->next_entry = ftemp;

		fe->prev_entry = ftemp;
	    }

	    if (ftemp->next_entry == NULL)
		fl->tail = ftemp;

	    fe = ftemp;
	    update_hp_screen(fl, firstvisible, win);
	    break;
	case 'a':
	case 'A':
	case 1:
	    if (!new_hp_entry(&ftemp))
		break;

	    gethostparams(&(ftemp->hp), &gh_aborted, "", "", "", "0.0.0.0",
			  "0.0.0.0", "0", "I");

	    if (gh_aborted) {
		free(ftemp);
		continue;
	    }

	    /*
	     * Add new node to the end of the list (or to the head if the
	     * list is empty.
	     */
	    if (fl->tail != NULL) {
		fl->tail->next_entry = ftemp;
		ftemp->prev_entry = fl->tail;
	    } else {
		fl->head = ftemp;
		fl->tail = ftemp;
		ftemp->prev_entry = ftemp->next_entry = NULL;
		firstvisible = fl->head;
		fe = ftemp;
		idx = 0;
	    }

	    ftemp->next_entry = NULL;
	    fl->tail = ftemp;
	    update_hp_screen(fl, firstvisible, win);
	    break;
	case 'd':
	case 'D':
	case KEY_DC:
	    if (fl->head != NULL) {
		/*
		 * Move firstvisible down if it's pointing to the target
		 * entry.
		 */

		if (firstvisible == fe)
		    firstvisible = fe->next_entry;

		/*
		 * Detach target node from list.
		 */
		if (fe->next_entry != NULL)
		    fe->next_entry->prev_entry = fe->prev_entry;
		else
		    fl->tail = fe->prev_entry;

		if (fe->prev_entry != NULL)
		    fe->prev_entry->next_entry = fe->next_entry;
		else
		    fl->head = fe->next_entry;

		/*
		 * Move pointer up if we're deleting the last entry.
		 * The list tail pointer has since been moved to the
		 * previous entry.
		 */
		if (fe->prev_entry == fl->tail) {
		    ftemp = fe->prev_entry;

		    /*
		     * Move screen pointer up. Really adjust the index if
		     * the pointer is anywhere below the top of the screen.
		     */
		    if (idx > 0)
			idx--;
		    else {
			/*
			 * Otherwise scroll the list down, and adjust the
			 * firstvisible pointer to point to the entry
			 * previous to the target.
			 */
			if (ftemp != NULL) {
			    wscrl(win, -1);
			    mvwprintw(win, 0, 0, "%78c", ' ');
			    print_hostparam_line(ftemp, idx, win);
			    firstvisible = ftemp;
			}
		    }
		} else
		    /*
		     * If we reach this point, we're deleting from before
		     * the tail of the list.  In that case, we point the
		     * screen pointer at the entry following the target.
		     */
		    ftemp = fe->next_entry;

		free(fe);
		fe = ftemp;
		update_hp_screen(fl, firstvisible, win);
	    }
	    break;
	case 13:
	    if (fe != NULL) {
		sprintf(s_portstr, "%u", fe->hp.sport);
		sprintf(d_portstr, "%u", fe->hp.dport);
		inexstr[0] = toupper(fe->hp.reverse);
		inexstr[1] = '\0';
		gethostparams(&hptemp, &gh_aborted,
			      fe->hp.s_fqdn, fe->hp.s_mask, s_portstr,
			      fe->hp.d_fqdn, fe->hp.d_mask, d_portstr,
			      inexstr);

		if (!gh_aborted)
		    memcpy(&(fe->hp), &hptemp, sizeof(struct hostparams));

		update_hp_screen(fl, firstvisible, win);
	    }

	    break;
	case 'x':
	case 'X':
	case 'q':
	case 'Q':
	case 27:
	case 24:
	    endloop_local = 1;
	    break;
	case 'l':
	case 'L':
	    refresh_screen();
	    break;
	}
	update_panels();
	doupdate();
    } while (!endloop_local);

    del_panel(panel);
    delwin(win);
    del_panel(bpanel);
    delwin(bwin);
    update_panels();
    doupdate();
}

void editfilter(unsigned int protocol, int *aborted)
{
    char filename[FLT_FILENAME_MAX];
    struct filterlist fl;
    struct ffnode *flist;
    struct ffnode *ffile;
    struct filterfileent *ffe;

    if (!mark_filter_change(protocol))
	return;

    if (loadfilterlist(protocol, &flist) == 1) {
	listfileerr(1);
	destroyfilterlist(flist);
	clear_flt_tag(protocol);
	return;
    }
    pickafilter(flist, &ffile, aborted);

    clear_flt_tag(protocol);
    if ((*aborted)) {
	destroyfilterlist(flist);
	clear_flt_tag(protocol);
	return;
    }
    ffe = &(ffile->ffe);

    get_filter_description(ffe->desc, aborted, ffe->desc);

    if (*aborted) {
	destroyfilterlist(flist);
	clear_flt_tag(protocol);
	return;
    }
    strncpy(filename, get_path(T_WORKDIR, ffe->filename),
	    FLT_FILENAME_MAX - 1);

    if (loadfilter(filename, &fl, FLT_DONTRESOLVE))
	return;

    modify_host_parameters(protocol, &fl);

    save_filterlist(protocol, flist);	/* This also destroys it */
    savefilter(protocol, filename, &fl);
    destroyfilter(&fl);
}
