/*
*
* APS - Advanced Packet Sniffer
*
*  Copyright (C) 1999 Christian Schulte
*
*    This program is free software; you can redistribute it and/or modify
*    it under the terms of the GNU General Public License as published by
*    the Free Software Foundation; either version 2 of the License, or
*    (at your option) any later version.
*    This program is distributed in the hope that it will be useful,
*    but WITHOUT ANY WARRANTY; without even the implied warranty of
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*    GNU General Public License for more details.
*                                
*    You should have received a copy of the GNU General Public License
*    along with this program; if not, write to the Free Software
*    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*                                            
*
*/

#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/ipc.h>

// ------- local includes 

#include <linux_if_ether.h>
#include <protocols.h>
#include <config.h>
#include <aps_types.h>
#include <user_config.h>
#include <lansi.h>


#define DIST_COMMENT "\n\
\n\
 Copyright (C) 1999 Christian Schulte (dg1nsw@saturn2.franken.de)\n\
\n\
 This program is free software; you can redistribute it and/or modify\n\
 it under the terms of the GNU General Public License as published by\n\
 the Free Software Foundation.\n\
\n"

#define COPYLEFT "\n\
 This program is distributed in the hope that it will be useful,\n\
 but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\
 See the GNU General Public License for more details.\n\
 You should have received a copy of the GNU General Public License\n\
 along with this program; if not, write to the Free Software\n\
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\
 \n\
 For a recent version of this code...\n\
   Have a look at: \"http://www.swrtec.de\"\n\
   \n\
>As this is still under development, help, bug-reports and encouragement\n\
>to the mentioned addresses is allways welcome :->\n\n"

#define HELP "\n\
 aps [options][-p <from_port> <to_port>][-o <frame_type>]\n\
 options: -d display rest-output in \n\
              (1)HEX (2)ASCII (3)BOTH (4)NONE\n\
	      (5)PROTECTED-ASCII (6)MODE-5 with HEX\n\
	      (7)ONLY rest in HEX (8)only rest in PROTECTED-ASCII\n\
	      (9)ONLY rest in PROTECTED-ASCII with HEX\n\
	      (10)RAW-HEX-DUMP (without interpretation and filters!!)\n\
	  -c do NOT use colorization (ANSI-CODES)\n\
	  -b activate beep for some protocolls (see manpage)\n\
          -q do not say hello and bye nor print the copyleft\n\
          -Q do not say anything (using X-GUI instead ?)\n\
          -s do not display summary statistics info \n\
          -n do not print info of IP-Header\n\
          -i display only this IP-ADDR (192.168.17.4) \n\
          -I do not display this IP-ADDR (192.168.17.4) \n\
          -h display only this HW-ADDR (de:ad:be:ef:00:00) \n\
          -H do not display this HW-ADDR (de:ad:be:ef:00:00) \n\
          -p display only these ports (23-80) or (23-23) \n\
          -P do not display this port (23-80) or (23-23) \n\
          -o display only one frame type \n\
              (can be combined: (\"-o arp -o tcp-udp\")\n\
          -h print this HELP (or --help)\n\
          -v print Version and COPYLEFT\n\
           \n\
 valid frame-types are: \n\
    smb,loop,arp,rarp,all-ip,tcp-ip,udp-ip,icmp-ip,other \n\
\n\n"


int sockt;
unsigned int n_byte,nmbr;
unsigned char buf[10002]={128};  // 10Kbyte Buffer

unsigned int G_OPTS=0;        //Global options: Bit1=QUIET-MODE
unsigned int IPF_MASK=0xff;   //frame type filter for ICMP,UDP,TCP ... (bits are set if visible)
unsigned int FRAMEXMASK=0xff; //frame type filter for ARP,RARP,IP ...  (bits are set if visible)
unsigned int PRINT_MODE=0x01; //Only-HEX=0x01, Only-ASCII=0x02 and 0x03 is both
unsigned int MASK_MODE=0;     //For mask out of IP or HW addresses !
unsigned int MASK_PORT_F=0,MASK_PORT_T=0;      //For PORT-Filter

unsigned char COLORS=TRUE;    // use ansi-colors ??
unsigned char BEEPING=FALSE;  // we don't cry before anyone tells us so !
unsigned char PRINT_STATS=TRUE;
char PRINT_IP_HEADER=TRUE;  //print it or not ... this is the question
unsigned int NUM_COLUMNS=80;  // this is the default i think ?:->

unsigned char pnn[PORTLIST_MAX_PNUM+2][27];  // one more may be sure ! :->

unsigned char cfn[100]={0};       // capture file name
unsigned char HW[6]={0,0,0,0,0,0};        // for masking out 
unsigned long int HWA[2],IPA,hwa[2],ipa;
unsigned long int RX_PACKET_COUNT;

unsigned int shmid=-1;      // shared memory id and IPC-Key
key_t ipck;

struct in_addr
{
 unsigned long int s_addr;
}c_ipa;                     // for converting dot-ip-addresses to binary ips


s_aps_cntr *p_cnt;  // struct for counting the packets of each type ...



// *****************

void termh(int signum);
void print_head (unsigned char);  // 0 for only count up n_byte 1 for full functionality
void print_tail (void);
void open_socket (int *fd);
void clean_exit(int ret);
void eth_loop(void);
void smb(void);
int ip(void);
int arp(char tp);
int ipx();
int other(int X);
int hw_filter ();
void sz_if_FF (unsigned int *ZVAR);
int main (int argc, char *argv[]);



// *****************


void termh(int signum)
{
 unsigned int p_all_icmp=0,p_all_ip=0,other=0,arp_rarp=0; 

 close(sockt);

 kill_pif();

 if (G_OPTS&0x01) clean_exit(0);   // Just go away ... 
 if (COLORS) RED;
 printf("\n APS: terminating after %i packets. (%i)\n\n",RX_PACKET_COUNT,signum);
 if (COLORS) NORMAL;

 if (PRINT_STATS==TRUE)
 {
  p_all_icmp+=p_cnt->icmp_ip_echo_req;
  p_all_icmp+=p_cnt->icmp_ip_echo_rep;//  
  p_all_icmp+=p_cnt->icmp_ip_unre_net;
  p_all_icmp+=p_cnt->icmp_ip_unre_host;
  p_all_icmp+=p_cnt->icmp_ip_unre_proto;
  p_all_icmp+=p_cnt->icmp_ip_unre_port;
  p_all_icmp+=p_cnt->icmp_ip_unre_frag_need;
  p_all_icmp+=p_cnt->icmp_ip_unre_src_route;
  p_all_icmp+=p_cnt->icmp_ip_unre_unknown;//  
  p_all_icmp+=p_cnt->icmp_ip_source_quench;//  
  p_all_icmp+=p_cnt->icmp_ip_redir_tonet;
  p_all_icmp+=p_cnt->icmp_ip_redir_tohost;
  p_all_icmp+=p_cnt->icmp_ip_redir_tos_net;
  p_all_icmp+=p_cnt->icmp_ip_redir_tos_host;
  p_all_icmp+=p_cnt->icmp_ip_redir_unknown;//
  p_all_icmp+=p_cnt->icmp_ip_exc_time_limit;
  p_all_icmp+=p_cnt->icmp_ip_dtgr_param_prob;
  p_all_icmp+=p_cnt->icmp_ip_time_req;
  p_all_icmp+=p_cnt->icmp_ip_time_rep; 
  p_all_icmp+=p_cnt->icmp_ip_info_req; 
  p_all_icmp+=p_cnt->icmp_ip_info_rep; 
  p_all_icmp+=p_cnt->icmp_ip_addr_mask_req; 
  p_all_icmp+=p_cnt->icmp_ip_addr_mask_rep; 
  p_all_icmp+=p_cnt->icmp_ip_unknown;// 
  p_all_ip+=p_cnt->udp_ip;
  p_all_ip+=p_cnt->tcp_ip;
  p_all_ip+=p_all_icmp;//  
  arp_rarp+=p_cnt->arp_req;
  arp_rarp+=p_cnt->arp_rep;
  arp_rarp+=p_cnt->rarp_req;
  arp_rarp+=p_cnt->rarp_rep;//  
  other+=p_cnt->smb;
  other+=p_cnt->loop;
  other+=p_cnt->unknown;
  if (COLORS) {REV;BOLD;}
  printf ("Package summary: \n");
  if (COLORS) {NORMAL;BLUE;}
  printf ("--> IP  :%.6d  ## TCP-IP:%.6d UDP-IP:%.6d ICMP-IP:%.6d\n\n",p_all_ip,p_cnt->tcp_ip,p_cnt->udp_ip,p_all_icmp);
  if (COLORS) {NORMAL;BOLD;}
  printf ("--> RSLV:%.6d  ## ARP-REQ:%.6d RARP-REQ:%.6d\n",arp_rarp,p_cnt->arp_req,p_cnt->rarp_req);
  printf ("                 ## ARP-REP:%.6d RARP-REP:%.6d\n\n",p_cnt->arp_rep,p_cnt->rarp_rep);
  if (COLORS) {NORMAL;GREEN;}
  printf ("--> ICMP:%.6d  ## ECHO-REQ:%.6d TIME-REQ:%.6d INFO-REQ:%.6d\n",p_all_icmp,p_cnt->icmp_ip_echo_req,p_cnt->icmp_ip_time_req,p_cnt->icmp_ip_info_req);
  printf ("                 ## ECHO-REP:%.6d TIME-REP:%.6d INFO-REP:%.6d\n",p_cnt->icmp_ip_echo_rep,p_cnt->icmp_ip_time_rep,p_cnt->icmp_ip_info_rep);
  printf ("                 ## ADDR-MASK-REQ:%.6d ADDR-MASK-REP:%.6d\n",p_cnt->icmp_ip_addr_mask_req,p_cnt->icmp_ip_addr_mask_rep);
  if (COLORS) {NORMAL; RED;}
  printf ("      unreachable # HOST:%.6d NETW:%.6d PROT:%.6d PORT:%.6d\n",
  		p_cnt->icmp_ip_unre_host,p_cnt->icmp_ip_unre_net,p_cnt->icmp_ip_unre_proto,p_cnt->icmp_ip_unre_port);
  printf ("      unreachable # FRAGM-NEEDED:%.6d SRC-ROUTE:%.6d UNKNOWN:%.6d\n",
  		p_cnt->icmp_ip_unre_frag_need,p_cnt->icmp_ip_unre_src_route,p_cnt->icmp_ip_unre_unknown);
  if (COLORS) {NORMAL;MAGENTA;}
  printf ("         redirect # TO-NET :%.6d TO-HOST :%.6d\n",p_cnt->icmp_ip_redir_tonet,p_cnt->icmp_ip_redir_tohost);
  printf ("         redirect # TOS-NET:%.6d TOS-HOST:%.6d REDIR-UNKNOWN:%.6d\n",p_cnt->icmp_ip_redir_tos_net,p_cnt->icmp_ip_redir_tos_host,p_cnt->icmp_ip_redir_unknown);
  printf ("            other # EXCEEDED-TIME-LIMIT:%.6d PARAMETER-PROBLEM:%.6d\n",
  		p_cnt->icmp_ip_dtgr_param_prob,p_cnt->icmp_ip_dtgr_param_prob);
  printf ("                  # OVERALL-ICMP-UNKNOWN:%.6d\n\n",p_cnt->icmp_ip_unknown);
  if (COLORS) {NORMAL;CYAN;}
  printf ("--> OTHER:%.6d ## LOOP:%.6d SMB:%.6d UNKNOWN-FRAMES:%.6d\n\n",other,p_cnt->loop,p_cnt->smb,p_cnt->unknown);
  if (COLORS) NORMAL;
 }
 
 clean_exit(0);
}


void init (void)   // do some initialization
{
 char cbufs[200],cbufn[200],ipc_str[30];
 register unsigned int cnt,dmy;
 char *ncp,un[10]="unknown",fname[150];
 uid_t euid; 
 int pnum,n;
 FILE *F_p;

 euid=getuid();
 if (euid!=0) { if (COLORS) RED; printf ("YOU ARE NOT ROOT !!! \n"); if (COLORS) NORMAL; exit(2); }

 signal(SIGKILL,termh);   // kill -9
 signal(SIGQUIT,termh);   // signal 3
 signal(SIGINT,termh);    // control+C
 signal(SIGHUP,termh);    // hangup

 CLEANS;  // better make it clean here than later and wipe away some messages ...

 un[8]=0;
 for (cnt=0;cnt<PORTLIST_MAX_PNUM;cnt++)
 {
  memcpy(pnn[cnt],un,8);  
 }
 
 strcpy(fname,INCLUDE_INST_DIR);
 strcpy(fname+strlen(INCLUDE_INST_DIR),"/portlist.aps");
 if ( (F_p=fopen(fname,"r")) == NULL) printf ("Can not read %s ... no port descriptions available \n",fname);
 else
 {
  while (dmy!=EOF)
  {
   cnt=0;dmy=0;
   pnum=0;

   while( dmy!=32 )
   {
    if ( (dmy=fgetc(F_p)) == EOF) break;
    if (dmy==0x0a) goto next_entry;
    if (dmy!='#') { cbufn[cnt]=dmy; cnt++; }
    else
    {
     while ( (dmy!=0x0a) && (dmy!=EOF) ) dmy=fgetc(F_p);
     goto next_entry;
    } 
   }

   cbufn[cnt-1]=0;

   cnt=0;dmy=0;
   while( dmy!=0x0a )
   {
    if ( (dmy=fgetc(F_p)) == EOF) break;
    
    if (dmy!='#')
    {
     cbufs[cnt]=dmy;
     cnt++;
     if (cnt>=25) { if (COLORS) RED; printf ("Format error !!  Entry %s is too long !!\n",cbufn); if (COLORS) NORMAL; clean_exit(1);}
    }
    else
     while ( (dmy!=0x0a) && (dmy!=EOF) ) dmy=fgetc(F_p);
   }
   if (cnt!=0) cbufs[cnt-1]=0;
 
   sscanf(cbufn,"%d",&pnum);
   if (pnum<PORTLIST_MAX_PNUM) sscanf(cbufs,"%s",&pnn[pnum]); else printf ("%s at %s is higher than allowed %d ! \n",cbufn,cbufs,PORTLIST_MAX_PNUM);
   
next_entry:  // not nice but fast i think ...
  } 
 }
 
 fclose(F_p);

 n=get_last_pif();  // get a sequence number for our process
 printf ("Allocating SHMEM-SEGMENT for APS_%.4d \n",n);
 sprintf (ipc_str,"%s/portlist.aps",INCLUDE_INST_DIR);
 ipck=ftok(ipc_str,n);
 if( (shmid = shmget( ipck, STRCTSZE_CNTR, IPC_CREAT | 0600 )) != -1)  // get a shared memory segment !!
 {
  if ( (int) (p_cnt=(s_aps_cntr*) shmat(shmid,0,0)) == -1 )   // attach the allocated segment 
  { perror ("shmat:"); exit (1); }
  memset(p_cnt,0,STRCTSZE_CNTR);   // set it all to 0
  p_cnt->icmp_ip_echo_req=0;

  if ( (create_new_pif(shmid)) == -1 ) { printf ("WARNING: Could not create %s ... no X-GUI available for this process !\n",fname); }
   // tell them where whe are (shmid and pid)
 }
 else
 {
  printf ("Could not allocate SHMEM-SEGMENT ...  no X-GUI available for this process \n");
  if ( (int) (p_cnt=malloc(STRCTSZE_CNTR)) == -1)  // allocate memory of size STRCTSZE_CNT=size of struct p_cnt 
  { perror ("malloc:"); exit (1); }
  memset(p_cnt,0,STRCTSZE_CNTR);   // set it all to 0
 }
   
 
 printf ("\n");
}




void print_head (unsigned char func)
{
 if (func>0)
 {
   printf("HW-ADDR: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x -----> %.2x:%.2x:%.2x:%.2x:%.2x:%.2x \n",
   buf[6],buf[7],buf[8],buf[9],buf[10],buf[11],
   buf[0],buf[1],buf[2],buf[3],buf[4],buf[5]);
 }
 
 n_byte+=14;  // The Mac-Header (+12) and the protocol ID (+2) ...
}


void print_tail (void)
{
unsigned int br,bs;

  if (G_OPTS&0x02) return;

  if (!(PRINT_MODE&0x10)) printf("Bytes until here: 0x%0.4x  Bytes over all: 0x%0.4x \n",n_byte,nmbr);
  if (PRINT_MODE&0x08) { printf("\n"); return; }
  if (PRINT_MODE&0x01) // HEX
  {
   for (br=n_byte;br<nmbr;br++)
   {
    printf ("\n");
    for (bs=br;(bs<nmbr)&&((bs-br)<NUM_COLUMNS/3);bs++)
    printf(" %.2x",buf[bs]);
    bs--;
    br=bs;
   }  
   if (COLORS) NORMAL;
   printf ("\n\n");return;
  }
  
  if (PRINT_MODE&0x04) printf("\n");
  if (PRINT_MODE&0x02) // ASCII
  {
   for (br=n_byte;br<nmbr;br++)
   {
    printf ("\n");
    for (bs=br;(bs<nmbr)&&((bs-br)<NUM_COLUMNS);bs++)
    printf("%c",buf[bs]);
    bs--;
    br=bs;
   }
   if (COLORS) NORMAL;
   printf ("\n\n");return;
  }

  if (PRINT_MODE&0x20) 
  for (br=n_byte;br<nmbr;br++)
  {
   printf ("\n");
   for (bs=br;(bs<nmbr)&&((bs-br)<NUM_COLUMNS);bs++)
    if (buf[bs]>31) printf("%c",buf[bs]); else printf (" ");
   bs--;
   br=bs;
  }  

  if (PRINT_MODE&0x40)
  for (br=n_byte;br<nmbr;br++)
  {
   printf ("\n");
   for (bs=br;(bs<nmbr)&&((bs-br)<NUM_COLUMNS);bs++)
    if ( (buf[bs]>31) && (buf[bs]<150) ) printf("%c",buf[bs]); 
    else if (buf[bs]>160) printf("%c",buf[bs]); else printf (" %.2x",buf[bs]);
   bs--;
   br=bs;
  }  

  if (COLORS) NORMAL;
  
  printf("\n\n");   // The next one please ?
}


void open_socket (int *fd)
{
  //  *fd=socket(PF_INET,SOCK_PACKET,htons(ETH_P_ALL));
  
  /* I include here the call PF_PACKET and SOCK_RAW in my Linux SuSE 6.3
  kernel 2.22. The SOCK_PACKET call is deprecated, and widely 
  non-portable, see 'man packet'...
  But we need the Libpcap support! 
  */
 
  *fd=socket(PF_PACKET,SOCK_RAW,htons(ETH_P_ALL)); 
  if (*fd<0) { perror ("opening socket"); abort(); }
}



void clean_exit(int ret)
{  // should we do something more ??
 if (COLORS) NORMAL;
 shmctl(shmid,IPC_RMID,0);  // mark segment for deletion ...
 shmdt(p_cnt);		    // detach the SHMEM-segment
 if (!(G_OPTS&0x01)) printf (" Detached SHMEM-Segment (ID=%d)\n",shmid);
 kill_pif();
 close (sockt);
 exit(ret);
}


void eth_loop(void)
{
 if ( (!(PRINT_MODE&0x80)) && (!(G_OPTS&0x02)) )
 {
  print_head(1);
  printf("Loopback\n");
 }
  
 print_tail;
 
 p_cnt->loop++;  // for the statistics
}


void smb(void)
{
 if ( (!(PRINT_MODE&0x80)) && (!(G_OPTS&0x02)) )
 {
  print_head(1);
  printf("SAMBA/NetBios\n");
 }
 
 print_tail();
 
 p_cnt->smb++;  // for the statistics
}




int ip(void)
{
 char *proto_str;
 char proto_s[50];
 int TOS=0,sp,dp;    // Type Of Service, source-port, dest-port

 proto_str=(char*) &proto_s;

 if (MASK_MODE&0x04)  // "i" only
 {
  ipa=(buf[26])+(buf[27]<<8)+(buf[28]<<16)+(buf[29]<<24);  // As a number
  if (ipa!=IPA)
  {
   ipa=(buf[30])+(buf[31]<<8)+(buf[32]<<16)+(buf[33]<<24);  // As a number
   if (ipa!=IPA) return -1;
  }
 }
 if (MASK_MODE&0x08)  // "i" not
 {     // "I" not
  ipa=(buf[26])+(buf[27]<<8)+(buf[28]<<16)+(buf[29]<<24);  // As a number
  if (ipa==IPA) return -1;
  ipa=(buf[30])+(buf[31]<<8)+(buf[32]<<16)+(buf[33]<<24);  // As a number
  if (ipa==IPA) return -1;
 }  

 TOS=buf[n_byte+23];  // n_byte should be 14 but then the header would have been already printed!
    // So at this point n_byte is 0 because we cant decide if we should print info or not
    // before we dont know anything about the packet !! :->
 if (PRINT_MODE&0x80) { print_head(0); goto cut_ip_header; }  // print_head(0)   for the MAC-Header

 if (MASK_MODE&0x10)  // print port only
 {
  if ( (TOS==IP_TCP) || (TOS==IP_UDP) )
  { 
   sp=(buf[n_byte+34]<<8)+buf[n_byte+35];
   dp=(buf[n_byte+36]<<8)+buf[n_byte+37];
   if ( ((sp<MASK_PORT_F)||(sp>MASK_PORT_T)) && ((dp<MASK_PORT_F)||(dp>MASK_PORT_T)) ) return -1;  // if both are not in scope ... throw away
  }
 }

 if (MASK_MODE&0x20)  // print port not 
 {
  if ( (TOS==IP_TCP) || (TOS==IP_UDP) )
  {
   sp=(buf[n_byte+34]<<8)+buf[n_byte+35];
   if ((sp>=MASK_PORT_F)&&(sp<=MASK_PORT_T)) return -1;		   // if the first is in scope  ... throw away
   dp=(buf[n_byte+36]<<8)+buf[n_byte+37];
   if ((dp>=MASK_PORT_F)&&(dp<=MASK_PORT_T)) return -1;		   // if the second is in scope  ... throw away
  }
 }

 
 switch(TOS)                       // Yah i know this is real dumb programming :->
 {
  case(IP_ICMP): if (!(IPF_MASK&0x02)) return(-1); proto_str="ICMP";break;
  case(IP_TCP): if (!(IPF_MASK&0x01)) return(-1); proto_str="TCP";break;
  case(IP_UDP): if (!(IPF_MASK&0x04)) return(-1); proto_str="UDP";break;
  case(IP_IGMP): proto_str="IGMP";break;
  case(IP_GGP): proto_str="GGP";break;
  case(IP_ST): proto_str="ST";break;
  case(IP_UCL): proto_str="UCL";break;
  case(IP_EGP): proto_str="EGP";break;
  case(IP_IGP): proto_str="IGP";break;
  case(IP_BBN_RCC_MON): proto_str="BBN_RCC_MON";break;
  case(IP_NVP_II): proto_str="NVP_II";break;
  case(IP_PUP): proto_str="PUP";break;
  case(IP_ARGUS): proto_str="ARGUS";break;
  case(IP_EMCON): proto_str="EMCON";break;
  case(IP_XNET): proto_str="XNET";break;
  case(IP_CHAOS): proto_str="CHAOS";break;
  case(IP_MUX): proto_str="MUX";break;
  case(IP_DCN_MEAS): proto_str="DCN_MEAS";break;
  case(IP_HMP): proto_str="HMP";break;
  case(IP_PRM): proto_str="PRM";break;
  case(IP_XNS_IDP): proto_str="XNS_IDP";break;
  case(IP_TRUNK1): proto_str="TRUNK1";break;
  case(IP_TRUNK2): proto_str="TRUNK2";break;
  case(IP_LEAF1): proto_str="LEAF1";break;
  case(IP_LEAF2): proto_str="LEAF2";break;
  case(IP_RDP): proto_str="RDP";break;
  case(IP_IRTP): proto_str="IRTP";break;
  case(IP_ISO_TP4): proto_str="ISO_TP4";break;
  case(IP_NETBLT): proto_str="NETBLT";break;
  case(IP_MFE_NSP): proto_str="MFE_NSP";break;
  case(IP_MERIT_INP): proto_str="INP";break;
  case(IP_SEP): proto_str="SEP";break;
  case(IP_3PC): proto_str="3PC";break;
  case(IP_CFTP): proto_str="CFTP";break;
  case(SAT_EXPAK): proto_str="EXPAK";break;
  case(IP_RVD): proto_str="RVD";break;
  case(IP_IPPC): proto_str="IPPC";break;
  case(IP_SAT_MON): proto_str="SAT_MON";break;
  case(IP_VISA): proto_str="VISA";break;
  case(IP_IPCV): proto_str="IPCV";break;
  case(IP_BR_SAT_MON): proto_str="BR_SAT_MON";break;
  case(IP_SUN_ND): proto_str="SUN_ND";break;
  case(IP_WB_MON): proto_str="WB_MON";break;
  case(IP_WB_EXPAK): proto_str="WB_EXPAK";break;
  case(IP_ISO_IP): proto_str="ISO_IP";break;
  case(IP_VMTP): proto_str="VMTP";break;
  case(IP_SECURE_VMTP): proto_str="SECURE_VMTP";break;
  case(IP_VINES): proto_str="VINES";break;
  case(IP_TTP): proto_str="TTP";break;
  case(NSFNET_IGP): proto_str="IGP";break;
  case(IP_DGP): proto_str="DGP";break;
  case(IP_TCF): proto_str="TCF";break;
  case(IP_IGRP): proto_str="IGRP";break;
  case(IP_OSPFIGP): proto_str="OSPFIGP";break;
  case(IP_SPRITE_RPG): proto_str="SPRITE_RPG";break;
  case(IP_LARP): proto_str="LARP";break;
  default:proto_str="unknown";break;
 }

 if (PRINT_IP_HEADER==FALSE) { print_head(0); goto cut_ip_header; } // I think this is the fastest way

 print_head(1);
 
 printf("IP-ADDR: %d.%d.%d.%d  -----> %d.%d.%d.%d\n",
 buf[n_byte+12],buf[n_byte+13],buf[n_byte+14],buf[n_byte+15],
 buf[n_byte+16],buf[n_byte+17],buf[n_byte+18],buf[n_byte+19]);
 
 printf("IP-Ver%d  ||  Head:0x%0.2x (bytes)  ||  Service(TOS):%d  ||  ",
 buf[n_byte]>>4,(buf[n_byte]&0xf)<<2,buf[n_byte+1]);

 printf("Length over all:%0.4d\n",((buf[n_byte+2]<<8)+buf[n_byte+3]));
 printf("Fragmentation:  ID:0x%0.4x - ",((buf[n_byte+4]<<8)+buf[n_byte+5]));
 printf("Flags: %d %d %d - ",(buf[n_byte+6]&0x80)>>7,(buf[n_byte+6]&0x40)>>6,(buf[n_byte+6]&0x20)>>5);
 printf("Offset:%0.5d\n",((buf[n_byte+6]&0x1f)<<8)+buf[n_byte+7]);
 printf("TTL:%0.3d  ||  Protokoll:%0.3d (%s)   ||  HeaderCRC:0x%0.4x\n",
 buf[n_byte+8],buf[n_byte+9],proto_str,(buf[n_byte+10]<<8)+buf[n_byte+11]);

cut_ip_header:

 n_byte+=20;  // Here we are ...

 if (TOS==IP_ICMP) icmp_info();  // print out ICMP-INFO
 if (TOS==IP_UDP) if(PRINT_MODE&0x80) n_byte+=8; else udp_info(); 
 if (TOS==IP_TCP) if(PRINT_MODE&0x80) n_byte+=20; else tcp_info();   // get the TCP-Header splitted of ...

 print_tail();

 return(1);
}


int arp(char tp)
{

 if (MASK_MODE&0x04)  // "i" only
 {
  ipa=(buf[28])+(buf[29]<<8)+(buf[30]<<16)+(buf[31]<<24);  // As a number
  if (ipa!=IPA)
  {
   ipa=(buf[31])+(buf[32]<<8)+(buf[33]<<16)+(buf[34]<<24);  // As a number
   if (ipa!=IPA) return -1;
  }
 }
 if (MASK_MODE&0x08)  // "I" not
 {
  ipa=(buf[28])+(buf[29]<<8)+(buf[30]<<16)+(buf[31]<<24);  // As a number
  if (ipa==IPA) return -1;
  ipa=(buf[31])+(buf[32]<<8)+(buf[33]<<16)+(buf[34]<<24);  // As a number
  if (ipa==IPA) return -1;
 }


 if ( (!(PRINT_MODE&0x80)) && (!(G_OPTS&0x02)) ) print_head(1);
 
 if (tp==0)
 { 
   if (COLORS) {YELLOW;BOLD;}
   if (!(G_OPTS&0x02)) printf("ARP: IP->Ethernet \n");
   if ((buf[20]==0)&(buf[21]==1)) { p_cnt->arp_req++; if ( (!(PRINT_MODE&0x80)) && (!(G_OPTS&0x02)) ) printf("Request"); }
   if ((buf[20]==0)&(buf[21]==2)) { p_cnt->arp_rep++; if ( (!(PRINT_MODE&0x80)) && (!(G_OPTS&0x02)) ) printf("Answer "); }
   if (COLORS) NORMAL;
 }
 else
 {
   if ( (COLORS) && (!(G_OPTS&0x02)) ) {YELLOW;BOLD;}
   if (!(G_OPTS&0x02)) printf("RARP: Ethernet->IP \n");
   if ((buf[20]==0)&(buf[21]==1)) { p_cnt->rarp_req++; if ( (!(PRINT_MODE&0x80)) && (!(G_OPTS&0x02)) ) printf("Request"); }
   if ((buf[20]==0)&(buf[21]==2)) { p_cnt->rarp_rep++; if ( (!(PRINT_MODE&0x80)) && (!(G_OPTS&0x02)) ) printf("Answer "); }
   if (COLORS) NORMAL;
 }

 if ( (!(PRINT_MODE&0x80)) && (!(G_OPTS&0x02)) )
 {
   if (COLORS) {YELLOW;BOLD;}
   printf(       " from %-3d.%-3d.%-3d.%-3d (HW:%0.2x:%0.2x:%0.2x:%0.2x:%0.2x:%0.2x) \n",
   buf[28],buf[29],buf[30],buf[31],buf[22],buf[23],buf[24],buf[25],buf[26],buf[27]);
   printf("Goes    to   %-3d.%-3d.%-3d.%-3d (HW:%0.2x:%0.2x:%0.2x:%0.2x:%0.2x:%0.2x) \n",
   buf[38],buf[39],buf[40],buf[41],buf[32],buf[33],buf[34],buf[35],buf[36],buf[37]);
   if (COLORS) NORMAL;
   n_byte+=28;  // 28 weitere Bytes ...
   // bis hier wurde das Packet ausgewertet .. der Rest wird hex-gedumpt B->> 
 }
  
 print_tail();

 return(1);
}


int ipx()
{
 if (!(PRINT_MODE&0x80))
 {
  print_head(1);
  printf("IPX-Packet? -- sorry no further information ... :-<\n");
 }
 
 print_tail();
 
 p_cnt->ipx++;
 
 return(1);
}




int other(int X)
{
 if ( (!(PRINT_MODE&0x80)) && (!(G_OPTS&0x02)) )
 {
  print_head(1);
  if (COLORS) RED;
  printf("Unknown Frame type 0x%.4x :-<\n",X);
  if (COLORS) NORMAL;
 }
 
 print_tail();
 
 p_cnt->unknown++;
 
 return(1);
}


int hw_filter ()
{
  if (MASK_MODE&0x01) // "h" only
  {
   hwa[0]=(buf[6])+(buf[7]<<8)+(buf[8]<<16)+(buf[9]<<24);
   hwa[1]=(buf[10])+(buf[11]<<8);
   if (hwa[0]!=HWA[0]||hwa[1]!=HWA[1])
   {
    hwa[0]=(buf[0])+(buf[1]<<8)+(buf[2]<<16)+(buf[3]<<24);
    hwa[1]=(buf[4])+(buf[5]<8);
    if (hwa[0]!=HWA[0]||hwa[1]!=HWA[1])  return (1);  // invalid
   }
  }
  if (MASK_MODE&0x02) // "h" not
  {
   hwa[0]=(buf[6])+(buf[7]<<8)+(buf[8]<<16)+(buf[9]<<24);
   hwa[1]=(buf[10])+(buf[11]<<8);
   if ((hwa[0]==HWA[0])&&(hwa[1]==HWA[1])) return (1);  // invalid
   hwa[0]=(buf[0])+(buf[1]<<8)+(buf[2]<<16)+(buf[3]<<24);
   hwa[1]=(buf[4])+(buf[5]<<8);
   if ((hwa[0]==HWA[0])&&(hwa[1]==HWA[1])) return (1);  // invalid
  }  // They cant be set both !!!
  return(0);
}  



void sz_if_FF (unsigned int *ZVAR)
{
 if (*ZVAR==0xff) *ZVAR=0;
 printf (" ---> %x \n",*ZVAR);
}




// *********************** HERE COMES THE MAIN  ***********************






int main (int argc, char *argv[])
{
unsigned int X;
int optc;
char arg_str[20],c;
extern char *optarg;


 init();


 if (argc!=1)
 {
  printf ("\n\n");
  while ((optc=getopt(argc,argv,"f:o:i:I:h:H:p:P:d:vnsqQbc"))!=EOF)
  switch (optc)
  {
   case'b':BEEPING=TRUE;break;		// okay he wants us to shout out :->
   case'c':COLORS=FALSE;break;          // okay so we do not print ansi-colors 
   case'q':G_OPTS|=0x01; break;		  // set QUIET-MODE (just the data ... no greetings)
   case'Q':G_OPTS|=0x03;                  // set both bits (be VERY QUIET !!)
   	   PRINT_IP_HEADER=FALSE;
	   PRINT_MODE=0x08;
   	   printf ("  Running in VERY-QUIET-MODE !\n");
	   break;
   case'f':strcpy(cfn,optarg);break;   // read in the filename for the capture FILE !!
   case'p':
   case'P':if(sscanf(optarg,"%d-%d",&MASK_PORT_F,&MASK_PORT_T) == 1) MASK_PORT_T=MASK_PORT_F;
   	   if (MASK_PORT_F>MASK_PORT_T) { X=MASK_PORT_F;  MASK_PORT_F=MASK_PORT_T;  MASK_PORT_T=X; }  // order the vars 
	   if (FRAMEXMASK!=0xff)
	    printf ("  WARNING: allready filtering a protocoll\n           (mayby there are no ports left to filter ie. for ICMP-IP)\n\n");
   	   if (optc=='p') { MASK_MODE|=0x10; FRAMEXMASK|=0x04; printf("  Only printing PORT %d - %d\n\n",MASK_PORT_F,MASK_PORT_T); } // display PORT only !
   	   if (optc=='P') { MASK_MODE|=0x20; FRAMEXMASK|=0x04; printf("  Not printing PORT %d - %d\n\n",MASK_PORT_F,MASK_PORT_T); } // display PORT not !
   	   break;
   case'h':
   case'H':arg_str[2]=0;
           for (X=0;X<6;X++)
           {
            if (X<5)
            if (optarg[2+X*3]!=*":") { printf (" Wrong number format !! \n %s",HELP); abort(); }
            arg_str[0]=optarg[0+X*3];
            arg_str[1]=optarg[1+X*3];
            sscanf(arg_str,"%x",&HW[X]);
           }
           HWA[0]=(HW[0])+(HW[1]<<8)+(HW[2]<<16)+(HW[3]<<24);  // As a number
           HWA[1]=(HW[4])+(HW[5]<<8);
           if (optc=='h') { MASK_MODE|=0x01; printf("  Only printing HWA %s\n\n",optarg); } // display HWA only !
           if (optc=='H') { MASK_MODE|=0x02; printf("  Not printing HWA %s\n\n",optarg); } // display HWA not  !
           break;
   case'i':
   case'I':if (optc=='i') 
      		{
                 MASK_MODE|=0x04; 
                 printf ("  Only printing %s \n\n",optarg);
                }  // display IP only !                   
           if (optc=='I')
              	{
                 MASK_MODE|=0x08;
                 printf ("  Not printing %s \n\n",optarg);
                }  // display IP not  !
           inet_aton(optarg,&c_ipa);
           IPA=c_ipa.s_addr;
           break;
   case'd':arg_str[0]=optarg[0];
           arg_str[1]=0;
      	   PRINT_MODE=atoi(optarg);
           switch(PRINT_MODE)
           {
            case(1):printf("  Printing rest-output in HEX(default) \n\n");PRINT_MODE=0x01;break;
            case(2):printf("  Printing rest-output in ASCII \n\n");PRINT_MODE=0x02;break;
            case(3):printf("  Printing rest-output in HEX and ASCII \n\n");PRINT_MODE=0x04;break;
	    
		    case(4):printf("  Printing no rest-output \n\n");PRINT_MODE=0x08;break;
	    
	        case(5):printf("  Printing rest-output in FILTERED ASCII \n\n");PRINT_MODE=0x20;break;
	        case(6):printf("  Printing rest-output in FILTERED ASCII with HEX \n\n");PRINT_MODE=0x40;break;
	    
	        case(7):printf("  Printing ONLY rest-output in HEX \n\n");PRINT_MODE=0x81;break;
	        case(8):printf("  Printing ONLY rest-output in FILTERED ASCII \n\n");PRINT_MODE=0xa0;break;
	        case(9):printf("  Printing ONLY rest-output in FILTERED ASCII with HEX \n\n");PRINT_MODE=0xc0;break;
	    
	        case(10):printf("  Printing raw dump in HEX\n\n");PRINT_MODE=0x11;break;
            default:printf("  Invalid Number for output format! \n\n%s",HELP); 
                    clean_exit(1);
           }
           break;
   case'o':
      X=0;
      if (FRAMEXMASK==0xff) FRAMEXMASK=0;
      strcpy(arg_str,optarg);
      if(strncmp(arg_str,"smb",3)==0)    { X=1; FRAMEXMASK|=0x01; printf ("  Only printing smb frames ... \n\n"); }
      if(strncmp(arg_str,"loop",4)==0)   { X=1; FRAMEXMASK|=0x02; printf ("  Only printing loop frames ... \n\n"); }
      if(strncmp(arg_str,"arp",3)==0)    { X=1; FRAMEXMASK|=0x08; printf ("  Only printing arp frames ... \n\n"); }
      if(strncmp(arg_str,"rarp",4)==0)   { X=1; FRAMEXMASK|=0x10; printf ("  Only printing rarp frames ... \n\n"); }
      if(strncmp(arg_str,"ipx",3)==0)    { X=1; FRAMEXMASK|=0x20; printf ("  Only printing ipx frames ... \n\n"); }
      if(strncmp(arg_str,"other",5)==0)  { X=1; FRAMEXMASK|=0x40; printf ("  Only printing unknown frames ...\n\n");}
      if(strncmp(arg_str,"all-ip",6)==0) { X=1; sz_if_FF(&IPF_MASK); IPF_MASK|=0x07; FRAMEXMASK|=0x04; printf ("  Only printing IP frames ...\n\n");}
      if(strncmp(arg_str,"tcp-ip",6)==0) { X=1; sz_if_FF(&IPF_MASK); IPF_MASK|=0x01; FRAMEXMASK|=0x04; printf ("  Only printing TCP-IP frames ...\n\n");}
      if(strncmp(arg_str,"icmp-ip",7)==0){ X=1; sz_if_FF(&IPF_MASK); IPF_MASK|=0x02; FRAMEXMASK|=0x04; printf ("  Only printing ICMP-IP frames ...\n\n");}
      if(strncmp(arg_str,"udp-ip",6)==0) { X=1; sz_if_FF(&IPF_MASK); IPF_MASK|=0x04; FRAMEXMASK|=0x04; printf ("  Only printing UDP-IP frames ...\n\n");}
      if(X==0) { printf("%s\n%s: %s\n",HELP,"Unknown frame-type",arg_str); abort(); }
      break;
   case'v': BOLD;printf ("%s",COPYLEFT); if (COLORS) NORMAL; clean_exit(0); break;
   case'n': PRINT_IP_HEADER=FALSE;
      	    printf ("  Ok cutting of IP Header information... \n\n");
            break;
   case's': PRINT_STATS=FALSE;break;
   case'?': printf ("%s\n%s",COPYLEFT,HELP);clean_exit(0);break;
   default: printf ("%s\n%s: %c\n",HELP,"Unknown argument character",optc); abort(); break;
  }
 }

 if (!(G_OPTS&0x01))  // If not quiet say hello to everybody
 {
  if (COLORS) {BGBLUE;}
  printf("\n\nAdvancedPacketSniffer V%s       (CONFIG-DATE:%s)",VERSION,BUILD_DATE);  // this is the VERSION from autoconf in "config.h"
  printf("%s",DIST_COMMENT); 
  if (COLORS) NORMAL;
 }

 
 open_socket(&sockt);

 if (PRINT_MODE&0x10)
  for(;;)
  {
   nmbr=recv(sockt,buf,10000,0); 
   RX_PACKET_COUNT++;
   n_byte=0;
   print_tail();
  }

 for(;;)
 {
forget:  // forget this package
  nmbr=recv(sockt,buf,10000,0); 
  RX_PACKET_COUNT++;
  
  if (MASK_MODE&0x03) { if (hw_filter()!=0) goto forget; }

  X=buf[12]*256+buf[13];  // this is really messy at this point !! who created the hardware layer ?? eh!?
  n_byte=0;   
  switch(X)
  {
   case(0x0025):if (FRAMEXMASK&0x01) smb();break;
   case(0x0060):if (FRAMEXMASK&0x02) eth_loop();break;
   case(0x0800):if (FRAMEXMASK&0x04) { if (ip()==-1) goto forget; }break;
   case(0x0806):if (FRAMEXMASK&0x08) { if (arp(0)==-1) goto forget; }break;
   case(0x8035):if (FRAMEXMASK&0x10) { if (arp(1)==-1) goto forget; }break;
   case(0x8137):if (FRAMEXMASK&0x20) ipx();break;
   default:if (FRAMEXMASK&0x40) other(X); break;
  }
  
 }
 
}



