/*
** Copyright 1999 by Todd Allen.  All Rights Reserved.  Permission to use,
** copy, modify, distribute, and sell this software and its documentation for
** any purpose is hereby granted without fee, provided that the above
** copyright notice appear in all copies and that both the copyright notice
** and this permission notice appear in supporting documentation.
**
** No representations are made about the suitability of this software for any
** purpose.  It is provided ``as is'' without express or implied warranty,
** including but not limited to the warranties of merchantability, fitness
** for a particular purpose, and noninfringement.  In no event shall Todd
** Allen be liable for any claim, damages, or other liability, whether in
** action of contract, tort, or otherwise, arising from, out of, or in
** connection with this software.
*/

#include "diags.h"
#include "x_util.h"

extern
Pixel
XUGetColor(Display*  display,
           String    name)
{
   int                screen;
   Window             root;
   XWindowAttributes  attrs;
   XColor             color;
   Status             status;

   screen = DefaultScreen(display);
   root = RootWindow(display, screen);
   if (root == (Window)None) {
      fatal(static_format("screen %d is not valid ", screen));
   }

   XGetWindowAttributes(display, root, &attrs);

   status = XParseColor(display, attrs.colormap, name, &color);
   if (!status) {
      fatal(static_format("undefined color %s", name));
   }
   status = XAllocColor(display, attrs.colormap, &color);
   if (!status) {
      fatal(static_format("unable to allocate color %s", name));
   }

   return color.pixel;
}

extern
Boolean
XULoadPixmap(Display*  display,
             XpmIcon*  icon,
             char*     default_pixmap[],
             Pixel     default_fg,
             Pixel     default_bg,
             String    purpose)
{
   int       screen = DefaultScreen(display);
   int       status;
   Pixmap    temp;

   icon->attributes.valuemask = 0;

   if (icon->name == None) {
      status = XpmCreatePixmapFromData(display,
                                       DefaultRootWindow(display),
                                       default_pixmap,
                                       &icon->pixmap,
                                       &icon->mask,
                                       &icon->attributes);
      switch (status) {
      case XpmColorError:
         warning(static_format("Xpm: unable to allocate exact colors"
                               " for pixmap for %s",
                               purpose));
         return True;
      case XpmSuccess:
         return True;
      case XpmNoMemory:
         error(static_format("Xpm: insufficient memory for pixmap for %s",
                             purpose));
         return False;
      case XpmColorFailed:
         error(static_format("Xpm: unable to allocate colors for pixmap for %s",
                             purpose));
         return False;
      default:
         error(static_format("Xpm: unable to create pixmap for %s:"
                             " failed with status %d", 
                             purpose, status));
         return False;
      }
   }

   status = XReadBitmapFile(display,
                            DefaultRootWindow(display),
                            icon->name,
                            &icon->attributes.width,
                            &icon->attributes.height,
                            &temp,
                            &icon->attributes.x_hotspot,
                            &icon->attributes.y_hotspot);

   if (status == BitmapSuccess) {
      icon->attributes.valuemask = XpmSize | XpmHotspot;

      icon->pixmap = XmuCreatePixmapFromBitmap(display,
                                               DefaultRootWindow(display),
                                               temp,
                                               icon->attributes.width,
                                               icon->attributes.height,
                                               DefaultDepth(display, screen),
                                               default_fg,
                                               default_bg);

      if (icon->mask_name == None) {
         icon->mask = (Pixmap)0;
      } else {
         unsigned int  width;
         unsigned int  height;
         int           x_hot;
         int           y_hot;

         status = XReadBitmapFile(display,
                                  DefaultRootWindow(display),
                                  icon->name,
                                  &width,
                                  &height,
                                  &icon->mask,
                                  &x_hot,
                                  &y_hot);
         if (status != BitmapSuccess) {
            icon->mask = (Pixmap)0;
         }
      }

      return True;
   }

   status = XpmReadFileToPixmap(display, 
                                DefaultRootWindow(display), 
                                icon->name,
                                &icon->pixmap,
                                &icon->mask,
                                &icon->attributes);

   switch (status) {
   case XpmColorError:
      warning(static_format("Xpm: unable to allocate exact colors"
                            " for pixmap %s for %s",
                            icon->name, purpose));
      return True;
   case XpmSuccess:
      return True;
   case XpmOpenFailed:
      error(static_format("Xpm: unable to read pixmap %s for %s:"
                          " unable to open file",
                          icon->name, purpose));
      return False;
   case XpmFileInvalid:
      error(static_format("Xpm: unable to read pixmap %s for %s: file invalid",
                          icon->name, purpose));
      return False;
   case XpmNoMemory:
      error(static_format("Xpm: unable to read pixmap %s for %s:"
                          " insufficient memory",
                          icon->name, purpose));
      return False;
   case XpmColorFailed:
      error(static_format("Xpm: unable to read pixmap %s for %s:"
                          " unable to allocate colors",
                          icon->name, purpose));
      return False;
   default:
      error(static_format("Xpm: unable to read pixmap %s for %s:"
                          " failed with status %d", 
                          icon->name, purpose, status));
      return False;
   }

   /* NOT REACHED */
}

#define TO_BYTES(bits) (((bits)+7)/8)

static
Pixmap
pad_pixmap(Display*   display,
           Pixmap     src,
           Dimension  pad_width,
           Dimension  pad_height)
{
   Window          root;
   int             x;
   int             y;
   unsigned int    width;
   unsigned int    height;
   unsigned int    border_width;
   unsigned int    depth;
   Pixmap          dest;
   XGCValues	    values;
   GC              gc;
   unsigned char*  full_bitmap;
   Pixmap          full;

   XGetGeometry(/* display             => */ display,
                /* d                   => */ src,
                /* root_return         => */ &root,
                /* x_return            => */ &x,
                /* y_return            => */ &y,
                /* width_return        => */ &width,
                /* height_return       => */ &height,
                /* border_width_return => */ &border_width,
                /* depth_return        => */ &depth);

   dest = XCreatePixmap(/* display  => */ display,
                        /* Drawable => */ root,
                        /* width    => */ width  + 2 * pad_width,
                        /* height   => */ height + 2 * pad_height,
                        /* depth    => */ depth);

   values.foreground = 1;
   values.foreground = 0;
   gc = XCreateGC(/* display   => */ display,
                  /* d         => */ dest,
                  /* valuemask => */ GCForeground | GCBackground,
                  /* values    => */ &values);

   XCopyArea(/* display => */ display,
             /* src     => */ src,
             /* dest    => */ dest,
             /* GC      => */ gc,
             /* src_x   => */ 0,
             /* src_y   => */ 0,
             /* width   => */ width,
             /* height  => */ height,
             /* dest_x  => */ pad_width,
             /* dest_y  => */ pad_height);

   XFreeGC(display, gc);

   return dest;
}

static
Pixmap
pad_pixmap_mask(Display*   display,
                Pixmap     src,
                Dimension  pad_width,
                Dimension  pad_height)
{
   Window          root;
   int             x;
   int             y;
   unsigned int    width;
   unsigned int    height;
   unsigned int    border_width;
   unsigned int    depth;
   unsigned char*  full_bitmap;
   Pixmap          dest;
   XGCValues	    values;
   GC              gc;

   XGetGeometry(/* display             => */ display,
                /* d                   => */ src,
                /* root_return         => */ &root,
                /* x_return            => */ &x,
                /* y_return            => */ &y,
                /* width_return        => */ &width,
                /* height_return       => */ &height,
                /* border_width_return => */ &border_width,
                /* depth_return        => */ &depth);

   full_bitmap = XtCalloc(TO_BYTES(width+2*pad_width) * (height+2*pad_height),
                          sizeof(unsigned char));
   dest = XCreatePixmapFromBitmapData(/* display => */ display,
                                      /* d       => */ root,
                                      /* data    => */ full_bitmap,
                                      /* width   => */ width  + 2 * pad_width,
                                      /* height  => */ height + 2 * pad_height,
                                      /* fg      => */ 1,
                                      /* bg      => */ 0,
                                      /* depth   => */ 1);
   XtFree(full_bitmap);

   values.foreground = 1;
   values.foreground = 0;
   gc = XCreateGC(/* display   => */ display,
                  /* d         => */ dest,
                  /* valuemask => */ GCForeground | GCBackground,
                  /* values    => */ &values);

   XCopyArea(/* display => */ display,
             /* src     => */ src,
             /* dest    => */ dest,
             /* GC      => */ gc,
             /* src_x   => */ 0,
             /* src_y   => */ 0,
             /* width   => */ width,
             /* height  => */ height,
             /* dest_x  => */ pad_width,
             /* dest_y  => */ pad_height);

   XFreeGC(display, gc);

   return dest;
}

extern
XpmIcon
XUPadPixmap(Display*   display,
            XpmIcon    src,
            Dimension  pad_width,
            Dimension  pad_height)
{
   XpmIcon  dest;

   if (pad_width == 0 && pad_height == 0) {
      return src;
   }

   dest.name       = NULL;
   dest.mask_name  = NULL;
   dest.pixmap     = pad_pixmap(display, src.pixmap, pad_width, pad_height);
   dest.mask       = pad_pixmap_mask(display, src.mask, pad_width, pad_height);
   dest.attributes = src.attributes;

   dest.attributes.width  = dest.attributes.width  + 2 * pad_width;
   dest.attributes.height = dest.attributes.height + 2 * pad_height;

   dest.attributes.x_hotspot = dest.attributes.x_hotspot + pad_width;
   dest.attributes.y_hotspot = dest.attributes.y_hotspot + pad_height;

   return dest;
}

