/*
** Copyright 1999 by Todd Allen.  All Rights Reserved.  Permission to use,
** copy, modify, distribute, and sell this software and its documentation for
** any purpose is hereby granted without fee, provided that the above
** copyright notice appear in all copies and that both the copyright notice
** and this permission notice appear in supporting documentation.
**
** No representations are made about the suitability of this software for any
** purpose.  It is provided ``as is'' without express or implied warranty,
** including but not limited to the warranties of merchantability, fitness
** for a particular purpose, and noninfringement.  In no event shall Todd
** Allen be liable for any claim, damages, or other liability, whether in
** action of contract, tort, or otherwise, arising from, out of, or in
** connection with this software.
*/

#include "types.h"
#include "pending.h"
#include "diags.h"
#include "timemath.h"
#ifdef DEBUG_PENDING_PARANOIA
#include <X11/Xaw/BoxCommand.h>
#endif

static  void      pending_heapify         (XPingRec*       xping,
                                           int             parent);


#define IS_PENDING(machine) ((machine)->pending != NOT_PENDING)


#ifdef DEBUG_PENDING_PARANOIA
static
void
pending_sanity_check(XPingRec*  xping)
{
	int  count = 0;
   int  i;

   printf("pending: ");
   for (i = 0; i < xping->numrows * xping->numcolumns; i++) {
      if (xping->machines[i].button != (Widget)NULL
          && XtClass(xping->machines[i].button) == boxCommandWidgetClass) {
         if (IS_PENDING(&xping->machines[i])) {
            printf("%s, ", xping->machines[i].name);
            assert(xping->machines[i].pid == NOT_RUNNING);
            assert(xping->pending[xping->machines[i].pending] 
                   == &xping->machines[i]);
            count++;
         }
      }
   }
   printf("\n");

   assert(count == xping->num_pending);
}
#endif

extern  
Machine*  
pending_push (XPingRec*  xping,
              Machine*   machine)
{
   int  child = xping->num_pending;

   assert(machine->pid == NOT_RUNNING);

   if (IS_PENDING(machine)) return;

#ifdef DEBUG_PENDING
   printf("pending push: %s, %d\n", machine->name, xping->num_pending+1);
#endif

   xping->pending[xping->num_pending] = machine;
   machine->pending = xping->num_pending;

   xping->num_pending++;

   assert(xping->num_pending <= xping->nummachines);

   while (child > 0) {
      int       parent = (child-1)/2;
      Machine*  temp;

      if (!lesstimeval(xping->pending[child]->last_info, 
                       xping->pending[parent]->last_info)) break;

      temp                   = xping->pending[parent];
      xping->pending[parent] = xping->pending[child];
      xping->pending[child]  = temp;

      xping->pending[parent]->pending = parent;
      xping->pending[child]->pending  = child;

      child = parent;
   }

#ifdef DEBUG_PENDING_PARANOIA
   pending_sanity_check(xping);
#endif   
}

static
void
pending_heapify(XPingRec*  xping,
                int        parent)
{
   for (;;) {
      int       left    = parent*2 + 1;
      int       right   = parent*2 + 2;
      int       soonest = parent;
      Machine*  temp;

      if (left < xping->num_pending 
          && lesstimeval(xping->pending[left]->last_info,
                         xping->pending[soonest]->last_info)) {
         soonest = left;
      }
      if (right < xping->num_pending 
          && lesstimeval(xping->pending[right]->last_info,
                         xping->pending[soonest]->last_info)) {
         soonest = right;
      }

      if (soonest == parent) break;

      temp                    = xping->pending[parent];
      xping->pending[parent]  = xping->pending[soonest];
      xping->pending[soonest] = temp;

      xping->pending[parent]->pending  = parent;
      xping->pending[soonest]->pending = soonest;

      parent = soonest;
   }
}

extern  
Machine*  
pending_pop (XPingRec*  xping)
{
   Machine*  result = xping->pending[0];

   assert(xping->num_pending > 0);

#ifdef DEBUG_PENDING
   printf("pending pop: %s, %d\n", result->name, xping->num_pending-1);
#endif

   xping->num_pending--;
   xping->pending[0] = xping->pending[xping->num_pending];
   xping->pending[0]->pending = 0;

   pending_heapify(xping, 0);

   assert(IS_PENDING(result));
   assert(result->pid == NOT_RUNNING);

   result->pending = NOT_PENDING;

#ifdef DEBUG_PENDING_PARANOIA
   pending_sanity_check(xping);
#endif   

   return result;
}

extern  
void
pending_remove (XPingRec*  xping,
                Machine*   machine)
{
   if (!IS_PENDING(machine)) return;

   assert(machine->pid == NOT_RUNNING);

#ifdef DEBUG_PENDING
   printf("pending remove: %s (%d), %d\n", 
          machine->name, machine->pending, xping->num_pending-1);
#endif

   assert(xping->num_pending > 0);
   assert(machine->pending < xping->num_pending);

   xping->num_pending--;
   xping->pending[machine->pending] = xping->pending[xping->num_pending];
   xping->pending[machine->pending]->pending = machine->pending;

   pending_heapify(xping, machine->pending);

   machine->pending = NOT_PENDING;

#ifdef DEBUG_PENDING_PARANOIA
   pending_sanity_check(xping);
#endif   
}
