/*
** Copyright 1999 by Todd Allen.  All Rights Reserved.  Permission to use,
** copy, modify, distribute, and sell this software and its documentation for
** any purpose is hereby granted without fee, provided that the above
** copyright notice appear in all copies and that both the copyright notice
** and this permission notice appear in supporting documentation.
**
** No representations are made about the suitability of this software for any
** purpose.  It is provided ``as is'' without express or implied warranty,
** including but not limited to the warranties of merchantability, fitness
** for a particular purpose, and noninfringement.  In no event shall Todd
** Allen be liable for any claim, damages, or other liability, whether in
** action of contract, tort, or otherwise, arising from, out of, or in
** connection with this software.
*/

#include "parse.h"
#include "diags.h"

extern
void
parse_machines (String  str,
                Grid*   grid,
                int*    numrows,
                int*    numcolumns,
                int*    nummachines)
{
   Grid   result;

   {
      Boolean  in_row     = False;
      Boolean  in_name    = False;
      int      rows       = 0;
      int      maxcolumns = 0;
      int      columns    = 0;
      int      machines   = 0;
      char*    s          = str;

      for (; *s != '\0'; s++) {
         switch (*s) {
         case '{':
            if (in_row) {
               *(s+1) = '\0';
               fatal(static_format("unexpected '{' in machine list when already"
                                   " in row: %s", 
                                   str));
            }
            if (in_name) {
               in_name = False;
               rows++;
               machines++;
            }
            assert(columns == 0);
            in_row = True;
            break;
         case '}':
            if (!in_row) {
               *(s+1) = '\0';
               fatal(static_format("unexpected '}' in machine list when not"
                                   " in row: %s", 
                                   str));
            }
            if (in_name) {
               in_name = False;
               columns++;
               machines++;
            }
            rows++;
            if (columns > maxcolumns) maxcolumns = columns;
            columns = 0;
            in_row = False;
            break;
         case ' ':
         case '\t':
            if (in_name) {
               in_name = False;
               if (in_row) {
                  columns++;
               } else {
                  rows++;
                  if (1 > maxcolumns) maxcolumns = 1;
               }
               machines++;
            }
            break;
         default:
            in_name = True;
            break;
         }
      }

      if (in_row) {
         fatal("unmatched '{' in machine list: row not completed");
      }

      if (in_name) {
         rows++;
         if (1 > maxcolumns) maxcolumns = 1;
         machines++;
      }

      result       = (Grid)XtCalloc(rows * maxcolumns, sizeof(Machine));
      *numrows     = rows;
      *numcolumns  = maxcolumns;
      *nummachines = machines;
   }

   {
      Boolean  in_row = False;
      char*    name   = (char*)NULL;
      int      row    = 0;
      int      column = 0;
      char*    s;
      
      s = XtNewString(str);
      for (; *s != '\0'; s++) {
         switch (*s) {
         case '{':
            assert(!in_row);
            if (name != (char*)NULL) {
               *s = '\0';
               result[row*(*numcolumns)+column].name = name;
               name = (char*)NULL;
               row++;
            }
            assert(column == 0);
            in_row = True;
            break;
         case '}':
            assert(in_row);
            if (name != (char*)NULL) {
               *s = '\0';
               result[row*(*numcolumns)+column].name = name;
               name = (char*)NULL;
               column++;
            }
            column = 0;
            row++;
            in_row = False;
            break;
         case ' ':
         case '\t':
            if (name != (char*)NULL) {
               *s = '\0';
               result[row*(*numcolumns)+column].name = name;
               name = (char*)NULL;
               if (in_row) {
                  column++;
               } else {
                  row++;
               }
            }
            break;
         default:
            if (name == (char*)NULL) {
               name = s;
            }
            break;
         }
      }

      assert(!in_row);

      if (name != (char*)NULL) {
         assert(*s == '\0');
         result[row*(*numcolumns)+column].name = name;
         row++;
      }

      assert(row == *numrows);
   }

#ifdef DEBUG_MACHINES
   {
      int  i;
      int  j;

      printf("numrows    = %d\n", *numrows);
      printf("numcolumns = %d\n", *numcolumns);

      for (i = 0; i < *numrows; i++) {
         for (j = 0; j < *numcolumns; j++) {
            if (result[i*(*numcolumns)+j].name == (String)NULL) {
               printf("NIL\t");
            } else {
               printf("%s\t", result[i*(*numcolumns)+j].name);
            }
         }
         printf("\n");
      }
   }
#endif

   *grid = result;
}  

extern
void
parse_command (String    str,
               String    description,
               String**  args,
               int*      arg_num)
{
   int      words  = 0;
   String*  result;
   int      num;
   
#ifdef DEBUG_COMMAND
   printf("command: %s\n", str);
#endif

   {
      Boolean  in_word         = False;
      Boolean  in_quote        = False;
      char*    quote_char;
      Boolean  after_backslash = False;
      char*    s               = str;

      for (; *s != '\0'; s++) {
         switch (*s) {
         case ' ':
         case '\t':
            if (after_backslash) {
               assert(in_word);
               after_backslash = False;
            } else if (in_quote) {
               assert(in_word);
            } else {
               if (in_word) {
                  words++;
                  in_word = False;
               }
            }
            break;
         case '\"':
         case '\'':
            if (after_backslash) {
               assert(in_word);
               after_backslash = False;
            } else {
               quote_char = s;
               in_quote   = !in_quote;
               in_word    = True;
            }
            break;
         case '\\':
            if (after_backslash) {
               assert(in_word);
               after_backslash = False;
            } else {
               after_backslash = True;
               in_word         = True;
            }
            break;
         default:
            in_word = True;
            break;
         }
      }

      if (after_backslash) {
         fatal("trailing \'\\\' character in command");
      }
      if (in_quote) {
         fatal(static_format("unmatched quote character in command: %s", 
                             quote_char));
      }
      if (in_word) {
         words++;
      }
   }

#ifdef DEBUG_COMMAND
   printf("words = %d\n", words);
#endif

   result = (String*)XtCalloc(words+1, sizeof(String));

   {
      Boolean  in_quote        = False;
      Boolean  after_backslash = False;
      Boolean  in_word         = False;
      int      word            = 0;
      char*    text            = XtCalloc(strlen(str)+1, sizeof(char));
      char*    character       = text;
      char*    s               = str;

      for (; *s != '\0'; s++) {
         switch (*s) {
         case ' ':
         case '\t':
            if (after_backslash) {
               assert(in_word);
               *character = *s;
               character++;
               after_backslash = False;
            } else if (in_quote) {
               assert(in_word);
               *character = *s;
               character++;
            } else {
               if (in_word) {
                  *character = '\0';
                  result[word] = XtNewString(text);
                  character = text;
                  in_word = False;
                  word++;
               }
            }
            break;
         case '\"':
         case '\'':
            if (after_backslash) {
               assert(in_word);
               *character = *s;
               character++;
               after_backslash = False;
            } else {
               in_quote = !in_quote;
               in_word  = True;
            }
            break;
         case '\\':
            if (after_backslash) {
               assert(in_word);
               *character = *s;
               character++;
               after_backslash = False;
            } else {
               after_backslash = True;
               in_word         = True;
            }
            break;
         default:
            *character = *s;
            character++;
            in_word = True;
            break;
         }
      }

      assert(!after_backslash);
      assert(!in_quote);
      if (in_word) {
         *character = '\0';
         result[word] = XtNewString(text);
      }

      XtFree(text);
   }

   {
      int  i;

      num = -1;

      for (i = 0; i < words; i++) {
         if (strcmp(result[i], "%s") == 0) {
            if (num == -1) {
               num = i;
            } else {
               fatal(static_format("replacement string \"%%s\" appears more"
                                   " than once in %s", 
                                   description));
            }
         }
      }

      if (num == -1) {
         fatal(static_format("replacement string \"%%s\" does not appear in %s",
                             description));
      }
   }

   result[words] = NULL;

   *args    = result;
   *arg_num = num;

#ifdef DEBUG_COMMAND
   {
      int  i;

      printf("arg_num = %d\n", num);

      printf("command:\n");
      for (i = 0; i < words; i++) {
         printf("\t%s\n", result[i]);
      }
   }
#endif
}

