/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   Samba system utilities
   Copyright (C) Andrew Tridgell 1992-1998
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

extern int DEBUGLEVEL;

#ifdef SMB_SNMP
static char lastPwd[255], lastVariable[255], lastOpenFile[64]; 
static time_t lastTime=0;
static short debug=0;
#define SNMP_READ_TIMEOUT 2 /* seconds */

#define RAW_OUTPUT       1
#define HTML_OUTPUT      2
#define TEXT_OUTPUT      3
#define SLK_OUTPUT       4
#endif /* SMB_SNMP */

/*
   The idea is that this file will eventually have wrappers around all
   important system calls in samba. The aims are:

   - to enable easier porting by putting OS dependent stuff in here

   - to allow for hooks into other "pseudo-filesystems"

   - to allow easier integration of things like the japanese extensions

   - to support the philosophy of Samba to expose the features of
     the OS within the SMB model. In general whatever file/printer/variable
     expansions/etc make sense to the OS should be acceptable to Samba.
*/


/*******************************************************************
this replaces the normal select() system call
return if some data has arrived on one of the file descriptors
return -1 means error
********************************************************************/
#ifdef NO_SELECT
static int pollfd(int fd)
{
  int     r=0;

#ifdef HAS_RDCHK
  r = rdchk(fd);
#elif defined(TCRDCHK)
  (void)ioctl(fd, TCRDCHK, &r);
#else
  (void)ioctl(fd, FIONREAD, &r);
#endif

  return(r);
}

int sys_select(fd_set *fds,struct timeval *tval)
{
  fd_set fds2;
  int counter=0;
  int found=0;

  FD_ZERO(&fds2);

  while (1) 
    {
      int i;
      for (i=0;i<255;i++) {
	if (FD_ISSET(i,fds) && pollfd(i)>0) {
	  found++;
	  FD_SET(i,&fds2);
	}
      }

      if (found) {
	memcpy((void *)fds,(void *)&fds2,sizeof(fds2));
	return(found);
      }
      
      if (tval && tval->tv_sec < counter) return(0);
      sleep(1);
      counter++;
    }
}

#else
int sys_select(fd_set *fds,struct timeval *tval)
{
  struct timeval t2;
  int selrtn;

  do {
    if (tval) memcpy((void *)&t2,(void *)tval,sizeof(t2));
    errno = 0;
    selrtn = select(255,SELECT_CAST fds,NULL,NULL,tval?&t2:NULL);
  } while (selrtn<0 && errno == EINTR);

  return(selrtn);
}
#endif


/*******************************************************************
just a unlink wrapper
********************************************************************/
int sys_unlink(char *fname)
{
#ifdef SMB_SNMP
  char* fileName = dos_to_unix(fname,False);
  int len = strlen(lastPwd);

  if(debug) printf("sys_unlink: '%s' [%s]\n", fileName, lastPwd);
 
  if((strcmp(fileName, "desktop.ini") != 0) && (fileName[strlen(fileName)-1] != '~')
     && (len >= 6) && (strcmp(&lastPwd[len-6], "/SNMP/") == 0)) {
    char  hostName[64], variable[64], command[1024];
    int idx, rc, idx1, port=161;

    for(idx=0; fileName[idx] !='/'; idx++) ;
    strncpy(hostName, fileName, idx);
    hostName[idx] = '\0';
    for(idx1=idx+1; fileName[idx1] !='\0'; idx1++)
      if(fileName[idx1] == '/')
	fileName[idx1] = '.';

    strcpy(variable, &fileName[idx+1]);

    for(idx=strlen(hostName)-1; idx>0; idx--) {      
      if(hostName[idx] == '@') {
	hostName[idx] = '\0';
	port=atoi(&hostName[idx+1]);
	break;
      }
    }


    /* In SNMP a variable can be deleted by setting it to an empty value */
    sprintf(command, "%s -p %d %s private %s s \"\"",
	    SNMPSET, port, hostName, variable);
    if(debug) printf("%s\n", command);
    rc = system(command);
    if(debug) printf("RC=%d\n", rc);

    if(rc == 0)
      return(unlink(fileName)); /* Delete the file */
    else
      return(rc);  
  } else
    return(unlink(fileName));
#else
  return(unlink(dos_to_unix(fname,False)));
#endif /* SMB_SNMP */
}


#ifdef SMB_SNMP
/*******************************************************************
********************************************************************/

int rawFormatter(FILE* fd_in, FILE* fd_out, int walkTable) {
  int numRows = 0;
  char strIn[1024];

  while(fgets(strIn, 1024, fd_in) != NULL) {

    strIn[strlen(strIn)-1] = '\0'; /* Remove CR */

    if(strIn[0] != '\0') {
      if(debug) printf("OUTPUT: '%s'\n", strIn);		

      if(numRows > 0) fprintf(fd_out, "\n");

      if(walkTable == 0) {
	char *token=strtok(strIn, "=");
	fprintf(fd_out, "%s", &token[strlen(token)+2]);
      } else { /* Table Walk */
	fprintf(fd_out, "%s", strIn);
      }
	
      numRows++;
    }
  }

  return(numRows);
}

/*******************************************************************
********************************************************************/
int textFormatter(FILE* fd_in, FILE* fd_out, int outputType) {
  int numColumns = 0, idx, i, numRows=0, count=0, columnIdx=0, first=1;
  char strIn[1024], tokenizer[1024], *token, *actToken;
  char *columnNames[1024], *columnValues[1024];

  memset(columnNames, 0, sizeof(char*)*1024);
  memset(columnValues, 0, sizeof(char*)*1024);

  while(fgets(strIn, 1024, fd_in) != NULL) {
    strIn[strlen(strIn)-1] = '\0'; /* Remove CR */
    if(debug) printf("OUTPUT: '%s'\n", strIn);
    columnValues[columnIdx++] = strdup(strIn);
  }

  if(columnIdx==0) return(0);

  strcpy(tokenizer, columnValues[0]);
  for(idx = 0, token=strtok(tokenizer, "."); token != NULL; token=strtok(NULL, ".")) {
    if(debug) printf("TOKEN: '%s'\n", token);
    
    if(!isdigit(token[0]))
      idx += strlen(token)+1;
    else {
      idx--;
      if(debug) printf("COLUMN='%s'\n", actToken);
      columnNames[numColumns++] = strdup(actToken);
      strncpy(tokenizer, columnValues[0], idx);
      break;
    }
    
    actToken = token;
  }
  

  numRows=1;
  for(idx=1; idx<columnIdx; idx++)
    if(strncmp(tokenizer, columnValues[idx], strlen(tokenizer)) == 0)
      numRows++;
    else
      break;

  if(debug) printf("[numRows = %d]\n", numRows);

  /* Print the head */
  for(; columnValues[idx] != NULL; idx=idx+numRows) {
   strcpy(tokenizer, columnValues[idx]);
   for(i = 0, token=strtok(tokenizer, "."); token != NULL; token=strtok(NULL, ".")) {
     if(debug) printf("TOKEN: '%s'\n", token);
     
     if(isdigit(token[0])) {
       if(debug) printf("COLUMN='%s'\n", actToken);
       columnNames[numColumns++] = strdup(actToken);
       break;
     }
     
     actToken = token;
   }   
  }

  if(debug) printf("[numColumns = %d]\n", numColumns);

  switch(outputType) {
  case HTML_OUTPUT:
    fprintf(fd_out, "<TABLE BORDER=1>\n<TR>\n");
    break;
  case SLK_OUTPUT:
    fprintf(fd_out, "ID;PWXL;N;E\nP;PGeneral\nP;P0\nP;P0.00\nP;P#,##0\nP;P#,##0.00\nP;P#,##0;;\-#,##0\nP;P#,##0;;[Red]\-#,##0\nP;P#,##0.00;;\-#,##0.00\nP;P#,##0.00;;[Red]\-#,##0.00\nP;P\"$\"\ #,##0;;\-\"$\"\ #,##0\nP;P\"$\"\ #,##0;;[Red]\-\"$\"\ #,##0\nP;P\"$\"\ #,##0.00;;\-\"$\"\ #,##0.00\nP;P\"$\"\ #,##0.00;;[Red]\-\"$\"\ #,##0.00\nP;P0%\nP;P0.00%\nP;P0.00E+00\nP;P##0.0E+0\nP;P#\ ?/?\nP;P#\ ??/??\nP;Pdd/mm/yy\nP;Pdd\-mmm\-yy\nP;Pdd\-mmm\nP;Pmmm\-yy\nP;Ph:mm\ AM/PM\nP;Ph:mm:ss\ AM/PM\nP;Ph:mm\nP;Ph:mm:ss\nP;Pdd/mm/yy\ h:mm\nP;Pmm:ss\nP;Pmm:ss.0\nP;P@\nP;P[h]:mm:ss\nP;P_-\"$\"\ * #,##0_-;;\-\"$\"\ * #,##0_-;;_-\"$\"\ * \"-\"_-;;_-@_-\nP;P_-* #,##0_-;;\-* #,##0_-;;_-* \"-\"_-;;_-@_-\nP;P_-\"$\"\ * #,##0.00_-;;\-\"$\"\ * #,##0.00_-;;_-\"$\"\ * \"-\"??_-;;_-@_-\nP;P_-* #,##0.00_-;;\-* #,##0.00_-;;_-* \"-\"??_-;;_-@_-\nP;FArial;M200\nP;FArial;M200\nP;FArial;M200\nP;FArial;M200\nP;EArial;M200\nP;EArial;M200\nP;EArial;M200\nP;EArial;M200\nP;EArial;M200\nP;EArial;M200\nP;EArial;M200\nP;EArial;M200\nP;EArial;M200\nP;EArial;M200\nF;P0;DG0G8;M255\nB;Y4;X3;D0 0 3 2\n");
    break;
  default:
    ;
  }

  for(idx=0; idx<numColumns; idx++) {
    switch(outputType) {
    case SLK_OUTPUT:
      fprintf(fd_out, "F;W%d %d 25\n", idx+1, idx+1);
      break;
    }
  }
  
  for(idx=0; idx<numColumns; idx++) {
    switch(outputType) {
    case HTML_OUTPUT:
      fprintf(fd_out, "<TH>%s</TH> \n", columnNames[idx]);
      break;
    case TEXT_OUTPUT:
      fprintf(fd_out, "%s\t", columnNames[idx]);
      break;
    case SLK_OUTPUT:
      fprintf(fd_out, "C;Y1;X%d;K\"%s\"\n", idx+1, columnNames[idx]);
      break;
    default:
      fprintf(fd_out, "\t%s", columnNames[idx]);
    }

    if(debug) printf("\t%s", columnNames[idx]);
    free(columnNames[idx]);
  }

  switch(outputType) {
  case HTML_OUTPUT:
    fprintf(fd_out, "</TR>\n");
    break;
  case SLK_OUTPUT:
    break;
  default:
    fprintf(fd_out, "\n");
    if(debug) printf("\n");    
  }
  


  for(i=0; i<numRows; i++) {
    switch(outputType) {
    case HTML_OUTPUT:
      fprintf(fd_out, "<TR> ", token);
      break;
    case SLK_OUTPUT:
      break;
    default:
      fprintf(fd_out, "\n", token);
    }
    if(debug) printf("\n");

    
    for(idx=i, count=1; idx<columnIdx; idx+=numRows, count++) {
      if(columnValues[idx] != NULL) {
	token=strtok(columnValues[idx], "=");
	token=&token[strlen(token)+2];
	
	switch(outputType) {
	case HTML_OUTPUT:
	  fprintf(fd_out, "<TD>%s</TD> ", token);
	  break;
	case TEXT_OUTPUT:
	  fprintf(fd_out, "%s\t", token);
	  break;
	case SLK_OUTPUT:
	  fprintf(fd_out, "C;Y%d;X%d;K\"%s\"\n",  i+2, count, token);
	  break;
	default:
	  fprintf(fd_out, "%s\t", token);
	}
	
	if(debug) printf("%s\s", token);
	
	free(columnValues[idx]);
      }
    }

    switch(outputType) {
    case HTML_OUTPUT:
      fprintf(fd_out, "</TR>\n", token);
      break;
    case SLK_OUTPUT:
      break;
    default:
      fprintf(fd_out, "\n", token);
      break;
    }
  }

  switch(outputType) {
  case HTML_OUTPUT:
    fprintf(fd_out, "</TR>\n</TABLE>\n");
    break;
  case SLK_OUTPUT:
    fprintf(fd_out, "E\n");
    break;
  default:
    fprintf(fd_out, "\n");
  }


  if(debug) printf("\n [numRows = %d]\n", numRows);
 
  return(numRows);
}

char* getLastPwd()      { return(lastPwd);      } /* L.Deri */
char* getLastOpenFile() { return(lastOpenFile); } /* L.Deri */

#endif /* SMB_SNMP */


/*******************************************************************
a simple open() wrapper
********************************************************************/
int sys_open(char *fname,int flags,int mode)
{
#ifdef SMB_SNMP
  char* fileName = dos_to_unix(fname,False);
  int len = strlen(lastPwd);
  struct stat sbuf;
  int writable;

  if(debug) printf("sys_open: '%s' [%s][mode=%d]\n", fileName, lastPwd, mode); /* L.Deri */

  strcpy(lastOpenFile, fileName);

  if((strcmp(fileName, "desktop.ini") != 0) && (fileName[strlen(fileName)-1] != '~')
     && (len >= 6) && (strcmp(&lastPwd[len-6], "/SNMP/") == 0)) {
    /* This is a leaf of the SNMP tree -> is a SNMP variable to read */
    char hostName[64], variable[64], command[256], tmpStr[256], *trailer;
    int idx, rc, variable_len, executable, port=161;
    FILE *fd;
    time_t _lastTime = time(NULL);

    /* printf("TIME: actual='%ld' last='%ld'\n", _lastTime, lastTime); */

    for(idx=0; fileName[idx] !='/'; idx++) ;
    strncpy(hostName, fileName, idx);

    hostName[idx] = '\0';
    strcpy(variable, &fileName[idx+1]);

    if(strncmp(variable, TRAPS_FOLDER, strlen(TRAPS_FOLDER)) == 0) {
      /* Make sure that the file exists first */
      sprintf(command, "touch %s%s", lastPwd, fileName, lastPwd, fileName);
      system(command);
      return(open(fileName,flags,mode));
    }

    for(idx=strlen(variable)-1; idx > 0; idx--)
      if(variable[idx] == '/')
	variable[idx] = '.';

    variable_len = strlen(variable);

    for(idx=strlen(hostName)-1; idx>0; idx--) {      
      if(hostName[idx] == '@') {
	hostName[idx] = '\0';
	port=atoi(&hostName[idx+1]);
	break;
      }
    }

    if(!((strcmp(variable, lastVariable) == 0) && (_lastTime < (lastTime+SNMP_READ_TIMEOUT)))) {
      struct stat sbuf;
      int rc, walkTable, outputType=RAW_OUTPUT;
      
      lastTime = _lastTime;

      sprintf(command, "%s%s.sh", lastPwd, fileName);
      rc = stat(command, &sbuf);
		
      if(rc == 0) {
	/* the file exists -> it's going to be executed */
	sprintf(command, "%s%s.sh > %s%s", lastPwd, fileName, lastPwd, fileName);
	/* sprintf(command, "%s%s.sh", lastPwd, fileName); */
	if(debug) printf("EXEC: '%s'\n", command);
	system(command);
      } else {      
	strcpy(lastVariable, variable);

	if(strcmp(&variable[variable_len-2], ".0") == 0) {
	  sprintf(command, "%s -p %d %s public %s", SNMPGET, port, hostName, variable);
	  walkTable = 0;
	}
	else {
	  if(strcmp(&variable[variable_len-5], ".html") == 0) {
	    outputType=HTML_OUTPUT;
	    variable[variable_len-5]='\0';
	    sprintf(command, "%s -p %d %s public %s", SNMPWALK, port, hostName, variable);
	  } else if(strcmp(&variable[variable_len-4], ".txt") == 0) {
	    outputType=TEXT_OUTPUT;
	    variable[variable_len-4]='\0';
	    sprintf(command, "%s -p %d %s public %s", SNMPWALK, port, hostName, variable);	  
	  } else if(strcmp(&variable[variable_len-4], ".slk") == 0) {
	    outputType=SLK_OUTPUT;
	    variable[variable_len-4]='\0';
	    sprintf(command, "%s -p %d %s public %s", SNMPWALK, port, hostName, variable);	  
	  } else
	    sprintf(command, "%s -p %d %s public %s", SNMPWALK, port, hostName, variable);

	  walkTable = 1;
	}

	if(debug) printf("EXEC: '%s'\n", command);
	fd = popen(command, "r");
	if(fd != NULL) {
	  FILE *fd_out;
	  int numRows=0;	  
	  int fileToBeCreated;
	  sprintf(command, "%s%s", lastPwd, fileName);
	  if(debug) printf("OUT: '%s'\n", command);
	  rc = stat(command, &sbuf);
	
	  if(rc == 0) {
	    /* The file exists already */
	    fileToBeCreated = 0;
	    if((sbuf.st_mode & S_IWUSR)
	       || (sbuf.st_mode & S_IWGRP)
	       || (sbuf.st_mode & S_IWOTH)) {
	      writable=1;
	    } else {
	      writable=0;
	      sprintf(tmpStr, "chmod 666 %s%s", lastPwd, fileName);
	      system(tmpStr);
	    }
	  } else {
	    /* The file does not exist */
	    fileToBeCreated = 1;
	    if(outputType != RAW_OUTPUT) {
	      /* The only file that can be created is a plain file because
		 the missing variable is supposed NOT to be a table entry */
	      return(-1);
	    }
	  }

	  variable_len++;
	  fd_out = fopen(command, "w+");
	  if(fd_out != NULL) {
	    switch(outputType) {
	    case HTML_OUTPUT:
	    case TEXT_OUTPUT:
	    case SLK_OUTPUT:
	      numRows = textFormatter(fd, fd_out, outputType);
	      break;
	    case RAW_OUTPUT:
	    default:
	      numRows = rawFormatter(fd, fd_out, walkTable);
	      break;
	    }

	    fclose(fd_out);

	    if(fileToBeCreated && (numRows == 0)) {	    
	      unlink(command);
	      return(-1);
	    }
	  }

	  if(writable == 0) {
	    sprintf(tmpStr, "chmod 444 %s%s", lastPwd, fileName);
	    system(tmpStr);
	  }

	  pclose(fd);

	  if(numRows == 0) /* No Response */
	    return(-1);
	}
      }
    } /* if */
  }

  return(open(fileName,flags,mode));
#else
  return(open(dos_to_unix(fname,False),flags,mode));
#endif /* SMB_SNMP */
}


/*******************************************************************
a simple opendir() wrapper
********************************************************************/
DIR *sys_opendir(char *dname)
{
  return(opendir(dos_to_unix(dname,False)));
}


/*******************************************************************
and a stat() wrapper
********************************************************************/
int sys_stat(char *fname,struct stat *sbuf)
{
#ifdef SMB_SNMP
  char* fileName = dos_to_unix(fname,False);
  int len = strlen(lastPwd);

  if(debug) printf("sys_stat: '%s' [%s]\n", fileName, lastPwd); /* L.Deri */
  if((len >= 6) && (strcmp(&lastPwd[len-6], "/SNMP/") == 0)) {
    /* This is a leaf of the SNMP tree -> is a SNMP variable to read */
    int rc;

    rc = stat(fileName,sbuf);

    if(rc != 0) {
      char myString[255];
      strcpy(myString, fileName);
      strcat(myString, ".sh");

      rc = stat(myString,sbuf);
    } else
      rc = stat(fileName,sbuf);

    /* Force reload (L.Deri) */
    sbuf->st_mtime = -1;
    sbuf->st_ctime = -1;

    return(rc);
  } else
    return(stat(fileName,sbuf));
#else
  return(stat(dos_to_unix(fname,False),sbuf));
#endif /* SMB_SNMP */
}

/*******************************************************************
The wait() calls vary between systems
********************************************************************/
int sys_waitpid(pid_t pid,int *status,int options)
{
#ifdef USE_WAITPID
  return waitpid(pid,status,options);
#else /* USE_WAITPID */
  return wait4(pid, status, options, NULL);
#endif /* USE_WAITPID */
}

/*******************************************************************
don't forget lstat()
********************************************************************/
int sys_lstat(char *fname,struct stat *sbuf)
{
  return(lstat(dos_to_unix(fname,False),sbuf));
}


/*******************************************************************
mkdir() gets a wrapper
********************************************************************/
int sys_mkdir(char *dname,int mode)
{
#ifdef SMB_SNMP
  char* name = dos_to_unix(dname,False);
  int len = strlen(lastPwd);
  
  /* if(debug) */ printf("sys_mkdir: %s [%s]\n", name, getLastPwd()); /* L.Deri */

  if((len >= 6) && (strcmp(&lastPwd[len-6], "/SNMP/") == 0)) 
    return(-1);
  else    
    return(mkdir(name, mode));
#else
  return(mkdir(dos_to_unix(dname,False),mode));
#endif /* SMB_SNMP */
}


/*******************************************************************
do does rmdir()
********************************************************************/
int sys_rmdir(char *dname)
{
#ifdef SMB_SNMP
  char* name = dos_to_unix(dname,False);
  int len = strlen(lastPwd);
  
  /* if(debug) */ printf("sys_rmdir: %s [%s]\n", name, getLastPwd()); /* L.Deri */

  if((len >= 6) && (strcmp(&lastPwd[len-6], "/SNMP/") == 0)) 
    return(-1);
  else    
    return(rmdir(name));
#else
  return(rmdir(dos_to_unix(dname,False)));
#endif /* SMB_SNMP */
}


/*******************************************************************
I almost forgot chdir()
********************************************************************/
int sys_chdir(char *dname)
{
#ifdef SMB_SNMP
  char* name = dos_to_unix(dname,False);
  strcpy(lastPwd, name);
  return(chdir(name));
#else
  return(chdir(dos_to_unix(dname,False)));
#endif /* SMB_SNMP */
}


/*******************************************************************
now for utime()
********************************************************************/
int sys_utime(char *fname,struct utimbuf *times)
{
  /* if the modtime is 0 or -1 then ignore the call and
     return success */
  if (times->modtime == (time_t)0 || times->modtime == (time_t)-1)
    return 0;
  
  /* if the access time is 0 or -1 then set it to the modtime */
  if (times->actime == (time_t)0 || times->actime == (time_t)-1)
    times->actime = times->modtime;
   
  return(utime(dos_to_unix(fname,False),times));
}

/*********************************************************
for rename across filesystems Patch from Warren Birnbaum 
<warrenb@hpcvscdp.cv.hp.com>
**********************************************************/

static int copy_reg(char *source, const char *dest)
{
  struct stat source_stats;
  int ifd;
  int full_write();
  int safe_read();
  int ofd;
  char *buf;
  int len;                      /* Number of bytes read into `buf'. */

  lstat (source, &source_stats);
  if (!S_ISREG (source_stats.st_mode))
    {
      return 1;
    }

  if (unlink (dest) && errno != ENOENT)
    {
      return 1;
    }

  if((ifd = open (source, O_RDONLY, 0)) < 0)
    {
      return 1;
    }
  if((ofd = open (dest, O_WRONLY | O_CREAT | O_TRUNC, 0600)) < 0 )
    {
      close (ifd);
      return 1;
    }

  if((buf = malloc( COPYBUF_SIZE )) == NULL)
    {
      close (ifd);  
      close (ofd);  
      unlink (dest);
      return 1;
    }

  while ((len = read(ifd, buf, COPYBUF_SIZE)) > 0)
    {
      if (write_data(ofd, buf, len) < 0)
        {
          close (ifd);
          close (ofd);
          unlink (dest);
          free(buf);
          return 1;
        }
    }
  free(buf);
  if (len < 0)
    {
      close (ifd);
      close (ofd);
      unlink (dest);
      return 1;
    }

  if (close (ifd) < 0)
    {
      close (ofd);
      return 1;
    }
  if (close (ofd) < 0)
    {
      return 1;
    }

  /* chown turns off set[ug]id bits for non-root,
     so do the chmod last.  */

  /* Try to copy the old file's modtime and access time.  */
  {
    struct utimbuf tv;

    tv.actime = source_stats.st_atime;
    tv.modtime = source_stats.st_mtime;
    if (utime (dest, &tv))
      {
        return 1;
      }
  }

  /* Try to preserve ownership.  For non-root it might fail, but that's ok.
     But root probably wants to know, e.g. if NFS disallows it.  */
  if (chown (dest, source_stats.st_uid, source_stats.st_gid)
      && (errno != EPERM))
    {
      return 1;
    }

  if (chmod (dest, source_stats.st_mode & 07777))
    {
      return 1;
    }
  unlink (source);
  return 0;
}

/*******************************************************************
for rename()
********************************************************************/
int sys_rename(char *from, char *to)
{
    int rcode;  
    pstring zfrom, zto;

    pstrcpy (zfrom, dos_to_unix (from, False));
    pstrcpy (zto, dos_to_unix (to, False));
    rcode = rename (zfrom, zto);

    if (errno == EXDEV) 
      {
        /* Rename across filesystems needed. */
        rcode = copy_reg (zfrom, zto);        
      }
    return rcode;
}

/*******************************************************************
for chmod
********************************************************************/
int sys_chmod(char *fname,int mode)
{
#ifdef SMB_SNMP
  char* fileName = dos_to_unix(fname,False);
  int len = strlen(lastPwd);

  if(debug) printf("sys_chmod: '%s' [%s]\n", fileName, lastPwd); /* L.Deri */

  if((len >= 6) && (strcmp(&lastPwd[len-6], "/SNMP/") == 0))
    return(-1);
  else
    return(chmod(fileName,mode));
#else
  return(chmod(dos_to_unix(fname,False),mode));
#endif
}

/*******************************************************************
for getwd
********************************************************************/
char *sys_getwd(char *s)
{
  char *wd;
#ifdef USE_GETCWD
  wd = (char *) getcwd (s, sizeof (pstring));
#else
  wd = (char *) getwd (s);
#endif
  if (wd)
    unix_to_dos (wd, True);
  return wd;
}

/*******************************************************************
chown isn't used much but OS/2 doesn't have it
********************************************************************/
int sys_chown(char *fname,int uid,int gid)
{
#ifdef NO_CHOWN
  DEBUG(1,("Warning - chown(%s,%d,%d) not done\n",fname,uid,gid));
#else
  return(chown(fname,uid,gid));
#endif
}

/*******************************************************************
os/2 also doesn't have chroot
********************************************************************/
int sys_chroot(char *dname)
{
#ifdef NO_CHROOT
  DEBUG(1,("Warning - chroot(%s) not done\n",dname));
#else
  return(chroot(dname));
#endif
}

/**************************************************************************
A wrapper for gethostbyname() that tries avoids looking up hostnames 
in the root domain, which can cause dial-on-demand links to come up for no
apparent reason.
****************************************************************************/
struct hostent *sys_gethostbyname(char *name)
{
#ifdef REDUCE_ROOT_DNS_LOOKUPS
  char query[256], hostname[256];
  char *domain;

  /* Does this name have any dots in it? If so, make no change */

  if (strchr(name, '.'))
    return(gethostbyname(name));

  /* Get my hostname, which should have domain name 
     attached. If not, just do the gethostname on the
     original string. 
  */

  gethostname(hostname, sizeof(hostname) - 1);
  hostname[sizeof(hostname) - 1] = 0;
  if ((domain = strchr(hostname, '.')) == NULL)
    return(gethostbyname(name));

  /* Attach domain name to query and do modified query.
     If names too large, just do gethostname on the
     original string.
  */

  if((strlen(name) + strlen(domain)) >= sizeof(query))
    return(gethostbyname(name));

  sprintf(query, "%s%s", name, domain);
  return(gethostbyname(query));
#else /* REDUCE_ROOT_DNS_LOOKUPS */
  return(gethostbyname(name));
#endif /* REDUCE_ROOT_DNS_LOOKUPS */
}

