#include <assert.h>
#include <ctype.h>
#include <endian.h>
#include <getopt.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "common.h"
#include "hash.h"

#define HBITS		12			 /* good performance for up to approx. 4096 entries */
#define HSIZE		(1 << HBITS)
#define MOD_HSIZE(a)	((a) & (HSIZE - 1))

typedef struct Entry {
  char                     name[64];
  char                     password[9];
  byte                     ip_len;
  byte                     ip[8];
  time_t                   expire;
} Entry;

int                      daemonInit();
int                      serverListen();
int                      serverAccept();
void                     serviceRequest(FILE * f);
int                      getToken(char *d, char *s, int *idx);
void                     entryPurge();
word                     nameHash(word size, char *s);
void                     strToUpper(char *s);
void                     perrQuit(char *name);

const char              *msgOK = "OK";
const char              *msgCommand = "Unknown Command.";
const char              *msgSyntax = "Syntax error.";
const char              *msgNoEntry = "Entry not found.";
const char              *msgPass = "Incorrect password.";

HashTable                hTable;

void
main(int argc, char **argv)
{
  int                      listenFD,
                           clientFD,
                           i;
  fd_set                   allSet,
                           rSet;
  FILE                    *clientFile;
  time_t                   nextPurge = 0;

  daemonInit();
  hashTableInit(&hTable, HSIZE, (HashFn) nameHash, (CompFn) strcmp, 0, TRUE);
  while ((i = getopt(argc, argv, "p")) != EOF)
    if (optind < argc)
      port = atoi(argv[optind++]);
  listenFD = serverListen();
  if (listenFD < 0)
    exit(1);
  FD_ZERO(&allSet);
  FD_SET(listenFD, &allSet);
  for (;;) {
    rSet = allSet;
    if (select(listenFD + 1, &rSet, NULL, NULL, NULL) < 0)
      exit(1);
    if (time(NULL) >= nextPurge) {
      entryPurge();
      nextPurge = time(NULL) + 60 * 60;
    }
    if ((clientFD = serverAccept(listenFD)) < 0)
      continue;
    clientFile = fdopen(clientFD, "r+");
    serviceRequest(clientFile);
  }
}

int
daemonInit()
{
  if (fork())
    exit(0);
  setsid();
  chdir("/");
  umask(0);
  return 0;
}

int
serverListen()
{
  int                      fd;
  struct sockaddr_in       addr;
  char                     hostname[256];
  struct hostent          *hP;

  if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    perrQuit("socket");
  bzero(&addr, sizeof(struct sockaddr_in));
  addr.sin_family = AF_INET;
  gethostname(hostname, 256);
  hP = gethostbyname(hostname);
  memcpy(&addr.sin_addr, hP->h_addr, hP->h_length);
  addr.sin_port = htons(port);
  if (bind(fd, (struct sockaddr *) &addr, sizeof(addr)))
    perrQuit("bind");
  if (listen(fd, 5) < 0)
    perrQuit("listen");
  return fd;
}

int
serverAccept(int listenFD)
{
  int                      clientFD,
                           len;
  fd_set                   rSet;
  struct sockaddr_in       addr;
  struct timeval           tv;
  len = sizeof(struct sockaddr_in);
  if ((clientFD = accept(listenFD, (struct sockaddr *) &addr, &len)) < 0)
    return -1;
  FD_ZERO(&rSet);
  FD_SET(clientFD, &rSet);
  tv.tv_sec = 5;
  tv.tv_usec = 0;
  if (select(clientFD + 1, &rSet, NULL, NULL, &tv) <= 0) {
    close(clientFD);
    return -1;
  }
  return clientFD;
}

void
serviceRequest(FILE * f)
{
  int                      i,
                           j,
                           li = 0;
  char                     line[256],
                           token[256];
  Entry                    entry,
                          *entryP;
  struct hostent          *hp;
  const char              *res = msgSyntax;

  fgets(line, 256, f);
  if (getToken(token, line, &li))
    goto done;
  if (!strcasecmp(token, "ADD")) {
    if (getToken(token, line, &li))
      goto done;
    if (strlen(token) > 8)
      goto done;
    strcpy(entry.password, token);
    if (getToken(token, line, &li))
      goto done;
    i = atoi(token);
    if (i < 1)
      i = 1;
    else if (i > 24)
      i = 24;
    entry.expire = time(NULL) + i * 60 * 60;
    if (getToken(token, line, &li))
      goto done;
    if ((hp = gethostbyname(token)) == NULL)
      goto done;
    if (hp->h_length > 8)
      goto done;
    entry.ip_len = hp->h_length;
    memcpy(entry.ip, hp->h_addr, hp->h_length);
    if (getToken(token, line, &li) || strlen(token) > 63)
      goto done;
    strcpy(entry.name, token);
    i = strlen(entry.name);
    while (!getToken(token, line, &li)) {
      j = i + 1 + strlen(token);
      if (j > 63)
	goto done;
      entry.name[i] = ' ';
      strcpy(entry.name + i + 1, token);
      i = j;
    }
    strToUpper(entry.name);
    if ((entryP = hashTableSearch(&hTable, entry.name)) != NULL) {
      if (strcmp(entry.password, entryP->password)) {
	res = msgPass;
	goto done;
      } else
	hashTableDel(&hTable, entryP->name);
    }
    entryP = (Entry *) malloc(sizeof(Entry));
    memcpy(entryP, &entry, sizeof(Entry));
    hashTableAdd(&hTable, entryP);
    res = msgOK;
  } else if (!strcasecmp(token, "REMOVE")) {
    if (getToken(token, line, &li))
      goto done;
    if (strlen(token) > 16)
      goto done;
    strcpy(entry.password, token);
    if (getToken(token, line, &li))
      goto done;
    strcpy(entry.name, token);
    i = strlen(entry.name);
    while (!getToken(token, line, &li)) {
      j = i + 1 + strlen(token);
      if (j > 63)
	goto done;
      entry.name[i] = ' ';
      strcpy(entry.name + i + 1, token);
      i = j;
    }
    strToUpper(entry.name);
    if ((entryP = hashTableSearch(&hTable, entry.name)) == NULL) {
      res = msgNoEntry;
      goto done;
    }
    if (strcmp(entry.password, entryP->password)) {
      res = msgPass;
      goto done;
    }
    hashTableDel(&hTable, entryP->name);
    res = msgOK;
  } else if (!strcasecmp(token, "SEARCH")) {
    if (getToken(token, line, &li))
      goto done;
    strcpy(entry.name, token);
    i = strlen(entry.name);
    while (!getToken(token, line, &li)) {
      j = i + 1 + strlen(token);
      if (j > 63)
	goto done;
      entry.name[i] = ' ';
      strcpy(entry.name + i + 1, token);
      i = j;
    }
    strToUpper(entry.name);
    if ((entryP = hashTableSearch(&hTable, entry.name)) == NULL) {
      res = msgNoEntry;
      goto done;
    }
    line[0] = '\0';
    hp = gethostbyaddr(entryP->ip, entryP->ip_len, AF_INET);
    if (hp)
      sprintf(line, "Hostname: %s     ", hp->h_name);
    strcat(line, "IP: ");
    sprintf(line + strlen(line), "%d", entryP->ip[0]);
    for (i = 1; i < entryP->ip_len; i++)
      sprintf(line + strlen(line), ".%d", entryP->ip[i]);
    res = line;
  } else
    res = msgCommand;
done:
  fprintf(f, "%s\n", res);
  fclose(f);
}

int
getToken(char *d, char *s, int *idx)
{
  int                      si,
                           di,
                           slen = strlen(s);

  for (si = *idx; si < slen && isspace(s[si]); si++);
  for (di = 0; si < slen && isgraph(s[si]); si++)
    d[di++] = s[si];
  d[di] = '\0';
  *idx = si;
  return di == 0;
}

void
entryPurge()
{
  time_t                   t;
  Entry                   *e;

  time(&t);
  hashTableIterateInit(&hTable);
  while ((e = hashTableIterate(&hTable)) != NULL) {
    if (e->expire <= t)
      hashTableDel(&hTable, e->name);
  }
}

word
nameHash(word size, char *s)
{
  int                      l = strlen(s);
  word                     w = (((word) s[0] & 0x0f) << 20) | (((word) s[l * (1 / 5)] & 0x0f) << 16) | (((word) s[l * (2 / 5)] & 0x0f) << 12) |
  (((word) s[l * (3 / 5)] & 0x0f) << 8) | (((word) s[l * (4 / 5)] & 0x0f) << 4) | ((word) s[l - 1] & 0x0f);
  word                     shift = (24 - HBITS) / 2;
  word                     mask = (HSIZE - 1) << shift;

  w = ((w * w) & mask) >> shift;
  return MOD_HSIZE(w);
}

void
strToUpper(char *s)
{
  int                      i,
                           l = strlen(s);

  for (i = 0; i < l; i++)
    s[i] = toupper(s[i]);
}

void
perrQuit(char *name)
{
  perror(name);
  exit(1);
}
