#include <malloc.h>
#include <string.h>
#include "hash.h"

void                     hashTableInit(HashTable * hTable, word size, HashFn hashFn,
				       CompFn compFn, word keyOffset, Boolean ownsItems);
void                     hashTableDeinit(HashTable * hTable);
void                     hashTableAdd(HashTable * hTable, void *item);
int                      hashTableDel(HashTable * hTable, void *key);
void                    *hashTableSearch(HashTable * hTable, void *key);

void
hashTableInit(HashTable * hTable, word size, HashFn hashFn,
	      CompFn compFn, word keyOffset, Boolean ownsItems)
{
  hTable->size = size;
  hTable->nItems = 0;
  hTable->table = malloc(size * sizeof(HashTableListNode *));
  bzero(hTable->table, size * sizeof(HashTableListNode *));
  hTable->hashFn = hashFn;
  hTable->compFn = compFn;
  hTable->keyOffset = keyOffset;
  hTable->ownsItems = ownsItems;
  hTable->iterationList = (word)-1;
  hTable->curNode = NULL;
}

void
hashTableDeinit(HashTable *hTable)
{
  word                     i;
  HashTableListNode       *curNode, *nextNode;

  for (i = 0; i < hTable->size; i++) {
    curNode = hTable->table[i];
    while (curNode) {
      nextNode = curNode->next;
      free(curNode);
      curNode = nextNode;
    }
  }
  hTable->size = 0;
  free(hTable->table);
}

word
hashTableNitems(HashTable *hTable)
{
  return hTable->nItems;
}

void
hashTableIterateInit(HashTable *hTable)
{
  hTable->iterationList = 0;
  hTable->curNode = hTable->table[0];
}

void *
hashTableIterate(HashTable *hTable)
{
  word                     tmpList = hTable->iterationList;
  void                    *res;

  /* skip past empty lists */
  while (hTable->iterationList < hTable->size &&
	 !hTable->table[hTable->iterationList])
    hTable->iterationList++;

  /* no more lists to iterate over? */
  if (hTable->iterationList == hTable->size) {
    hashTableIterateDeinit(hTable);
    return NULL;
  }

  /* if we moved ahead at least 1 list, reset 'curNode' */
  if (tmpList < hTable->iterationList)
    hTable->curNode = hTable->table[hTable->iterationList];

  /* found an item in the current list? */
  if (hTable->curNode) {
    res = hTable->curNode->item;
    hTable->curNode = hTable->curNode->next;
    return res;
  }

  /* reached end of current list -- make recursive call */
  hTable->iterationList++;
  return hashTableIterate(hTable);
}

void
hashTableIterateDeinit(HashTable *hTable)
{
  hTable->iterationList = (word)-1;
  hTable->curNode = NULL;
}

void
hashTableAdd(HashTable *hTable, void *item)
{
  word                     h = hTable->hashFn(hTable->size, (byte *) item + hTable->keyOffset);
  HashTableListNode       *newNode, *head = hTable->table[h];

  newNode = (HashTableListNode*) malloc(sizeof(HashTableListNode));
  newNode->item = item;
  newNode->next = head;
  hTable->table[h] = newNode;
  hTable->nItems++;
}

int
hashTableDel(HashTable *hTable, void *key)
{
  word                     h = hTable->hashFn(hTable->size, key);
  HashTableListNode       *prevNode = NULL, *curNode = hTable->table[h];

  while (curNode) {
    if (!hTable->compFn(key, (byte*)curNode->item + hTable->keyOffset)) {
      if (curNode == hTable->curNode)
	hTable->curNode = hTable->curNode->next;
      if (prevNode) prevNode->next = curNode->next;
      else hTable->table[h] = NULL;
      if (hTable->ownsItems)
	free(curNode->item);
      free(curNode);
      hTable->nItems--;
      return 0;
    }
    prevNode = curNode;
    curNode = curNode->next;
  }
  return 1;
}

void *
hashTableSearch(HashTable *hTable, void *key)
{
  word                     h = hTable->hashFn(hTable->size, key);
  HashTableListNode       *curNode;

  curNode = hTable->table[h];
  while (curNode) {
    if (!hTable->compFn(key, (byte*)curNode->item + hTable->keyOffset)) {
      hTable->curNode = curNode;
      return curNode->item;
    }
    curNode = curNode->next;
  }
  hTable->curNode = NULL;
  return NULL;
}

void *
hashTableSearchNext(HashTable *hTable, void *key)
{
  HashTableListNode       *p = hTable->curNode;

  if (p) p = p->next;
  while (p) {
    if (!hTable->compFn(key, (byte*)p->item + hTable->keyOffset)) {
      hTable->curNode = p;
      return p->item;
    }
    p = p->next;
  }
  hTable->curNode = NULL;
  return NULL;
}
