#include "fhostd.h"

/************************************************************
 *                                                          *
 *  This file contains a collection of signal handling      *
 *  routines:                                               *
 *                                                          *
 *  block_sigchld() and unblock_sigchld() block and         *
 *  unblock the SIGCHLD interrupt.                          *
 *  sighup_setup() turns on the SIGHUP signal handler       *
 *  for the daemon process. Sending SIGHUP to the daemon    *
 *  process causes fhostd() to mail and purge the host      *
 *  table. The routine that gets called is dump_table().    *
 *  Finally reap_children() sets up the SIGCHLD handling    *
 *  routine called reap_child() that waits on the zombie.   *
 *                                                          *
 ************************************************************/

void block_sigchld (void)
/*
 *	Block SIGCHLD.
 */
{
  sigset_t set;

  sigemptyset(&set);
  sigaddset(&set, SIGCHLD);
  sigprocmask(SIG_BLOCK, &set,  NULL);

  return;
}

void unblock_sigchld (void)
/*
 *	Unblock SIGCHLD.
 */
{
  sigset_t set;

  sigemptyset(&set);
  sigaddset(&set, SIGCHLD);
  sigprocmask(SIG_UNBLOCK, &set,  NULL);

  return;
}

int sighup_setup(void)
{
  sigset_t set;
  struct sigaction sact, osact;
  void *sighup (int);

  sigemptyset(&set);

  sact.sa_handler = (void *)dump_table;
  sact.sa_mask = set;
  sact.sa_flags = 0;
  sact.sa_flags |= SA_RESTART;

  sigaction (SIGHUP, &sact, &osact);

  return (0);
}

void *dump_table (int signo)
{
  int i, pipefds[2];

  switch (fork()) { /* Spawn a child to do the work */
  case -1:
    log_err (use_syslog, P_ERROR, "sighup: fork1");
    /* NOTREACHED */
  default:  /* The parent zeros out the table and returns. */
    itab  = 0;
    itabf = 0;
    return;
  case 0:;
    /* DROPTHROUGH */
  }

  /* The child continues. */

  dup(0); dup(1); dup(2);  /* Re-open standard IO */

  if (itab == 0) {
    log_err(use_syslog, SYS_LOG, "fhostd: dump_table: empty table.");
    send_mail ("Fhostd: Empty HOST TABLE!", "HOST TABLE is empty; nothing to report.");
    exit (0); /* Nothing to write so we quit */
  }

/* The child writes out its copy of the host table. */

  if (pipe(pipefds) < 0)
    log_err (use_syslog, P_ERROR, "sighup: pipe");

 /* Spawn a grandchild. */

  switch (fork()) {

  case -1:

    log_err (use_syslog, P_ERROR, "sighup: fork2");
    /* NOTREACHED */

  case 0:  /* The grandchild execs Mail and listens. */

    dup2(pipefds[0],0);
    close(pipefds[0]);
    close(pipefds[1]);

    if (signo < 0)
      execl(MPATH, "Mail","-n", "-s",
         "Fhostd(Timeout): HOST TABLE", MAIL_AD, (char *) 0);
    else
      execl(MPATH, "Mail", "-n", "-s",
	 "Fhostd(SIGHUP): HOST TABLE", MAIL_AD, (char*)0);

    log_err (use_syslog, P_ERROR, "sighup: execl(Mail)");

    /* NOTREACHED */

  default:  /* The child writes to the pipe and exits. */

    dup2(pipefds[1],1);
    dup2(pipefds[1],2);
    close(0);
    close(pipefds[0]);
    close(pipefds[1]);

    printf("HOST TABLE:\n");

    for (i=0; i<itab; i++)
      printf("%s-%s %i\n",table[i]->service,
        table[i]->name,table[i]->hits);

    exit(0);

  } /* end switch */

  /* NOTREACHED */
}

int reap_children (void)
{
  sigset_t set;
  struct sigaction sact, osact;
  void *reap_child (int);

  sigemptyset(&set);

  sact.sa_handler = (void *)reap_child;
  sact.sa_mask = set;
  sact.sa_flags = 0;
  sact.sa_flags |= SA_RESTART | SA_NOCLDSTOP;

  sigaction (SIGCHLD, &sact, &osact);

  return (0);
}

void *reap_child (int signo)
{
  waitpid (0, NULL, WNOHANG);
  return;
}
