/*
	This is the finger host server. 

	This server program collects remote host names and
	requested services from the client program. The first
	time that it receives a particular remote host-name
	from the client it spawns "safe finger" and optionally
	"nmap" towards the remote host; the results get mailed
	to a designated administrator. After a specified length
	of time, if no more events occur, the server mails out
	a table of the accumulated events to the same designated
	administrator and then, if it is not running as a daemon
	process, terminates.
	
    Copyright (C) 1996 <Ed Alley, Email: wea@llnl.gov>

*/

#include "fhostd.h"

main(argc, argv)
int argc;
char **argv;
{
	char c;
	char localhost[MAXCHARS], host[MAXCHARS], service[MAXCHARS];
	char message[MAXMESS], err_mess[512];
	int fromlen, len;
	int i, j, s, ns, nsel, nbr, cl;
	int is_daemon, use_nmap, is_r00t, attack;
	struct sockaddr_un saun;
	struct sockaddr_un fsaun;
	struct timeval timeout;
	fd_set readfds;
	extern char *optarg;
	extern int opterr, optind, optopt;

	openlog ("fhostd", LOG_CONS | LOG_NDELAY | LOG_PID, LOG_DAEMON);

	/*
	  Number of arguments is less than six.
	*/

	if (argc > 5) {
	  syslog (LOG_ERR, "Too many arguments!\n");
	  syslog (LOG_ERR, "For usage enter the command: fhost -h\n");
	  exit (1);
	}

	/*
	  Check argument lengths (must be less than MAXCHARS characters).
	*/

        attack = 0;
	for (i=0; i<argc; i++)
	  if (strlen(argv[i]) > MAXCHARS-1)
	    attack = 1;

	if (attack) {
	  syslog (LOG_ERR, "Possible buffer overrun attack! Args follow:");
	  for (i=0; i<argc; i++)
	    syslog(LOG_ERR, "arg[%d] = %s", i, argv[i]);
	  send_mail("ALERT! fhostd: Possible attack!",
	  "Possible commandline attack! See syslog messages for more info.");
	  exit (1);  /* Dont' even continue if this is a commandline attack */
	}

	/* Process the command line options. */

	is_daemon = 0;
	use_nmap  = 0;
	use_syslog = 1;	 /* Syslog is the default logging mode. */
	while ((cl = getopt(argc, argv, "dhnsv")) != -1) {
          switch (cl) {
            case 'd':		/* Run as a daemon */
	      is_daemon = 1;
	      break;
	    case 'n':		/* Run nmap against the remote host */
	      use_nmap = 1;
	      break;
	    case 's':		/* Send logs to stdout instead of syslog */
	      use_syslog = 0;
	      break;
	    case 'v':
	      printf("Fhostd version number: "VERSION"\n");
	      exit (0);
	    case 'h':
	    default:
	      print_usage (0);
          }
        }

	/*
	 Check to see if we are running as r00t.
	*/

	  if (!getuid())
	    is_r00t = 1;
	  else
	    is_r00t = 0;

	/*
	  Check to see if we're a daemon.
	*/

	if (is_daemon) {
	  daemon_init();	/* Turn into a daemon */
	  reap_children();	/* Handle death of children */
	  sighup_setup();	/* Handle SIGHUP */
	  itabf = 0;		/* Table wrap-around flag */
	  itab  = 0;		/* Table entry count */
	  use_syslog = 1;	/* Daemons always use syslog */
	  goto ldae;
	}

	/*
	  If we're not a daemon, then come here and process the input args.
	*/

	if (argc-optind < 2) {
	  syslog(LOG_ERR, "Incorrect number of arguments!\n");
	  syslog(LOG_ERR, "For usage enter the command: fhost -h\n");
	  exit (1);
	}

	if (argv[optind][0] != '@') {  /* host name begins with @ */
	  syslog (LOG_ERR, "ALERT! Remote hostname error:");
	  syslog (LOG_ERR, "hostname = %s", argv[optind]);
	  send_mail("ALERT! fhostd: Possible attack!",
	            "Incorrect hostname address. Check syslog.");
	  exit (1); /* Don't even continue. */
	}

	/*
	  Store the remote host-name and the requested
	  service at the beginning of the table.
	*/

	if ((table[0] = (table_ev*)malloc(sizeof(table_ev))) == NULL)
	  log_err (use_syslog, P_ERROR, "malloc");

	len = strlen(argv[optind]) + 1;
	if ((table[0]->name = (char*)malloc(sizeof(char)*len)) == NULL)
	  log_err (use_syslog, P_ERROR, "malloc");

	for (i=0; i<len; i++)
		table[0]->name[i] = argv[optind][i];

	len = strlen(argv[optind+1]) + 1;
	if ((table[0]->service = (char*)malloc(sizeof(char)*len)) == NULL)
	  log_err (use_syslog, P_ERROR, "malloc");

	for (i=0; i<len; i++)
		table[0]->service[i] = argv[optind+1][i];

	table[0]->hits = 1;

	/*
	  Set the table index to point to the next entry.
	  itabf is a flag to indicate loop around in the table.
	*/

	itabf = 0;
	itab  = 1;

	reap_children();	/* Handle the death of children */

	finger_host ("localhost", argv[optind]); /* Finger the remote host. */

	if (use_nmap)	/* Map the remote host if requested. */
          map_host("localhost", argv[optind], is_r00t);

 ldae:
	block_sigchld();

	/*
	  Get a UNIX domain stream socket.
	*/

	if ((s = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
	  log_err (use_syslog, P_ERROR, "socket");

	fcntl (s, F_SETFD, 1);	/* Close on exec */

	/*
	  Create the address we will be binding to by
	  copying into the sockaddr_un structure that is
	  defined in /usr/include/sys/un.h.
	*/

	bzero((char *) &saun, sizeof(saun));

	saun.sun_family = AF_UNIX;
	strcpy(saun.sun_path, ADDRESS);

	/*
	  Bind the address to the socket. We unlink the name
	  first so that the bind won't fail.

	  The third argument indicates the "length" of the
	  structure, not just the length of the socket name.
	  We can use the total length of the structure here
	  without fear (we just waste a little kernel space). ;)
	*/

	unlink(ADDRESS);

	if (bind(s, (struct sockaddr *) &saun, sizeof(saun)) < 0)
	  log_err (use_syslog, P_ERROR, "bind");

	if (listen(s, 5) < 0)
	  log_err (use_syslog, P_ERROR, "listen");

	unblock_sigchld();

    for (;;) { /* Forever loop */

	/*
	  Use select() as a fancy timer.

	  Set the select bits to correspond to our socket.

	  We allow select to be interrupted by SIGCHLD. If that
	  happens then we handle the signal and then go to lsel.
	*/

 lsel:
	block_sigchld();

	FD_ZERO (&readfds);
	FD_SET  (s, &readfds);

	/*
	  Set the time out interval.
	*/

	timeout.tv_sec  = time_out;
	timeout.tv_usec = 0L;
	len = (int) getdtablesize();

        unblock_sigchld();

	nsel = select (len, &readfds, (fd_set *) 0, (fd_set *) 0, &timeout);

	block_sigchld();

	if (nsel < 0) {
          if (errno == EINTR) goto lsel;
	  log_err (use_syslog, P_ERROR, "select");
	}

	/*
	  See if a descriptor is ready or we timed out.
	*/

	if (FD_ISSET(s, &readfds) <= 0) {

	/*
	  We timed out. If we're a daemon check to see
	  if we have any entries in the host table.
	  If we do then send the host table to mail
	  and return to select.
	*/

	  if (is_daemon && itab == 0) {
	    unblock_sigchld();
	    goto lsel;
	  }

	  dump_table (-1);

	/*
	  If a daemon, then return to select, otherwise, exit.
	*/

	  if (!is_daemon) { /* Not a daemon so bye-bye! */
	    close (s);
	    if (unlink(ADDRESS) < 0)
	      log_err (use_syslog, P_ERROR, "unlink");
	    exit (0);
          }

	  itab  = 0;  /* We're a daemon so return to select. */
	  itabf = 0;
	  unblock_sigchld();
	  goto lsel;

	} /* end if FD_ISSET */

	/*
	  Come here if we haven't timed out yet and
	  accept connections. When we accept one, ns will
	  be connected to the client.  fsaun will contain
	  the address of the client.
	*/

	fromlen = sizeof(fsaun);
	if ((ns = accept(s, (struct sockaddr *) &fsaun, &fromlen)) < 0)
	  log_err (use_syslog, P_ERROR, "accept");

	/*
	  Read the message from the client and truncate it
	  if it is longer that MAXMESS.
	*/

	if ((nbr = recv(ns, message, MAXMESS, 0)) < 0)
	  log_err (use_syslog, P_ERROR, "recv");

	close (ns);  /* We're only interested in one reading. */

	/*
	  Pad the message with nulls to prevent over-runs.
	*/

        nbr = (nbr > MAXMESS-3) ?  MAXMESS-3 : nbr;

	message[nbr++] = '\0';
        message[nbr++] = '\0';
        message[nbr]   = '\0';

	/*
	  Decompose the message into a host-name and a service.
	*/

	attack = 0;

	i=0;
	j=0;
	while ((c = message[j++]) != '\0') {
	  localhost[i++] = c;
	  if (i > MAXCHARS-1) {
	    strcpy(err_mess, "Localhost: ");
	    strcat(err_mess, localhost);
	    strcat(err_mess, ": Possible buffer overrun attack!");
	    log_err(use_syslog, SYS_LOG, err_mess);
	    attack = 1;
	    i = MAXCHARS-1;
	    break;
	  }
	}
	localhost[i] = '\0';

        i=0;
	while ((c = message[j++]) != '\0') {
	  host[i++] = c;
	  if (i > MAXCHARS-1) {
	    strcpy(err_mess, "Host: ");
	    strcat(err_mess, host);
	    strcat(err_mess, ": Possible buffer overrun attack!");
	    log_err(use_syslog, SYS_LOG, err_mess);
	    attack = 1;
	    i = MAXCHARS-1;
	    break;
	  }
	}
	host[i] = '\0';

	if (host[0] != '@') {
	  strcpy(err_mess, "Improper host address follows:\n");
	  strcat(err_mess, host);
	  log_err(use_syslog, SYS_LOG, err_mess);
	  attack = 1;
	}

	i=0;
	while ((c = message[j++]) != '\0') {
	  service[i++] = c;
	  if (i > MAXCHARS-1) {
	    strcpy(err_mess, "Service: ");
	    strcat(err_mess, service);
	    strcat(err_mess, ": Possible buffer overrun attack!");
	    log_err(use_syslog, SYS_LOG, err_mess);
	    attack = 1;
	    i = MAXCHARS-1;
	    break;
	  }
	}
	service[i] = '\0';

	/*
	   If this is an attack then send an alert through the mail
	   and don't store the information in the table.
	*/

      if (attack) {

	send_mail("ALERT! Fhostd: Possible attack!",
         "Possible attack through the socket! Check syslogs for more info.");

      } else {

	/*
	  Compare the new host-name with the table.
	  If a match is found then update the table
	  and go back to the select call.
	*/

	j = 0;
	for (i=0; i<itab; i++) {
	  if (strcmp(table[i]->name, host) == 0) {
	    j = 1;
	    if (strcmp(table[i]->service, service) == 0) {
	      ++table[i]->hits;
	      j = 2;
	      break;
	    }
	  }
	}

	/*
	  If j = 0 then we have a new host name so finger
	  the remote host and store the new name in
	  hosttable. If j = 1 then we have a new service
	  associated with the hostname so store into the
	  table.
	*/

	if (j == 0)  {              /* Finger the host if j = 0 */
          finger_host ("localhost", host);
	  if (use_nmap)             /* Also run nmap on the host if requested */
	    map_host ("localhost", host, is_r00t);
	}

	if (j == 0 || j == 1) {
	  if (itabf != 0) {
	    free((char *)table[itab]->name);
	    free((char *)table[itab]->service);
	  }
	else

	  if ((table[itab] = (table_ev*)malloc(sizeof(table_ev))) == NULL)
	    log_err (use_syslog, P_ERROR, "malloc");

	  len = strlen(host) + 1;
	  if ((table[itab]->name = (char*)malloc(sizeof(char)*len)) == NULL)
	    log_err (use_syslog, P_ERROR, "malloc");

	  for (i=0; i<len; i++)
	    table[itab]->name[i] = host[i];

	  len = strlen(service) + 1;
	  if ((table[itab]->service = (char*)malloc(sizeof(char)*len)) == NULL)
	    log_err (use_syslog, P_ERROR, "malloc");

	  for (i=0; i<len; i++)
	    table[itab]->service[i] = service[i];

	  table[itab++]->hits = 1;

	  if (itab >= MHOSTS) {
	    itab  = 0;
	    itabf = 1;
	  }
	}
     } /* end if ATTACK */

	unblock_sigchld();

    } /* End of the forever loop */
}

void print_usage (int exit_flag)
{
  printf("\n");
  printf("Usage: fhostd -d [-n]\n");
  printf("   or: fhostd [-n] [-s] @host service\n");
  printf("   or: fhostd -v\n");
  printf("   or: fhostd -h\n");
  printf("where\n");
  printf("       -d = run in daemon mode\n");
  printf("       -n = also run nmap\n");
  printf("       -s = log to stdout instead of syslog\n");
  printf("       -v = version number\n");
  printf("       -h = this help message\n");
  printf("    @host = @hostaddress\n");
  printf("  service = service name\n");

  exit (exit_flag);
}

