// DSTART
//                  sFTP - a curses-based FTP client for Linux.
// 
//                            Current version is 0.81
// 
//                  Copyright 1997-1998, Double Precision, Inc.
// 
// This program is distributed under the terms of the GNU General Public
// License. See COPYING for additional information.
// DEND
#if HAVE_CONFIG_H
#include	"autoconfig.h"
#endif
#include	"afx.h"
#include	"afxwindow.h"
#include	"interpreter.h"
#include	"ftpsystem.h"
#include	"config.h"
#include	"debug.h"
#include	<pwd.h>
#if HAVE_UNISTD_H
#include	<unistd.h>
#endif
#include	"my_dirent.h"
#include	<errno.h>
#include	<signal.h>
#include	<ctype.h>
#include	<sys/types.h>
#if HAVE_SYS_WAIT_H
#include	<sys/wait.h>
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(stat_val)	((unsigned)(stat_val)>>8)
#endif
#ifndef WIFEXITED
#define WIFEXITED(stat_val)	(((unsigned)(stat_val) & 255)) == 0)
#endif

static const char rcsid[]="$Id: interpreter.C,v 1.3 1998/06/17 04:12:30 mrsam Exp $";

/////////////////////////////////////////////////////////////////////////////
//
// Initialize the FTP interpreter.  This is invoked by an instance of the
// CFTPSystem class.  When our job is done here, we call
// CFTPSystem::Initialized

BOOL CFTPInterpreter::Initialize(CString sysid, CString sysname)
{
CString dir1, dir2, dir3;
struct passwd *pwd;

	m_sysid=sysid;
	m_sysname=sysname;

	// Ok, we read the directories where we expect to find interpreter
	// scripts.  The contents of the directories are sorted and put
	// together in the list containing names of shell script we want
	// to test.

	if ((pwd=getpwuid(getuid())) == NULL)
	{
		m_ftp->ReportError(_T("You don't have a password entry???"));
		return (FALSE);
	}

	dir1=pwd->pw_dir;
	dir1 += IDIR3;
	dir2= IDIR1;
	dir3= IDIR2;

	m_scripts.RemoveAll();

CString *pdirs[3]={&dir1, &dir2, &dir3};
int	i;

	for (i=0; i<3; i++)
		init( *pdirs[i] );
	TestNextScript();
	return (TRUE);
}

void CFTPInterpreter::init(CString filename)
{
CStringList scripts;
DIR	*dirp=opendir(filename);

	if (!dirp)	return;
	try
	{
	struct dirent *dire;

		while ((dire=readdir(dirp)) != 0)
		{

		// We want only files that start with 'sftp'

			if (_tcsncmp(dire->d_name, SHELLPREFIX,
				sizeof(SHELLPREFIX)-1))
				continue;

		// Put me in alphabetical order.

		POSITION p=scripts.GetHeadPosition();
		POSITION lastp;

			while ((lastp=p) != 0)
			{
				if (_tcscmp(scripts.GetNext(p),
					dire->d_name) > 0)
					break;
			}
			if (lastp)
				scripts.InsertBefore(lastp,
					dire->d_name);
			else
				scripts.AddTail(dire->d_name);
		}
		closedir(dirp);
	}
	catch (...)
	{
		closedir(dirp);
		throw;
	}
POSITION p;

	for (p=scripts.GetHeadPosition(); p; )
		m_scripts.AddTail(filename + '/' + scripts.GetNext(p));
}

// Run the next shell script to see if it will accept this SYST

void CFTPInterpreter::TestNextScript()
{
	if (m_scripts.IsEmpty())	// Didn't find one!
	{
		m_ftp->Error("Unable to locate a default script.  Please make sure that a default");
		m_ftp->Error((CString)"script is installed in " + IDIR1 + " or " + IDIR2 + ".");
		return;
	}

	m_script=m_scripts.RemoveHead();

CStringArray	args;

	args.SetSize(1);
	args[0]= _T("SYST");
	if (!Fork(args, &ReadEcho, &ErrorEcho, &WriteEOF, &CheckTestScript))
		CheckTestScript(-1);
	m_boolInitializing=TRUE;
}

// We get here after the shell script returns.  Check the exit code.
// If it's zero, we'll take it.

void	CFTPInterpreter::CheckTestScript(int rc)
{
	if (rc == 0)
	{
		m_scripts.RemoveAll();

		m_ftp->m_log->AddLine("[Using " + m_script + "]");

	CStringArray args;

		args.SetSize(1);
		args[0]= _T("LIST");
		if (!Fork(args, &ReadDir, &ErrorEcho, &WriteEOF, &Initialized))
			Initialized(-1);
		return;
	}
	TestNextScript();
}

void	CFTPInterpreter::ReadDir(CString string)
{
	m_readBuf += string;
}

void	CFTPInterpreter::Initialized(int rc)
{
	if (rc)
	{
		ShellScriptFailure(rc);
		return;
	}
	if (m_readBuf.GetLength() == 0)
	{
		ShellScriptUnexpected();
		return;
	}

int	l=m_readBuf.Find('\n');

	if (l < 0)
		m_strDir=m_readBuf;
	else
		m_strDir=m_readBuf.Left(l);

	m_ftp->m_log->AddLine(_T("The directory command is ") + m_strDir
			+ _T("."));
	m_ftp->Initialized(TRUE);
}

void (CFTPInterpreter::* CFTPInterpreter::FilterDirectory(
		void (CFTPSystem::*writeDirFunc)(CString),
			CIoh *ReadDirPtr,
		void (CFTPSystem::*completedFunc)(int))
	)(CString)
{
	m_writeDirFunc=writeDirFunc;
	m_ReadDirPtr=ReadDirPtr;
	m_completedFunc=completedFunc;

CStringArray args;

	args.SetSize(1);
	args[0]= _T("DIR");
	if (!Fork(args, &ReadFmtDir, &ErrorEcho, &WriteRawDir, &FormatFinished))
	{
		(m_ftp->*m_completedFunc)(-1);
		return (&SaveRawDirToWrite);
	}

	// Ok, we need to change a few things that Fork did, and do a few
	// extra things:

	// 1.  Start reading from the FTP server, with timeout

	m_ReadDirPtr->ClearRead(TRUE);
	m_ReadDirPtr->SetTimeout(FTP_TIMEOUT);

	// 2.  Stop writing to the shell script (unformatted directory)
	//     However, set the timeout function

	m_shwrite.ClearWrite(FALSE);
	m_shwrite.CancelTimeout();
	m_shwrite.Timeout(&readTimeout);

	// 3.  Start reading from the shell script (formatted directory)
	//     without a timeout.

	m_shread.ClearRead(TRUE);
	m_shread.CancelTimeout();
	return (&SaveRawDirToWrite);
}

// This callback is called when formatted directory is read from the shell
// script.

void CFTPInterpreter::ReadFmtDir(CString str)
{
	if (m_shwrite.fd() < 0)	// Finished writing to shell script, must
				// use a timeout now.
		m_shread.SetTimeout(SHELL_TIMEOUT);
	(m_ftp->*m_writeDirFunc)(str);
}

// CFTPSystem calls this function when it read raw directory contents from
// FTP server.  The contents are an empty string if the remote FTP server
// connection has been closed.

void CFTPInterpreter::SaveRawDirToWrite(CString str)
{
	m_rawdirstring=str;	// Save the raw data

	// Stop reading from the FTP server.

	m_ReadDirPtr->ClearRead(FALSE);
	m_ReadDirPtr->CancelTimeout();

	// Start writing to the shell script

	m_shwrite.ClearWrite();
	m_shwrite.SetTimeout(SHELL_TIMEOUT);
}

// We have something to write to the shell script:

BOOL CFTPInterpreter::WriteRawDir(CString &str)
{
	// After we're done, stop writing to the shell script

	m_shwrite.CancelTimeout();
	m_shwrite.ClearWrite(FALSE);

	// Start reading from the FTP server again.

	m_ReadDirPtr->ClearRead(TRUE);
	m_ReadDirPtr->SetTimeout(FTP_TIMEOUT);

	if (m_rawdirstring.GetLength() == 0)	// End of raw data
	{
		// Start timing out on the shell script, it should now
		// start returning data.

		m_shread.SetTimeout(SHELL_TIMEOUT);
		return (FALSE);	// End of listing
	}
	str=m_rawdirstring;
	return (TRUE);
}

void CFTPInterpreter::FormatFinished(int rc)
{
	(m_ftp->*m_completedFunc)(rc);
}

//////////////////////////////////////////////////////////////////////////
//
// Try to interpret byte count

void CFTPInterpreter::InterpretByteCount(
			void (CFTPSystem::*byteCountFunc)(long),
			CString message)
{
TRACE_FUNCTION("CFTPInterpreter::InterpretByteCount")

	m_byteCountFunc=byteCountFunc;
	m_byteCountMsg=message;
	m_byteCountNum=_T("");

CStringArray args;

	args.SetSize(1);
	args[0]= _T("COUNT");
	if (!Fork(args, &SaveByteCount, &ErrorEcho, &ReadByteCount,
			&ProcessByteCount))
	{
		ProcessByteCount(0);
		return;
	}
}

void CFTPInterpreter::SaveByteCount(CString msg)
{
	m_byteCountNum += msg;
}

BOOL CFTPInterpreter::ReadByteCount(CString &str)
{
	if (m_byteCountMsg.GetLength() == 0)	return (FALSE);
	str=m_byteCountMsg;
	m_byteCountMsg=_T("");
	return (TRUE);
}

void CFTPInterpreter::ProcessByteCount(int)
{
TRACE_FUNCTION("CFTPInterpreter::InterpretByteCount")
long	n=0;
LPCTSTR	p=m_byteCountNum;

	while (isdigit(*p))
		n=n*10 + (*p++ - '0');

	(m_ftp->*m_byteCountFunc)(n);
}

//////////////////////////////////////////////////////////////////////////
//
// Compose name of the subdirectory

void CFTPInterpreter::ComposeDirName(CString dir, CString subdir,
			void (CFTPSystem::*newdirFunc)(CString))
{
	m_newdirFunc=newdirFunc;
	m_byteCountNum=_T("");	// Reuse this field :-)

CStringArray	args;

	args.SetSize(3);
	args[0]= _T("CWD");
	args[1]= dir;
	args[2]= subdir;
	if (!Fork(args, &SaveByteCount, &ErrorEcho, &WriteEOF, &ComposedDirName))
	{
		ComposedDirName(-1);
	}
}

void CFTPInterpreter::ComposedDirName(int rc)
{
	if (rc)
	{
		(m_ftp->*m_newdirFunc)(_T(""));
		return;
	}

int	l;

	if ( (l=m_byteCountNum.Find('\r')) >= 0 ||
		(l=m_byteCountNum.Find('\n')) >= 0)
	{
		m_byteCountNum.GetBuffer(l);
		m_byteCountNum.ReleaseBuffer(l);	// Chop \r\n
	}

	(m_ftp->*m_newdirFunc)(m_byteCountNum);
}

//////////////////////////////////////////////////////////////////////////

void	CFTPInterpreter::ShellScriptFailure(int rc)
{
char	buf[20];

	sprintf(buf, "%d", rc);
	m_ftp->Error(
		(CString)_T("Shell script terminated with error code ")
			+ buf + _T(", closing connection."));
}

void	CFTPInterpreter::ShellScriptUnexpected()
{
	m_ftp->Error( _T("Unexpected output from shell script, closing connection."));
}

// Time out reading from the shell script.  The script is hung.  Kill it,
// then call whatever function is supposed to handle the shell return code
// with the return code of -1.

void	CFTPInterpreter::readTimeout()
{
	m_ftp->m_log->AddLine(_T("ERROR: Shell script hung, aborting."));
	Kill();
	(this->*m_shellHandler)(-1);
}

// Read from shell script's output.  Call handler function to process read data.

void	CFTPInterpreter::readShell()
{
	m_shread.SetTimeout(SHELL_TIMEOUT);

int	l=Read(m_shread.fd(), m_readshell);

	if (l < 0)	return;

	(this->*m_readHandler)(m_readshell);
	if (l == 0)
	{
		m_shread.ClearRead(FALSE);
		if (!m_sherror.CanRead())	// When both stdout and stderr are
						// drained, kill the process
			(this->*m_shellHandler)(Kill());
	}
}

void	CFTPInterpreter::ReadEcho(CString buf)
{
int	l=0;

	m_readBuf += buf;

	while (m_readBuf.GetLength() > 0 &&
		((l=m_readBuf.GetLength(), buf.GetLength()) == 0 ||
			(l=m_readBuf.Find('\n')) >= 0))
	{
	CString line=m_readBuf.Left(l);

		m_readBuf=m_readBuf.Mid(l+1);
		m_ftp->m_log->AddLine(line);
	}
}

// Read from shell script's error output.

void	CFTPInterpreter::errorShell()
{
	m_shread.SetTimeout(SHELL_TIMEOUT);

int	l=Read(m_sherror.fd(), m_errorshell);

	if (l < 0)	return;

	(this->*m_errorHandler)(m_errorshell);
	if (l == 0)
	{
		m_sherror.ClearRead(FALSE);
		if (!m_shread.CanRead())	// When both stdout and stderr are
						// drained, kill the process
			(this->*m_shellHandler)(Kill());
	}
}

void	CFTPInterpreter::ErrorEcho(CString buf)
{
int	l=0;

	m_errorBuf += buf;

	while (m_errorBuf.GetLength() > 0 &&
		((l=m_errorBuf.GetLength(), buf.GetLength()) == 0 ||
			(l=m_errorBuf.Find('\n')) >= 0))
	{
	CString line=m_errorBuf.Left(l);

		m_errorBuf=m_errorBuf.Mid(l+1);
		m_ftp->m_log->AddLine(_T("ERROR: ") + line);
	}
}

// Function to read something into a CString

int	CFTPInterpreter::Read(int fd, CString &buf)
{
int	l=256;

	l=::read(fd, buf.GetBuffer(l), l);
	if (l < 0)
	{
		buf.ReleaseBuffer(0);
		m_ftp->ErrnoError();
	}
	else buf.ReleaseBuffer(l);
	return (l);
}

// Function to write to the shell script.

void	CFTPInterpreter::writeShell()
{
	if (m_writeBuf.GetLength() == 0 &&
		!(this->*m_writeHandler)(m_writeBuf))
	{
		// EOF
		m_shwrite.ClearWrite(FALSE);
		close(m_shwrite.fd());
		m_shwrite.fd(-1);
		return;
	}

int	l= ::write(m_shwrite.fd(), (LPCTSTR)m_writeBuf, m_writeBuf.GetLength());

	if (l < 0)
	{
		m_ftp->ReportErrnoError();
		m_shwrite.ClearWrite(FALSE);
		close(m_shwrite.fd());
		m_shwrite.fd(-1);
	}
	else
		m_writeBuf=m_writeBuf.Mid(l);
}
	
BOOL	CFTPInterpreter::WriteEOF(CString &)
{
	return (FALSE);
}

////////////////////////////////////////////////////////////////////////////
//
//  This is where the shell script is forked off.  Give me the arguments to
//  the shell script, and the functions which read/write to the shell script,
//  and I'll run it for you.

BOOL	CFTPInterpreter::Fork( CStringArray &args,
		void (CFTPInterpreter::*readHandler)(CString),
		void (CFTPInterpreter::*errorHandler)(CString),
		BOOL (CFTPInterpreter::*writeHandler)(CString &),
		void (CFTPInterpreter::*shellHandler)(int) )
{
	Kill();	// Just in case.

	m_boolInitializing=FALSE;

	// Set up function pointers.

	m_shread.Read(&readShell);
	m_sherror.Read(&errorShell);
	m_shwrite.Write(&writeShell);
	m_shellHandler=shellHandler;
	m_readHandler=readHandler;
	m_errorHandler=errorHandler;
	m_writeHandler=writeHandler;

	m_writeBuf=_T("");
	m_readBuf= _T("");
	m_errorBuf= _T("");

	// Create argv vector.

CArray<char *, char *> argv;

	argv.SetSize(args.GetSize()+4);
	argv[0]=(char *)(LPCSTR)m_script;	// Broken include file.
	argv[1]=(char *)(LPCSTR)m_sysid;
	argv[2]=(char *)(LPCSTR)m_sysname;

int	i;

	for (i=0; i<args.GetSize(); i++)
		argv[i+3]=(char *)(LPCSTR)args[i];	// Broken include file.
	argv[i+3]=NULL;

	// Create three pipes to read/write to the child process.

int	readfd[2], writefd[2], errorfd[2];

	if (pipe(readfd) >= 0)
	{
		if (pipe(writefd) >= 0)
		{
			if (pipe(errorfd) >= 0)
			{
			int	cpid=fork();

				if (cpid >= 0)
				{
					if (cpid == 0)
					{
						// This is the child process

						setuid(getuid());
						setpgid(0,0);

						// Close all descriptors except
						// stdin/out/err

						CIoh::Forked();

						// Now, set my stdin/out/err
						// to the pipes I just created

						close(0);
						dup(readfd[0]);
						close(readfd[0]);
						close(readfd[1]);
						close(1);
						dup(writefd[1]);
						close(writefd[0]);
						close(writefd[1]);
						close(2);
						dup(errorfd[1]);
						close(errorfd[0]);
						close(errorfd[1]);
						
						execv(argv[0], argv.GetData());

						// Uh oh, couldn't exec, fine,
						// write error message & bail.

					static const char emsg1[]=
						"exec of ";
					static const char emsg2[]= " failed.";

						write(2, emsg1,
							sizeof(emsg1)-1);

						write(2, argv[0], strlen(argv[0]));
						write(2, emsg2,
							sizeof(emsg2)-1);
						kill(getpid(), SIGKILL);
						exit(0);
					}

					// Process succesfully forked.
					// Wrap up and exit.

					m_pid=cpid;
					close(readfd[0]);
					m_shwrite.fd(readfd[1]);
					close(writefd[1]);
					m_shread.fd(writefd[0]);
					close(errorfd[1]);
					m_sherror.fd(errorfd[0]);

					m_shread.ClearWrite(FALSE);
					m_sherror.ClearWrite(FALSE);
					m_shread.Timeout(&readTimeout);
					m_sherror.Timeout(&readTimeout);
					m_shread.SetTimeout(SHELL_TIMEOUT);
					return (TRUE);
				}

// Uh, oh...  Error.  Clean up whatever's been partially built, produce error
// message.

				m_ftp->ErrnoError();
				close(errorfd[0]);
				close(errorfd[1]);
			}
			else	m_ftp->ErrnoError();
			close(writefd[0]);
			close(writefd[1]);
		}
		else	m_ftp->ErrnoError();
		close(readfd[0]);
		close(readfd[1]);
	}
	else	m_ftp->ErrnoError();
	return (FALSE);
}

//	Kill the running shell process.
//	This is not asynchronous, but I don't care here, something is
//	terribly wrong if the shell script hangs.

static int wait4pid(pid_t pid, pid_t &waitpid)
{
int	waitstat;

	while ((waitpid=wait( &waitstat)) != pid)
		if (waitpid == -1 && errno != EINTR)
			break;

	return (waitstat);
}

int	CFTPInterpreter::Kill()
{
pid_t	m_waitpid;

	if (m_pid < 0)	return (-1);	// Already killed

int	fd=m_shwrite.fd();

	if (fd >= 0)	close(fd);
	m_shwrite.fd(-1);

	fd=m_sherror.fd();
	if (fd >= 0)	close(fd);
	m_sherror.fd(-1);

BOOL	rc=KillReadEOF();

	if (!rc && !m_boolDestructing)	m_ftp->m_log->AddLine(
		_T("ERROR: Shell script refuses to die, I'll have to kill it.")
		);

	kill(m_pid, SIGTERM);

	if (!rc)	rc=KillReadEOF();

	if (!rc && !m_boolDestructing)	m_ftp->m_log->AddLine(
		_T("ERROR: Shell script still refuses to die, I'll have to ")
			_T("kill it for sure."));
	kill(m_pid, SIGKILL);

int waitstat=wait4pid(m_pid, m_waitpid);

	return (Kill2(waitstat, m_waitpid));
}

int	CFTPInterpreter::Kill2(int waitstat, pid_t m_waitpid)
{
int	fd;

	try
	{
		fd=m_shread.fd();
		if (fd >= 0)	close(fd);
		m_shread.fd(-1);
	}
	catch (...)
	{
		m_pid= -1;
		throw;
	}

	if (m_waitpid == m_pid && WIFEXITED(waitstat))
	{
		m_pid= -1;

	int	rc=WEXITSTATUS(waitstat);

		if (rc && !m_boolDestructing && !m_boolInitializing)
		{
		char	buf[40];

			sprintf(buf, "%d", rc);
			m_ftp->ReportError(
				(CString)_T("[Shell script terminated with exit code ")
					+ buf + _T("]"));
		}
		return (rc);
	}
	m_pid= -1;
	if (!m_boolDestructing)
		m_ftp->ReportError(  (CString)_T("[Shell script terminated with a signal]"));
	return (-1);
}

//	Try to read EOF from the READ handler.
//	This may called in the destructor, so we can't use CIoh::Select

BOOL	CFTPInterpreter::KillReadEOF()
{
time_t	tmout;
int	fd=m_shread.fd();

	if (fd < 0)	return (TRUE);

	if (!m_boolDestructing)
		CCWindow::Flush();
	time(&tmout);
	tmout += 2;

	for (;;)
	{
	time_t	t;
	fd_set	rset;
	struct	timeval	tv;

		time(&t);
		tv.tv_sec=0;
		tv.tv_usec=0;
		if (t < tmout)
			tv.tv_sec=tmout-t;

		FD_ZERO(&rset);
		FD_SET(fd, &rset);

		if (::select(fd+1, &rset, NULL, NULL, &tv) >= 0 &&
			FD_ISSET(fd, &rset))
		{
		char	buffer[256];

			if (::read(fd, buffer, sizeof(buffer)) == 0)
				return (TRUE);
		}
		if (t >= tmout)
			break;
	}
	return (FALSE);
}

