// DSTART
//                  sFTP - a curses-based FTP client for Linux.
// 
//                            Current version is 0.81
// 
//                  Copyright 1997-1998, Double Precision, Inc.
// 
// This program is distributed under the terms of the GNU General Public
// License. See COPYING for additional information.
// DEND
#if HAVE_CONFIG_H
#include	"autoconfig.h"
#endif
#include	"ftpsystem.h"
#include	"afxwindow.h"
#include	"filetype.h"
#ifdef	clear
#undef	clear
	// Curses
#endif
#include	"filelist.h"
#include	"config.h"
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<sys/socket.h>
#include	<netinet/in.h>
#include	<arpa/inet.h>
#include	<netdb.h>
#include	<string.h>
#include	<stdlib.h>
#include	<errno.h>
#if	HAVE_FCNTL_H
#include	<fcntl.h>
#endif
#include	<ctype.h>
#include	<pwd.h>

#include	"debug.h"

static const char rcsid[]="$Id: ftpsystem.C,v 1.7 1998/06/16 01:20:56 mrsam Exp $";

//#undef	FTP_TIMEOUT
//#define	FTP_TIMEOUT	1

//////////////////////////////////////////////////////////////////////////////
CFTPSystem::CFTPSystem() :
		m_local_system(NULL), m_filetypeptr(NULL),
		m_busy(FALSE), m_control_port(FTP_CONTROL_PORT_DEFAULT),
		m_log(NULL), m_doQuit(FALSE)
{
	m_control=this;
	m_data=this;
	m_control.Timeout( &ErrorTimeout );
	m_data.Timeout( &ErrorTimeout );
	m_interpreter=this;
	m_pwddialog=this;
	m_userdialog=this;
	m_followSymLinks=FALSE;
	NotBusy();
}

void CFTPSystem::Connect(CString host, CString port, CString uid,
	CString pwd, CString acct, CString dir)
{
TRACE_FUNCTION("CFTPSystem::Connect")

	if (Busy())	return;

	m_doQuit=FALSE;
	NotBusy();	// Temporarily

	if (m_control.fd() >= 0)
	{
		m_log->AddLine(_T("Already logged on to remote system, close existing connection first."));
		NotBusy();
		return;
	}
	if (m_data.fd() >= 0)
	{
		close(m_data.fd());
		m_data.fd(-1);
	}
	m_control_port=(unsigned short)atoi(port);
	m_data_port=m_control_port-1;
	m_host=host;
	m_uid=uid;
	m_pwd=pwd;
	m_acct=acct;
	m_curdir=dir;

	m_connect_addr.s_addr=inet_addr(m_host);

	if (m_connect_addr.s_addr == INADDR_NONE)
	{
		m_log->AddLine(_T("Looking up ") + m_host + _T("..."));
		m_log->Flush();

	struct hostent *hptr=gethostbyname(m_host);

		if (hptr == NULL)
		{
			m_log->AddLine(_T("Unable to resolve address for ")
					+ m_host);
			NotBusy();
			return;
		}
		m_connect_addr=*(struct in_addr *)hptr->h_addr;
	}
	m_log->AddLine((CString)_T("Connecting to ") + inet_ntoa(m_connect_addr)			+ _T(", port ") + port + _T("..."));

int	fd=socket(AF_INET, SOCK_STREAM, 0);

	if (fd < 0)
	{
		ErrnoError();
		return;
	}

	m_control.fd(fd);

struct	sockaddr_in sin;

	sin.sin_family=AF_INET;
	sin.sin_port=htons(m_control_port);
	sin.sin_addr=m_connect_addr;

int	fflags;

	if ( (fflags=fcntl(fd, F_GETFL, 0)) == -1 ||
		fcntl(fd, F_SETFL, fflags | O_NONBLOCK) < 0 ||
		(connect(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0
			&& errno != EINPROGRESS)
	)
	{
		close(fd);
		m_control.fd(-1);
		ErrnoError();
		return;
	}

	m_control.Write(&Connected);
	m_control.SetTimeout(FTP_TIMEOUT);
	Busy();
}

//  Task completed - cancel any timeouts.  NotBusy() may also be called
//  when task terminated abnormally, so close any status windows.
//  If user requested a quit action, we can do it now.

void	CFTPSystem::NotBusy()
{
	m_interpreter.Abort();	// Abort any running shell script

	m_control.CancelTimeout();
	if (m_data.fd() >= 0)
	{
		close(m_data.fd());
		m_data.fd(-1);
	}
	m_status.Destroy();
	m_pwddialog.Destroy();
	m_userdialog.Destroy();
	if (m_transferFile.is_open())
	{
		m_transferFile.close();
		TRACE(1, "Closed file." << endl);
	}
	m_viewList.RemoveAll();

	if (m_doQuit)	// User wanted to quit
	{
		// Be nice, if we are connected, send the quit command.
		if (IsConnected())
		{
			Busy();
			SendQuit();
		}
		else
			CCWindow::QuitWindows();
	}
	if (m_busy)
	{
		m_busy=FALSE;
		m_log->AddLine(_T("[Ready]"));
	}
}

void	CFTPSystem::Quit()
{
	m_doQuit=TRUE;
	if (!Busy())
		NotBusy();	// Quit immediately if we weren't busy
}

void CFTPSystem::ErrnoError()
{
TRACE_FUNCTION("CFTPSystem::ErrnoError")

	NotBusy();
	m_fwindow->OpenDir(_T(""));
	m_fwindow->CloseDir();

	if (errno)
		ReportErrnoError();
	if (m_control.fd() >= 0)
	{
		close(m_control.fd());
		m_control.fd(-1);
	}
	if (m_data.fd() >= 0)
	{
		close(m_data.fd());
		m_data.fd(-1);
	}
}

void CFTPSystem::ReportErrnoError()
{
	ReportError(strerror(errno));
}

void CFTPSystem::ReportError(CString emsg)
{
	m_log->AddLine((CString)_T("ERROR: ") + emsg);
}

void CFTPSystem::ErrorTimeout()
{
	Error(_T("FTP Server timeout, connection closed."));
}

void CFTPSystem::Error(CString what)
{
	m_log->AddLine( _T("ERROR: ") + what );
	errno=0;
	ErrnoError();
}

void CFTPSystem::Connected()
{
TRACE_FUNCTION("CFTPSystem::Connected")

	m_control.ClearWrite(FALSE);
	m_control.ClearRead();
	m_log->AddLine(_T("Connected."));
//	m_control.m_pWrite= (void (CFTPSystem::*)()) NULL;
	m_control.Write((void (CFTPSystem::*)()) NULL);

	ReadReply(&SendUser);	// Wait for initial reply, then send USER cmd
}

///////////////////////////////////////////////////////////////////////////////
//	Begin reading FTP server reply.
//
//      ReadReply sets things up to process server replies in the normal way.
//      ReadReplyRetrieve does mainly the same thing, except the server
//      replies will be filtered through the shell script, looking for the
//      message containing the size of the file that is being retrieved.
//
//	The mechanics of this may seem confusing, but it's real simple.
//	There are two entry points:  ReadReply - which is used to process
//	replies in the 'usual fashion', and ReadReplyRetrieve - which is used
//	to process the replies to the RETR command.  The reason there is a
//	difference is that the replies from the RETR command must be filtered
//	for indications of the size of the file being transferred, so the
//	status display is updatable.
//
//	ReadAndProcessControlLine is used to read from the control socket in
//	either case.  ReadAndProcessControlLine uses method pointers to hand
//	off data to various functions to process, herein lies the difference
//	between ReadReply and ReadReplyRetrieve processing.  The functions
//	are set up as follows:
//
//	m_processlinehandler - function which will find data read from the
//	socket in m_ctrlbuf.  The data may not be the entire line of the
//	response, or, if the server shoots multiple lines at me, and they
//	come in very fast, there may be several lines in m_ctrlbuf.  Basically,
//	at this point, we don't have much that is usefull.
//
//	ReadReply sets m_processlinehandler to ProcessControlLine,
//	ReadReplyRetrieve sets it to FilterControlLine.
//
//	Either function first calls ReceivedReplyLine() which examines m_ctrlbuf
//	and if there is at least one complete line there, it is removed and
//	placed in m_filterLine and the function returns TRUE.  If it returns
//	FALSE, a complete line has not been received, so the sockets should
//	be reset to time out for further reads.
//
//	m_readlinehandler - function which will process a line of response.
//
//	ProcessControlLine calls m_readlinehandler immediately after getting
//	the go-ahead from ReceivedReplyLine.  ReadReplyRetrieve first filters
//	the line through the shell script.  While the line is being filtered,
//	reading on the control socket is temporarily turned off until
//	the line is filtered.  When the line is filtered, m_readlinehandler
//	is called.
//
//	Initially, this function pointer is set by ReadReply and ReadReplyRetrieve
//	to the function ReadReplyCode.
//
//	ReadReplyCode checks if the line received is an extended reply code.
//	If it is, it changes m_readlinehandler to point to ReadReplyExtCode
//	and calls m_processlinehandler.  Since there may've been multiple
//	lines received at once into m_ctrlbuf, we need to restart the
//	processing logic back at either ProcessControlLine or FilterControlLine
//	to try to fetch another line of response.  Note, that since
//	m_readlinehandler now will point to ReadReplyExtCode, the next line
//	of the result from the FTP server will be processed by this function,
//	which will check if the line is still an extended reply, if it is
//	m_processlinehandler is called back.
//	When a non-extended result code is received by either one of those
//	functions, CallReplyHandler will be called.
//
//	CallReplyHandler will check if the result code is an intermediate
//	result, message number begins with a 1.  If it is, we should continue
//	to wait for the final result, and back we go to m_processlinehandler!
//
//	Well, we finally have the final result code, now what?  Well, there
//	was an argument passed to ReadReply, and ReadReplyRetrieve, way back
//	when.  The argument is a function pointer, and it was saved in
//	m_replyHandler.  So, CallReplyHandler will now call this function,
//	setting the arguments to this function to indicate whether this was
//	a positive or a negative result!
//
//	Now, wasn't that simple?  Isn't event-driven parsing fun!

void CFTPSystem::ReadReply( void (CFTPSystem::*replyHandler)(int, BOOL, CString))
{
TRACE_FUNCTION("CFTPSystem::ReadReply")

	m_replyHandler=replyHandler;
	m_control.Read(&ReadAndProcessControlLine);
	m_control.ClearRead();
	m_control.SetTimeout(FTP_TIMEOUT);
	m_processlinehandler=&ProcessControlLine;
	m_readlinehandler= &ReadReplyCode;
	m_ctrlbuf=_T("");
}

void CFTPSystem::ReadAndProcessControlLine()
{
TRACE_FUNCTION("CFTPSystem::ReadAndProcessControlLine")

	if (ReadControlLine() > 0)
		(this->*m_processlinehandler)();
}

void CFTPSystem::ReadReplyRetrieve(
		void (CFTPSystem::*replyHandler)(int, BOOL, CString))
{
TRACE_FUNCTION("CFTPSystem::ReadReplyRetrieve")

	m_replyHandler=replyHandler;
	m_control.Read(&ReadAndProcessControlLine);
	m_control.ClearRead();
	m_processlinehandler= &FilterControlLine;
	m_readlinehandler= &ReadReplyCode;
	m_ctrlbuf=_T("");
}

size_t CFTPSystem::ReadControlLine()
{
TRACE_FUNCTION("CFTPSystem::ReadControlLine")

size_t	cnt=m_ctrlbuf.GetLength();
size_t	l=256;

	l=read(m_control.fd(), m_ctrlbuf.GetBuffer(cnt+l)+cnt, l);
	TRACE(4, "read from control socket returned " << l << " bytes." << endl)

	if ((int)l < 0)
	{
		m_ctrlbuf.ReleaseBuffer(cnt);
		ErrnoError();
		return (l);
	}
	m_ctrlbuf.ReleaseBuffer(cnt+l);

	if (l == 0)
	{
		m_log->AddLine(_T("ERROR: FTP Server closed connection."));
		errno=0;
		ErrnoError();
	}
	return (l);
}

void CFTPSystem::ProcessControlLine()
{
TRACE_FUNCTION("CFTPSystem::ProcessControlLine")

	if (ReceivedReplyLine())
		(this->*m_readlinehandler)(m_filterLine);
	else
	{
		m_control.ClearRead();
		m_control.SetTimeout(FTP_TIMEOUT);
		m_data.SetTimeout(FTP_TIMEOUT);
				// In case we're transferring data, we
				// can accept activity on the data socket
	}
}

BOOL CFTPSystem::ReceivedReplyLine()
{
TRACE_FUNCTION("CFTPSystem::ReceivedReplyLine")
int	l=m_ctrlbuf.Find('\n');

	if (l < 0)
	{
		TRACE(4, "Line not fully received." << endl)
		return (FALSE);
	}

	m_filterLine=m_ctrlbuf.Left(l);

	m_ctrlbuf = m_ctrlbuf.Mid(l+1);

	// Remove a '\r', if it's there

	l=m_filterLine.Find('\r');
	if (l >= 0)
	{
		m_filterLine.GetBuffer(m_filterLine.GetLength());
		m_filterLine.ReleaseBuffer(l);
	}
	m_log->AddLine(m_filterLine);
	TRACE(4, "Received: '" << m_filterLine << "'" << endl)
	return (TRUE);
}

void CFTPSystem::FilterControlLine()
{
TRACE_FUNCTION("CFTPSystem::FilterControlLine")

	if (ReceivedReplyLine())
	{
		m_control.ClearRead(FALSE);
		m_control.CancelTimeout();
		m_interpreter.InterpretByteCount( &FilteredControlLine,
							m_filterLine);
	}
	else
	{
		m_control.ClearRead();
		m_control.SetTimeout(FTP_TIMEOUT);
		m_data.SetTimeout(FTP_TIMEOUT);
				// In case we're transferring data, we
				// can accept activity on the data socket
	}
}

void CFTPSystem::FilteredControlLine(long byteCount)
{
TRACE_FUNCTION("CFTPSystem::FilteredControlLine")

	m_control.ClearRead();
	m_control.SetTimeout(FTP_TIMEOUT);
	m_data.SetTimeout(FTP_TIMEOUT);
	if (byteCount)
	{
		m_status.SetSize(byteCount);
	}
	(this->*m_readlinehandler)(m_filterLine);
}

////////////////////////////////////////////////////////////////////////////
//
//  The first line of response is received from the FTP server
//  Determine if this a single line, or a multiline reponse.

void CFTPSystem::ReadReplyCode(CString newline)
{
TRACE_FUNCTION("CFTPSystem::ReadReplyCode")

	if (newline.GetLength() >= 3)
	{
	LPCSTR p=newline;

		if (isdigit(p[0]) && isdigit(p[1]) && isdigit(p[2]))
		{
			if (p[3] == '-')	// Extended reply
			{
				m_extreplycode=newline.Left(3);
				m_readlinehandler= &ReadExtReply;
				(this->*m_processlinehandler)();
				return;
			}
			if (p[3] == ' ')
			{
				CallReplyHandler(newline);
				return;
			}
		}
	}
	Error(_T("Unexpected response from FTP server."));
}

void CFTPSystem::ReadExtReply(CString string)
{
TRACE_FUNCTION("CFTPSystem::ReadExtReply")

	if (_tcsncmp(string, m_extreplycode, 3) == 0 &&
		((LPCSTR)string)[3] == ' ')
	{
		CallReplyHandler(string);
	}
	else	
		(this->*m_processlinehandler)();
}

void CFTPSystem::CallReplyHandler(CString string)
{
TRACE_FUNCTION("CFTPSystem::CallReplyHandler")

int	n=0;
LPCTSTR p=string;

	while (isdigit(*p))
		n=n*10 + (*p++-'0');

	if (*string == '1')	// Preliminary reply
	{
		(this->*m_processlinehandler)();
				// Try to process another line.
		return;
	}

	m_control.ClearRead(FALSE);		// Stop reading.
	m_control.SetTimeout(FTP_TIMEOUT);	// Maybe we're writing
	(this->*m_replyHandler)(n, *string == '2' || *string == '3', string);
}

void CFTPSystem::WriteLine(CString line)
{
TRACE_FUNCTION("CFTPSystem::WriteLine")

	TRACE(4, "Writing " << line << endl)

	m_log->AddLine(line);
	Write(line + _T("\r\n"));
}

void CFTPSystem::Write(CString line)
{
	m_writeBuffer=line;
	m_writePtr=0;
//	m_control.m_pWrite= &WriteHandler;
	m_control.Write(&WriteHandler);
	m_control.ClearWrite();
	m_control.SetTimeout(FTP_TIMEOUT);
}

void CFTPSystem::WriteHandler()
{
TRACE_FUNCTION("CFTPSystem::WriteHandler")

int	cnt=m_writeBuffer.GetLength();

	if (cnt <= m_writePtr)
	{
		TRACE(4, "Nothing more to write." << endl)

		m_control.ClearWrite(FALSE);
		m_control.SetTimeout(FTP_TIMEOUT);
		return;	// Done writing
	}

	cnt=write(m_control.fd(), ((LPCSTR)m_writeBuffer) + m_writePtr,
			cnt - m_writePtr);

	TRACE(4, "Wrote " << cnt << " bytes." << endl)
	if (cnt < 0)
	{
		ErrnoError();
		return;
	}
	m_writePtr += cnt;
	m_control.SetTimeout(FTP_TIMEOUT);
}

void CFTPSystem::SendUser(int n, BOOL good, CString str)
{
	if (!good)
	{
		QuitSent(n, good, str);
		return;
	}
	if (m_uid.GetLength() == 0)
	{
		m_userdialog.Create();
		return;
	}
	SendUser();
}

void CFTPSystem::SendUser()
{
	WriteLine(_T("USER ") + m_uid);
	ReadReply(&SendPass);
}

void CFTPSystem::SendPass(int n, BOOL good, CString str)
{
	if (!good)
	{
		QuitSent(n, good, str);
		return;
	}
	if (*(LPCTSTR)str != '3')	// Server does not want a password???
	{
		SendAcct(n, good, str);
		return;
	}

	if (m_pwd.GetLength() == 0)
	{
		m_pwddialog.Create();
		return;
	}
	SendPass();
}

void CFTPSystem::SendPass()
{
	WriteLine(_T("PASS ****"));
	m_writeBuffer=_T("PASS ") + m_pwd + _T("\r\n");
	ReadReply(&SendAcct);
}

CString CFTPSystem::AnonPassword()
{
CString	buf;
struct passwd *pwd;

	if (gethostname(buf.GetBuffer(128), 127) < 0)
		AfxThrowInternalException();
	buf.ReleaseBuffer();

	if ((pwd=getpwuid(getuid())) == NULL)
		AfxThrowInternalException();

	return (pwd->pw_name + ('@' + buf));
}

void CFTPSystem::SendAcct(int n, BOOL good, CString str)
{
	if (!good)
	{
		QuitSent(n, good, str);
		return;
	}

	if (m_acct.GetLength() == 0)
	{
		SendSyst(n, good, str);
		return;
	}
	WriteLine(_T("ACCT ") + m_acct);
	ReadReply(&SendSyst);
}

void CFTPSystem::SendSyst(int n, BOOL good, CString str)
{
	if (!good)
	{
		QuitSent(n, good, str);
		return;
	}

	WriteLine(_T("SYST"));
	ReadReply(&ProcessSyst);
}

void CFTPSystem::ProcessSyst(int, BOOL good, CString str)
{
BOOL	rc;

	if (!good)
		rc=m_interpreter.Initialize(_T("UNKNOWN"), m_host);
	else
	{
	CStringArray tokens=CStringTok(str, _T(" "));

		if (tokens.GetSize() < 2)
			rc=m_interpreter.Initialize(_T("UNKNOWN"), m_host);
		else
			rc=m_interpreter.Initialize(tokens[1], m_host);
	}
}

void CFTPSystem::Initialized(BOOL flag)
{
	if (!flag)
	{
		SendQuit();
		return;
	}
	if (m_curdir == _T(""))
		CFTPSystem::DoReadCurrentDir();	// No directory specified by user
	else
	{
		DoChangeDir(m_curdir);
	}
}

BOOL CFTPSystem::MakeDir(CString str)
{
	if (!IsConnected() || Busy())	return (FALSE);
	DoMakeDir(str);
}

void CFTPSystem::DoMakeDir(CString str)
{
	m_transferDir=str;
	WriteLine(_T("MKD ") + str);
	ReadReply(&ProcessedMKD);
	m_treeTransfer=FALSE;
}

void CFTPSystem::ProcessedMKD(int, BOOL good, CString)
{
	if (!good && !m_treeTransfer)	// Ignore error, because the directory
					// may already exist, now if we get
					// an error changing into the directory,
					// then we'd be in trouble!
	{
		NotBusy();
		return;
	}
	if (m_treeTransfer)	// We should change into the directory
				// that was just created
	{
		DoChangeDir(m_transferDir);
		m_treeTransfer=TRUE;	// Keep the flag going
		return;
	}
	DoReadCurrentDir();
}

BOOL CFTPSystem::RemoveDir(CString str)
{
	if (!IsConnected() || Busy())	return (FALSE);
	WriteLine(_T("RMD ") + str);
	ReadReply(&ProcessedCWD);
	m_treeTransfer=FALSE;
	return (TRUE);
}

BOOL CFTPSystem::Rename(CString old, CString newf)
{
	if (!IsConnected() || Busy())	return (FALSE);

	m_dataFilename=newf;	// "Borrow" this field to store the new filename
	WriteLine(_T("RNFR ") + old);
	ReadReply(&ProcessedRNFR);
	m_treeTransfer=FALSE;
	return (TRUE);
}

void CFTPSystem::ProcessedRNFR(int, BOOL good, CString)
{
	if (!good)
	{
		NotBusy();
		return;
	}
	WriteLine(_T("RNTO ") + m_dataFilename);
	ReadReply(&ProcessedCWD);
}

BOOL CFTPSystem::Delete(CString str)
{
	if (!IsConnected() || Busy())	return (FALSE);
	WriteLine(_T("DELE ") + str);
	ReadReply(&ProcessedCWD);
	m_treeTransfer=FALSE;
	return (TRUE);
}

void CFTPSystem::ReReadCurrentDir()
{
	if (!IsConnected() || Busy())	return;
	DoReadCurrentDir();
}

void CFTPSystem::DoReadCurrentDir()
{
	WriteLine(_T("PWD"));
	ReadReply(&ProcessPWD);
}

BOOL CFTPSystem::UpFromCurrentDir()
{
	if (!IsConnected() || Busy())	return (FALSE);
	DoUpFromCurrentDir();
	return (TRUE);
}

void CFTPSystem::DoUpFromCurrentDir()
{
	m_treeTransfer=FALSE;		// Until further notice
	WriteLine(_T("CDUP"));
	ReadReply(&ProcessedCWD);
}

BOOL CFTPSystem::ChangeDir(CString dir)
{
	if (!IsConnected() || Busy())	return (FALSE);
	DoChangeDir(dir);
	return (TRUE);
}

void CFTPSystem::DoChangeDir(CString dir)
{
	WriteLine(_T("CWD "+dir));
	ReadReply(&ProcessedCWD);

	m_treeTransfer=FALSE;		// Until further notice
}

void CFTPSystem::ProcessedCWD(int, BOOL good, CString)
{
	if (!good)
	{
		NotBusy();
		return;
	}

	DoReadCurrentDir();
}

// Processed a CWD when doing a tree transfer, if we fail, we may have simply
// hit a bad symbol link, try to recover.

void CFTPSystem::ProcessedCwdTransfer(int, BOOL good, CString)
{
	if (!good)
	{
		m_transferList.RemoveHead();	// Remove DIRNAME_CURRENT ENTRY
		StartTransfer();
		return;
	}
	DoReadCurrentDir();
}

void CFTPSystem::ProcessPWD(int, BOOL good, CString msg)
{
CStringArray parse_pwd;

	if (good && (parse_pwd=CStringTok(msg, _T("\"")),
			parse_pwd.GetSize()) > 1)
	{
		m_curdir=parse_pwd[1];

		if (m_treeTransfer && m_transferMode != retrieve_file)
			// When we are storing files, there's no need to reread
			// the directory, just change the title, and restart
			// the transfer.
		{
			m_fwindow->OpenDir(m_curdir);
			m_fwindow->CloseDir();
			StartTransfer();
			return;
		}
		m_fwindow->OpenDir(m_curdir);
	}
	else
	{
		NotBusy();
		return;
	}

	m_transferMode=retrieve_file;
	m_transferType=ASCII_TRANSFER;
	SendPort(&SendDIR);
}

// When user hits enter in the FTP window, we retrieve files

void CFTPSystem::Transfer(const CList<CDirEntry, const CDirEntry &> &list)
{
	Transfer(list, retrieve_file);
}

// In the local window, we store files

void CFTPSystem::TransferStore(const CList<CDirEntry, const CDirEntry &> &list)
{
	Transfer(list, store_file);
}

void CFTPSystem::Transfer(const CList<CDirEntry, const CDirEntry &> &list,
	enum FTPTransferMode mode)
{
	if (!IsConnected() || Busy())	return;

	try
	{
		m_transferMode=mode;
		m_transferList.RemoveAll();
		m_transferList=list;
		StartTransfer();
	}
	catch (...)
	{
		NotBusy();
		throw;
	}
}

void CFTPSystem::StartTransfer()
{
TRACE_FUNCTION("CFTPSystem::StartTransfer")

	if (m_transferFile.is_open())
	{
		m_transferFile.close();
		TRACE(1, "Closed file." << endl)
	}

	m_treeTransfer=FALSE;
	m_viewFile=FALSE;
	for (;;)
	{
		// Update 'selected' status in the remote window to select
		// all files we will RETRIEVE from this directory

		if (!m_fwindow->IsTransferring()
			&& m_transferMode == retrieve_file)
		{
		POSITION	p;

			m_fwindow->Transferring();
			for (p=m_transferList.GetHeadPosition(); p ; )
			{
			CDirEntry	entry=m_transferList.GetNext(p);

				if (entry.m_isdirectory &&
					(entry.m_name == DIRNAME_PARENT ||
					entry.m_name == DIRNAME_CURRENT)
					)
					break;

				m_fwindow->SelectEntry(entry.m_name, TRUE);
			}
		}

		if (m_transferList.IsEmpty())
		{
			NotBusy();
			if (m_local_system)
				m_local_system->ReReadCurrentDir();

			if (m_transferMode != retrieve_file)
				// If we stored files, time to reread directory
			{
				ReReadCurrentDir();
			}
			return;
		}

		if (m_transferList.GetHead().m_isdirectory)
		{
		CDirEntry m_entry=m_transferList.RemoveHead();
		CString &m_dirname=m_entry.m_name;

			TRACE(2, "Next transfer entry: directory "
					<< m_dirname << endl)

			if (m_dirname == DIRNAME_PARENT)
				// Time to go up a directory
			{
				if (!m_local_system->ChangeDir(m_entry.m_lclparent))
				{
					NotBusy();
					return;
				}
				DoChangeDir(m_entry.m_ftpparent);
				m_treeTransfer=TRUE;	// Restart transfer
							// after CDUP completes
				return;
			}

			if (m_dirname == DIRNAME_CURRENT)
					// Flag to reread the current directory
			{
				if (m_transferMode == retrieve_file)
					// First, check for symbolic links
				{
					if (m_followSymLinks)
						CheckSymbolicLink(m_curdir);
						// We should, pretend it's not
					else
						m_interpreter.ComposeDirName(
							m_saveftpcurdir, m_savesubdir,
							&CheckSymbolicLink);
					return;
				}

				StartTransferNewDir();
				continue;
			}

			//
			// Save current directory and subdirectory in order to
			// check for symbolic links
			//

			m_saveftpcurdir=m_curdir;
			m_savelclcurdir=m_local_system->m_curdir;
			m_savesubdir=m_dirname;

			// Let's transfer a subdirectory, shall we?

			if (m_transferMode != retrieve_file)
				// We're storing, so we must MAKEDIR here
			{
				if (!m_local_system->ChangeDir(m_dirname))
					continue;	// Probably a bad symlink

			CString m_newdir=m_local_system->m_curdir;

				if (!m_followSymLinks)
				{
				CString str=(m_savelclcurdir == _T("/") ?
					(CString)_T(""):m_savelclcurdir)
					+ _T("/") + m_dirname;

					if (str != m_newdir)
					{
						if (!m_local_system->ChangeDir(
							m_savelclcurdir))
						{
							ReportErrnoError();
							NotBusy();
							return;	// Bad
						}
						continue;
					}
				}
				// Hokay, check for symbolic links.

				DoMakeDir(m_dirname);
			}

			// Remind myself to reread the subdirectory
		CDirEntry reread;

			reread.m_isdirectory=TRUE;
			reread.m_name=DIRNAME_CURRENT;
			m_transferList.AddHead(reread);

			if (m_transferMode == retrieve_file)
			{
				WriteLine(_T("CWD ")+m_dirname);
				ReadReply(&ProcessedCwdTransfer);
			}
			m_treeTransfer=TRUE;
			return;
		}

		m_transferFileName=m_transferList.GetHead().m_name;

		TRACE(2, "Next transfer entry: file "
				<< m_transferFileName << endl)

		if (m_transferMode == retrieve_file)
		{
		int	i;

			for (i=0; i<m_transferFileName.GetLength(); i++)
				if (m_transferFileName[i] < ' '
					|| m_transferFileName[i] >= 127 ||
					m_transferFileName[i] == '/')
					m_transferFileName[i]='_';
		}

		if (!OpenTransferFile())
		{
			NotBusy();
			return;
		}
		break;
	}
	SendTYPE();
}

// We just changed into a subdirectory.  The shell script tells me
// what the name of the directory should be.  If that's not what we've got,
// we must've chdired into a symbol link, so backtrack!

void CFTPSystem::CheckSymbolicLink(CString dirname)
{
	if (dirname == _T(""))	// Error
	{
		NotBusy();
		return;
	}

	if (dirname == m_curdir)
	{
		m_local_system->MakeDir(m_savesubdir);
			// Don't check for failure, dir may exist
		if (!m_local_system->ChangeDir(m_savesubdir))
		{	// This is bad

			NotBusy();
			return;
		}
		StartTransferNewDir();
		StartTransfer();
		return;
	}

	// WOA!

	ReportError( m_savesubdir + _T(" was a symbolic link in ") + m_saveftpcurdir);
	ReportError( _T("... new directory should be ") + dirname);
	ReportError( _T("... but it is ") + m_curdir );

	m_treeTransfer=TRUE;
	WriteLine(_T("CWD ") + m_saveftpcurdir);	// This is where we were!
	ReadReply(&ProcessedCWD);
}

void CFTPSystem::StartTransferNewDir()
{
CFileList *fw=m_fwindow;

	if (m_transferMode != retrieve_file)
		fw=m_local_system->m_fwindow;
	// Grab the window we're transferring FROM into 'fw'

	fw->Transferring();
	fw->SelectAll();	// Select all files in this directory

	// First, push onto stack the marker to go up the directory hierarchy

CDirEntry gobackup;

	gobackup.m_isdirectory=TRUE;
	gobackup.m_name=DIRNAME_PARENT;
	gobackup.m_lclparent=m_savelclcurdir;
	gobackup.m_ftpparent=m_saveftpcurdir;
	m_transferList.AddHead(gobackup);

CList<CDirEntry, const CDirEntry &> dir;

	fw->GetSelectedFiles(dir);

	// Now, push onto the stack the contents of this directory

	m_transferList.AddHead(&dir);
}

BOOL CFTPSystem::OpenTransferFile()
{
	m_transferType=m_filetypeptr ?
			m_filetypeptr->TransferType(
				m_transferList.GetHead().m_name):
			IMAGE_TRANSFER;
	m_transferFile.clear(0);
	if (m_transferMode == retrieve_file)
		m_transferFile.open(m_transferFileName,
			ios::out | ios::trunc | ios::bin);
	else
		m_transferFile.open(m_transferFileName,
			ios::in | ios::bin);

	if (!m_transferFile.is_open())
	{
		ReportErrnoError();
		return (FALSE);
	}
	return (TRUE);
}

void CFTPSystem::View(CString view_filename)
{
	if (!IsConnected() || Busy())	return;

	m_treeTransfer=FALSE;
	m_transferList.RemoveAll();
	m_transferMode=retrieve_file;
	m_transferType=ASCII_TRANSFER;
	m_view.Destroy();
	m_viewFile=TRUE;
	m_viewList.RemoveAll();
	m_viewList.AddTail(_T(""));

	if (m_transferFile.is_open())
		m_transferFile.close();

CDirEntry	dummy_entry;

	dummy_entry.m_name=view_filename;
	m_transferList.AddTail(dummy_entry);
	SendTYPE();
}

void CFTPSystem::RetrieveFile(CString local_name, CString remote_name)
{
	if (!IsConnected() || Busy())	return;

	m_treeTransfer=FALSE;
	m_transferList.RemoveAll();
	m_transferMode=retrieve_file;
	m_transferFileName=local_name;
	m_viewFile=FALSE;
	if (m_transferFile.is_open())
		m_transferFile.close();

CDirEntry	dummy_entry;

	dummy_entry.m_name=remote_name;
	m_transferList.AddTail(dummy_entry);
	if (!OpenTransferFile())
	{
		NotBusy();
		return;
	}
	SendTYPE();
}

void CFTPSystem::StoreFile(CString local_name, CString remote_name)
{
	if (!IsConnected() || Busy())	return;

	m_treeTransfer=FALSE;
	m_transferList.RemoveAll();
	m_transferMode=store_file;
	m_transferFileName=local_name;
	m_viewFile=FALSE;
	if (m_transferFile.is_open())
		m_transferFile.close();

CDirEntry	dummy_entry;

	dummy_entry.m_name=remote_name;
	m_transferList.AddTail(dummy_entry);

	if (!OpenTransferFile())
	{
		NotBusy();
		return;
	}
	SendTYPE();
}

void CFTPSystem::SendTYPE()
{
	WriteLine(m_transferType == IMAGE_TRANSFER ?
			_T("TYPE I") : _T("TYPE A"));
	ReadReply(&ProcessTransferType);
}

void CFTPSystem::ProcessTransferType(int, BOOL good, CString)
{
	if (!good)
	{
		NotBusy();
		m_transferList.RemoveAll();
		if (m_local_system)
			m_local_system->ReReadCurrentDir();
		return;
	}

	if (m_transferMode == retrieve_file)
		SendPort(&SendRETR);
	else
		SendPort(&SendSTOR);
}

void CFTPSystem::SendRETR()
{
	try
	{
	CString	filename=m_transferList.RemoveHead().m_name;

		BeginDataTransfer(_T("RETR ") + filename,
			filename,
			0,
			&RetrieveComplete,
			&ReadFile,
			(BOOL (CFTPSystem::*)(CString &) )NULL, TRUE);
	}
	catch (...)
	{
		NotBusy();
	}
}

void CFTPSystem::SendSTOR()
{
	try
	{
	CString	filename=m_transferList.RemoveHead().m_name;
	long	cnt=0;

		if (m_transferFile.seekg(0, ios::end).bad() ||
			(cnt=m_transferFile.tellg()) == -1 ||
			m_transferFile.seekg(0, ios::beg).bad())
		{
			ErrnoError();
			return;
		}

		BeginDataTransfer(_T("STOR ") + filename,
			filename,
			cnt,
			&StoreComplete,
			(void (CFTPSystem::*)(CString) )NULL,
			&WriteFile,
			FALSE);
	}
	catch (...)
	{
		NotBusy();
	}
}

void CFTPSystem::RetrieveComplete()
{
	m_fwindow->SelectEntry(m_dataFilename, FALSE);
	if (m_local_system)
		m_local_system->ReReadCurrentDir();
	if (m_viewFile)
	{
		m_view=m_viewList;
		m_viewList.RemoveAll();
		m_view.Create();
	}
	StartTransfer();
}

void CFTPSystem::ViewDir()
{
	if (!IsConnected())	return;
	m_viewDir=m_dirList;
	m_viewDir.Create();
}

void CFTPSystem::StoreComplete()
{
	m_local_system->m_fwindow->SelectEntry(m_dataFilename, FALSE);
	StartTransfer();
}

void CFTPSystem::ReadFile(CString str)
{
	if (m_transferType == ASCII_TRANSFER)
	{
	size_t	l=str.GetLength(), i, j;
	LPTSTR p=str.GetBuffer(0);

		for (i=j=0; i<l; i++)
			if (p[i] != '\r')
				p[j++]=p[i];
		str.ReleaseBuffer(j);
	}
	if (str.GetLength() == 0 ||
		(m_viewFile && m_transferFile.fail()) )	return;

	if (m_viewFile)	// Save contents for viewing
	{
	LPCTSTR p=str;
	size_t	i,j,l=str.GetLength();

		for (i=j=0; i<l; i++)
		{
			if (p[i] == '\n')
			{
				if (j < i)
				{
					m_viewList.GetTail() += str.Mid(j, i-j);
				}
				m_viewList.AddTail(_T(""));
				j=i+1;
			}
		}
		if (j < l)	m_viewList.GetTail() += str.Mid(j);
		return;
	}

	m_transferFile.write( (LPCTSTR)str, str.GetLength());
	if ( m_transferFile.fail())
		ReportErrnoError();
}

BOOL CFTPSystem::WriteFile(CString &s)
{
size_t	l=256;

	m_transferFile.clear();
	m_transferFile.read(s.GetBuffer(l), l);
	if (m_transferFile.bad() || (int)(l=m_transferFile.gcount()) < 0)
	{
		ReportErrnoError();
		l=0;
	}
	s.ReleaseBuffer(l);
	if (l <= 0)
		return (FALSE);

	if (m_transferType == ASCII_TRANSFER)	// Add carriage returns
	{
	CString	c=s;
	size_t	i, cnt;
	LPCTSTR	p=c;

		for (i=0, cnt=0; i<l; i++)
			if (p[i] == '\n')	++cnt;

	LPTSTR	q=s.GetBuffer(l+cnt);

		for (i=0; i<l; i++)
		{
			if (*p == '\n')
				*q++='\r';
			*q++=*p++;
		}
		s.ReleaseBuffer(l+cnt);
	}
	return (TRUE);
}

void CFTPSystem::SendPort( void (CFTPSystem::*portHandler)())
{
int	fd;

	m_status.Create();
	m_portHandler=portHandler;
	if (m_data.fd() >= 0)
	{
		fd=m_data.fd();
		m_data.fd(-1);
		close(fd);
	}

struct	sockaddr_in sin;
socklen_t sin_sizeof;

	if ((fd=socket(AF_INET, SOCK_STREAM, 0)) < 0 ||
		(sin_sizeof=sizeof(sin), getsockname(m_control.fd(),
			(struct sockaddr *)&sin, &sin_sizeof)) < 0 ||
		(sin.sin_port=htons(0), bind(fd,
			(struct sockaddr *)&sin, sizeof(sin))) < 0 ||
		(sin_sizeof=sizeof(sin), getsockname(fd,
			(struct sockaddr *)&sin, &sin_sizeof)) < 0 ||
		listen(fd, 5) < 0)
	{
		if (fd)	close(fd);
		ReportErrnoError();
		NotBusy();
		return;
	}
	m_data.fd(fd);

char	buf[40];
int	i;

	sprintf(buf, "PORT %s,%d,%d", inet_ntoa(sin.sin_addr),
		ntohs(sin.sin_port)/256, ntohs(sin.sin_port) % 256);
	for (i=0; buf[i]; i++)
		if (buf[i] == '.')
			buf[i]=',';

	WriteLine( buf );
	ReadReply(&ProcessPort);
}

void CFTPSystem::ProcessPort(int, BOOL good, CString)
{
	if (!good)
	{
		close(m_data.fd());
		m_data.fd(-1);
		NotBusy();
		return;
	}
	(this->*m_portHandler)();
}

void CFTPSystem::SendDIR()
{
	WriteLine(_T("TYPE A"));
	ReadReply(&ProcessDIRType);
}

void CFTPSystem::ProcessDIRType(int, BOOL good, CString)
{
	if (!good)
	{
		NotBusy();
		return;
	}

	m_dirList.RemoveAll();
	m_writeDirToFilterPtr=m_interpreter.FilterDirectory(
		&SaveDirContents,
		&m_data,
		&SavedDirContents);

	BeginDataTransfer(m_interpreter.m_strDir,
		m_curdir, 0,
		&ProcessedDir, &DirReadHandler,
		(BOOL (CFTPSystem::*)(CString &) )NULL, FALSE);
}

void CFTPSystem::DirReadHandler(CString read_bytes)
{
	// First, save directory listing in a buffer

	if (m_dirList.IsEmpty())
		m_dirList.AddTail(_T(""));

	if (read_bytes.GetLength())
	{
	LPCSTR	p=read_bytes;
	size_t	n=read_bytes.GetLength();
	size_t	i=0,j;

		for (j=0; j<n; j++)
		{
			if (p[j] == '\n')
			{
			CString	l(p+i, j-i);
			int	n=l.Find('\r');

				if (n >= 0)
				{
					l.GetBuffer(0);
					l.ReleaseBuffer(n);
				}
				m_dirList.GetTail() += l;
				m_dirList.AddTail(_T(""));
				i=j+1;
			}
		}

	CString	l(p+i, n-i);
	int	o=l.Find('\r');

		if (o >= 0)
		{
			l.GetBuffer(0);
			l.ReleaseBuffer(o);
		}
		m_dirList.GetTail() += l;
	}

	if (m_writeDirToFilterPtr)
		(m_interpreter.*m_writeDirToFilterPtr)(read_bytes);
}

void CFTPSystem::ProcessedDir()
{
TRACE_FUNCTION("CFTPSystem::ProcessedDir")

	if (m_writeDirToFilterPtr)
		(m_interpreter.*m_writeDirToFilterPtr)(_T(""));
}

void CFTPSystem::SaveDirContents(CString str)
{
	m_readLineBuf += str;

int	l;
CString	line;
CDirEntry dirEntry;

	while ((l=m_readLineBuf.Find('\n')) >= 0)
	{
		line=m_readLineBuf.Left(l);
		m_readLineBuf=m_readLineBuf.Mid(l+1);

	int	l=line.GetLength();

		if (l <= 0)	continue;	// Bad!!!
		dirEntry.m_isdirectory=FALSE;
		if (l && line[l-1] == '/')
		{
			dirEntry.m_isdirectory=TRUE;
			line.GetBuffer(l);
			line.ReleaseBuffer(l-1);
		}
		dirEntry.m_name=line;
		m_fwindow->AddDirEntry(dirEntry);
	}
}

void CFTPSystem::SavedDirContents(int)
{
TRACE_FUNCTION("CFTPSystem::SavedDirContents")

	m_writeDirToFilterPtr= (void (CFTPInterpreter::*)(CString))NULL;

	if (m_data.fd() >= 0)
	{
		m_data.ClearRead(TRUE);
		m_data.SetTimeout(FTP_TIMEOUT);
		ReportError(_T("Unable to format the entire directory listing."));
		NotBusy();
		return;
	}
	TRACE(4, "Closed directory contents." << endl);
	m_fwindow->CloseDir();

	if (m_treeTransfer)	// We are xferring a directory
		StartTransfer();
	else
		NotBusy();
}

//////////////////////////////////////////////////////////////////////////
//
//  Initialize data transfer from the data socket.

void CFTPSystem::BeginDataTransfer( CString command,
			CString filename,
			long nbytes,
			void (CFTPSystem::*funcPtr)(),
			void (CFTPSystem::*ptrDataTransferRead)(CString),
			BOOL (CFTPSystem::*ptrDataTransferWrite)(CString &),
			BOOL isRetrieve)
{
	m_boolDataTransferComplete=FALSE;
	m_boolDataTransferStatusReceived=FALSE;
	m_dataFilename=filename;
	m_dataBytes=nbytes;
	m_ptrDataTransferred=funcPtr;
	m_data.Read(&AcceptDataConnection);
	m_data.SetTimeout(FTP_TIMEOUT);
	time(&m_listenTimeout);
	m_listenTimeout += FTP_TIMEOUT;
	m_ptrDataTransferRead=ptrDataTransferRead;
	m_ptrDataTransferWrite=ptrDataTransferWrite;
	WriteLine( command );
	m_status.SetEscapeHandler(this, &SendAbort);
	m_AbortSent=FALSE;

	if (isRetrieve)
		ReadReplyRetrieve(&ProcessedTransferCommand);
	else
		ReadReply(&ProcessedTransferCommand);

	m_control.SetTimeout(FTP_TIMEOUT);
}

void CFTPSystem::AcceptDataConnection()
{
time_t	t;

	time(&t);

int	tout=t > m_listenTimeout ? 0: m_listenTimeout - t;

	m_data.SetTimeout(tout);

struct	sockaddr_in sin;
socklen_t	sin_len=sizeof(sin);
int	fd=accept(m_data.fd(), (struct sockaddr *)&sin, &sin_len);

	if (fd < 0)
	{
		ErrnoError();
		return;
	}

	if (sin.sin_family != AF_INET ||
		sin.sin_addr.s_addr != m_connect_addr.s_addr ||
		ntohs(sin.sin_port) != m_data_port)
	{
	char	buf[40];

		ReportError(_T("Unexpected connection, continuing to listen."));

		sprintf(buf, "%ld", (long)ntohs(sin.sin_port));

		ReportError(CString(_T("Received connection from ")) +
				inet_ntoa(sin.sin_addr) + _T(", port ") + buf);
		sprintf(buf, "%ld", (long)m_data_port);
		ReportError(CString(_T("Expected connection from ")) +
				inet_ntoa(m_connect_addr) + _T(", port ") + buf);
		NotBusy();
		return;
	}

	close(m_data.fd());
	m_data.fd(fd);

	// Reset all timeouts, since we are still alive and kicking

	m_control.SetTimeout(FTP_TIMEOUT);
	m_data.SetTimeout(FTP_TIMEOUT);

	m_data.Read((void (CFTPSystem::*)())NULL);
	m_data.Write((void (CFTPSystem::*)())NULL);
	if (m_ptrDataTransferRead)
		m_data.Read(&ReadDataConnection);
	else
		m_data.Write(&WriteDataConnection);
	m_data.ClearRead( m_ptrDataTransferRead ? TRUE:FALSE);
	m_data.ClearWrite( m_ptrDataTransferWrite ? TRUE:FALSE);
	m_readLineBuf=_T("");
	m_writeBuf=_T("");
	m_writeBufPtr=0;
	m_log->AddLine(_T("[Accepted data connection]"));
	m_status.Start(m_dataFilename, m_dataBytes);
}

void CFTPSystem::SendAbort()
{
	m_writeBuffer += _T("ABOR\r\n");
	m_log->AddLine(_T("[ABORT command queued]"));
	m_control.ClearWrite();
	m_control.SetTimeout(FTP_TIMEOUT);
	m_data.SetTimeout(FTP_TIMEOUT);
	m_status.SetEscapeHandler(NULL, (void (CFTPSystem::*)())NULL);
	m_AbortSent=TRUE;
}

void CFTPSystem::ProcessedTransferCommand(int, BOOL good, CString
#ifdef DEBUG
	str
#endif
	)
{
TRACE_FUNCTION("CFTPSystem::ProcessTransferCommand");

	TRACE(1, "DATA TRANSFER: " << str << endl)

	m_data.SetTimeout(FTP_TIMEOUT);
	m_status.SetEscapeHandler(NULL, (void (CFTPSystem::*)())NULL);
	if (m_AbortSent)
	{
		m_transferList.RemoveAll();
		m_AbortSent=FALSE;
		ReadReply(&ProcessedTransferCommand);
					// Wait for the ABORT status to be
					// received
		return;
	}

	// An error message means we must shut down data transfer, the socket
	// may not've been established.

	m_boolDataTransferStatusReceived=TRUE;
	if (!good)
	{
		TRACE(1, "DATA TRANSFER: Status is error." << endl)
		NotBusy();
		return;
	}
	TRACE(1, "DATA TRANSFER: Status received." << endl)
	CheckTransferComplete();
}

void CFTPSystem::ReadDataConnection()
{
	// Reset all timers, since we're alive and kicking

	m_data.SetTimeout(FTP_TIMEOUT);
	m_control.SetTimeout(FTP_TIMEOUT);

int	l=256;

	l= ::read(m_data.fd(), m_readBuf.GetBuffer(l), l);
	if (l < 0)
	{
		m_readBuf.ReleaseBuffer(0);
		ErrnoError();
		return;
	}
	m_readBuf.ReleaseBuffer(l);

int	processed_cnt=l;

	if (m_transferType == ASCII_TRANSFER)	// Don't count CRs
	{
	int	i;
	LPCTSTR	p;

		for (i=0, p=m_readBuf; i<l; i++)
			if (*p++ == '\r')
				--processed_cnt;
	}
	m_status.Processed(processed_cnt);

	if (l == 0)	// EOF
	{
		TRACE(1, "DATA TRANSFER: END OF FILE" << endl)
		DataTransferComplete();
		return;
	}
	(this->*m_ptrDataTransferRead)(m_readBuf);
}

void CFTPSystem::DataTransferComplete()
{
TRACE_FUNCTION("CFTPSystem::DataTransferComplete")

	TRACE(1, "DATA TRANSFER: complete." << endl)

	m_status.Finish();
	m_control.SetTimeout(FTP_TIMEOUT);
	m_data.ClearWrite(FALSE);
	m_data.ClearRead(FALSE);
	close(m_data.fd());
	m_data.fd(-1);
	m_boolDataTransferComplete=TRUE;
	m_log->AddLine(_T("[Data connection closed]"));
	if (m_ptrDataTransferRead)
	{
		m_readBuf=_T("");
		(this->*m_ptrDataTransferRead)(m_readBuf);
	}
	CheckTransferComplete();
}

void CFTPSystem::WriteDataConnection()
{
	m_data.SetTimeout(FTP_TIMEOUT);
	m_control.SetTimeout(FTP_TIMEOUT);

size_t	n= m_writeBuf.GetLength() - m_writeBufPtr;

	if (n == 0)
	{
		if (!(this->*m_ptrDataTransferWrite)(m_writeBuf))
		{
			TRACE(1, "DATA TRANSFER: WROTE FILE" << endl)
			DataTransferComplete();
			return;
		}
		m_writeBufPtr=0;
		n=m_writeBuf.GetLength();
	}

	n= ::write(m_data.fd(), (LPCTSTR)m_writeBuf + m_writeBufPtr, n);
	if (n == (size_t)-1)
	{
		ReportErrnoError();
		TRACE(1, "DATA TRANSFER: ERROR WRITING FILE" << endl)
		DataTransferComplete();
		return;
	}
	m_writeBufPtr += n;
	m_status.Processed(n);
}

void CFTPSystem::Close()
{
	if (!IsConnected() || Busy())	return;
	try
	{
	int	fd=m_data.fd();
		m_data.fd(-1);
		if (fd >= 0)	close(fd);
		SendQuit();
	}
	catch (...)
	{
	int	fd=m_control.fd();

		m_control.fd(-1);
		if (fd >= 0)	close(fd);
		throw;
	}
}

void CFTPSystem::SendQuit()
{
	m_fwindow->OpenDir(_T(""));
	m_fwindow->CloseDir();
	WriteLine(_T("QUIT"));
	ReadReply(&QuitSent);
}

void CFTPSystem::QuitSent(int, BOOL, CString)
{
BOOL	quitflag;

	errno=0;
	quitflag=m_doQuit;
	m_doQuit=FALSE;	// This must be so, else we get confused

	ErrnoError();
	m_log->AddLine(_T("Connection closed."));
	if (quitflag)
		CCWindow::QuitWindows();
}
