// DSTART
//                  sFTP - a curses-based FTP client for Linux.
// 
//                            Current version is 0.81
// 
//                  Copyright 1997-1998, Double Precision, Inc.
// 
// This program is distributed under the terms of the GNU General Public
// License. See COPYING for additional information.
// DEND
#if HAVE_CONFIG_H
#include	"autoconfig.h"
#endif
#include	"filelist.h"
#include	"system.h"
#include	"debug.h"

static const char rcsid[]="$Id: filelist.C,v 1.3 1998/06/09 02:12:11 mrsam Exp $";

CDirEntry::CDirEntry() : m_isdirectory(FALSE)
{
}

CDirEntry::CDirEntry(const CDirEntry &o)
{
	operator=(o);
}

CDirEntry &CDirEntry::operator=(const CDirEntry &o)
{
	m_name=o.m_name;
	m_lclparent=o.m_lclparent;
	m_ftpparent=o.m_ftpparent;
	m_isdirectory=o.m_isdirectory;
	return (*this);
}

CFileList::CFileList() : m_mySystem(NULL), m_transferring(FALSE)
{
}

void CFileList::Create(int row, int col, int width, int height)
{
	m_title.Create(row, col, width, height);
	CCTextListWindow::Create(row, col, width, height);
	MoveInFrontOf(&m_title);
}

void CFileList::Destroy()
{
	CCTextListWindow::Destroy();
	m_title.Destroy();
}

void CFileList::OpenDir(CString dirname)
{
TRACE_FUNCTION("CFileList::OpenDir")

	TRACE(7, "In OpenDir" << endl);
	m_transferring=FALSE;
	m_contents.RemoveAll();
	CFileList::Redraw();

	if (dirname != _T(""))
	{
	CDirEntry new_entry;

		new_entry.m_isdirectory=TRUE;
		new_entry.m_name=DIRNAME_CURRENT;
		DoAddDirEntry(new_entry);
		new_entry.m_name=DIRNAME_PARENT;
		DoAddDirEntry(new_entry);
	}

	if (dirname.GetLength() == 0)
	{
		m_title.Title(_T(""));
		return;
	}

int	w=Width();

	m_curdir=dirname;

	w -= m_prefix.GetLength();
	if (w < 0)	w=0;

	m_title.Title(m_prefix + m_curdir.FileTitleTruncate(w));
}

void CFileList::AddDirEntry(const CDirEntry &newEntry)
{
TRACE_FUNCTION("CFileList::AddDir")

	TRACE(7, "Adding directory entry " << newEntry.m_name << endl)

	if (newEntry.m_name == _T(""))	return;
	if (newEntry.m_isdirectory &&
		(newEntry.m_name == DIRNAME_CURRENT ||
		newEntry.m_name == DIRNAME_PARENT))	return;
					// Already taken care of
	DoAddDirEntry(newEntry);
}

void CFileList::DoAddDirEntry(const CDirEntry &newEntry)
{
POSITION insert_pos;

	for (POSITION p=m_contents.GetHeadPosition();
		(insert_pos=p); )
	{
	CDirEntry cmpentry=m_contents.GetNext(p);
	int	cmp;

		// Directories should be listed first

		if (cmpentry.m_isdirectory != newEntry.m_isdirectory)
			cmp= newEntry.m_isdirectory ? -1:1;
		else
			cmp= newEntry.m_name.Compare(cmpentry.m_name);

		if (cmp < 0) break;
	}

	if (insert_pos)
		m_contents.InsertBefore(insert_pos, newEntry);
	else
		m_contents.AddTail(newEntry);
}

void CFileList::CloseDir()
{
TRACE_FUNCTION("CFileList::CloseDir")

	CFileList::Redraw();
}

void	CFileList::Redraw()
{
int	n=m_contents.GetCount();
CStringArray strs;
int	y;
POSITION p;
CString	s;

	strs.SetSize(n);

	for (y=0, p=m_contents.GetHeadPosition(); p; )
	{
	CDirEntry &dirp=m_contents.GetNext(p);

		s=dirp.m_name;
		if (dirp.m_isdirectory)
			s += _T("/");

		strs[y++]=s;
	}

	m_listStrings=strs;
	Reset();
}

void	CFileList::Refresh()
{
	CCTextListWindow::Refresh();
}

void	CFileList::Selected(size_t ndx)
{
POSITION	p=m_contents.FindIndex(ndx);
CDirEntry &dir=m_contents.GetNext(p);

	// If ENTER was hit on a directory entry, and the directory entry
	// is not manually selected, change to that directory, otherwise
	// we have some kind of a transfer to do.

	if (dir.m_isdirectory && m_mySystem)
	{
		if (dir.m_name == DIRNAME_CURRENT)
		{
			m_mySystem->ReReadCurrentDir();
			return;
		}

		if (dir.m_name == DIRNAME_PARENT)
		{
			m_mySystem->UpFromCurrentDir();
			return;
		}

		if (!IsSelected(ndx))
		{
			m_mySystem->ChangeDir(dir.m_name);
			return;
		}
	}
	SelectedFiles();
}

////////////////////////////////////////////////////////////////////////////
//
// Ok, we know we have to transfer something.  Figure out what.

void	CFileList::SelectedFiles()
{
CList<CDirEntry, const CDirEntry &>	selected_list;

	GetSelectedFiles(selected_list);

	// If there was nothing marked, transfer what the cursor is pointing
	// to.

	if (selected_list.IsEmpty())
	{
	size_t n=CurrentItem();

		if (n < m_contents.GetCount())
		{
		POSITION p=m_contents.FindIndex(n);

			selected_list.AddTail(m_contents.GetNext(p));
		}
	}

	if (selected_list.IsEmpty())	return;
	if (m_mySystem)
		m_mySystem->Transfer(selected_list);
}

void CFileList::SelectAll()
{
	Redraw();	// Reset counts, and everything

TRACE_FUNCTION("CFileList::SelectAll")
size_t		n=m_contents.GetCount(), i;

	TRACE(7, "CFileList::SelectAll - " << n << " records" << endl)

	// Just select everything.  Depend on GetSelectedFiles()
	// to turn off "." and ".."

	for (i=0; i<n; i++)
	{
		IsSelected(i, TRUE);
	}
	Redraw();
}

void CFileList::GetSelectedFiles(
	CList<CDirEntry, const CDirEntry &> &selected_list)
{
	selected_list.RemoveAll();

POSITION	p;
size_t		n;

	// Ok, figure out what has to be transferred.  Look for something
	// that has been explicitly marked.

	for (p=m_contents.GetHeadPosition(), n=0; p; n++)
	{
	CDirEntry entry=m_contents.GetNext(p);

		if (IsSelected(n))
		{
			if (entry.m_isdirectory && (
				entry.m_name == DIRNAME_CURRENT ||
				entry.m_name == DIRNAME_PARENT))
			{
				// Come on, stop playing games

				IsSelected(n, FALSE);
			}
			else
				selected_list.AddTail(entry);
		}
	}
	TRACE(7, "CFileList::GetSelectedFiles - " <<
			selected_list.GetCount() << " records" << endl)
}

void	CFileList::SelectEntry(CString filename, BOOL flag)
{
size_t	i,n;

	if (!m_transferring)	return;

	n=m_listStrings.GetSize();
	for (i=0; i<n; i++)
	{
		if (m_listStrings[i] == filename)
		{
			MoveTo(i);
			IsSelected(i, flag);
			return;
		}
	}
}

CString	CFileList::CurrentItemText()
{
size_t n=CurrentItem();

	if (n >= m_contents.GetCount())	return (_T(""));

POSITION p=m_contents.FindIndex(n);

	return (m_contents.GetNext(p).m_name);
}
