// DSTART
//                  sFTP - a curses-based FTP client for Linux.
// 
//                            Current version is 0.81
// 
//                  Copyright 1997-1998, Double Precision, Inc.
// 
// This program is distributed under the terms of the GNU General Public
// License. See COPYING for additional information.
// DEND
#if HAVE_CONFIG_H
#include	"autoconfig.h"
#endif
#include	"afxwindow.h"
#include	<ctype.h>

static const char rcsid[]="$Id: afxwindow4.C,v 1.2 1998/06/09 02:12:11 mrsam Exp $";

CCEditWindow::CCEditWindow()
	: m_cursorPos(0), m_firstPos(0), m_maxWidth(0),
	m_isPwd(FALSE), m_isRequired(FALSE), m_justEntered(TRUE),
	m_isComboBox(FALSE)
{
}

CCEditWindow::~CCEditWindow()
{
}

void CCEditWindow::CreateEdit(int row, int col, int width, int maxwidth)
{
	Create(row, col, width, 1);
	m_maxWidth=maxwidth;
	CCEditWindow::Redraw();
	m_justEntered=TRUE;
}

void CCEditWindow::Destroy()
{
	CCWindow::Destroy();
}

void CCEditWindow::Redraw()
{
	if ((size_t)m_contents.GetLength() > m_maxWidth)
		m_contents=m_contents.Left(m_maxWidth);

size_t	w=m_contents.GetLength();
size_t	ww=Width();

	if (m_isComboBox && ww)	--ww;

	if (w && w == m_maxWidth)	--w;

	if (m_cursorPos > w)	m_cursorPos=w;
	if (m_firstPos > m_cursorPos)	m_firstPos=m_cursorPos;
	if (m_cursorPos >= ww && m_cursorPos-ww >= m_firstPos)
		m_firstPos=m_cursorPos-ww+1;

	if (!m_window)	return;

CString	buf=m_contents.Mid(m_firstPos);

	w=buf.GetLength();
	if (w < ww)
	{
		memset(buf.GetBuffer(ww)+w, '_', ww-w);
		buf.ReleaseBuffer(ww);
	}
	if (IsPassword())
	{
		memset(buf.GetBuffer(ww), '*', w);
		buf.ReleaseBuffer(ww);
	}
	Write(0, 0, buf, ww);
	if (m_isComboBox)
	{
		WriteAttrOn(A_REVERSE);

	int	w=Width();

		if (w)	--w;
//		Write(0, w, ACS_RARROW);
		Write(0, w, 'V');
		WriteAttrOff(A_REVERSE);
	}
	SetCursorPos(0, m_cursorPos - m_firstPos);
	Refresh();
}

BOOL CCEditWindow::Validate()
{
	if (IsRequired() && m_contents.GetLength() == 0)
		return (FALSE);
	return (TRUE);
}

BOOL CCEditWindow::AcceptInput(int ch)
{
BOOL	entered=m_justEntered;

	m_justEntered=FALSE;
	switch (ch)	{
	case KEY_LEFT:
		if (m_cursorPos > 0)
		{
			--m_cursorPos;
			CCEditWindow::Redraw();
			return (TRUE);
		}
		break;
	case KEY_RIGHT:
		{
		size_t w=m_contents.GetLength();

			if (w && w == m_maxWidth)	--w;

			if (m_cursorPos < w)
			{
				++m_cursorPos;
				CCEditWindow::Redraw();
				return (TRUE);
			}
		}
		break;
	case KEY_BACKSPACE:
		if (m_cursorPos)	--m_cursorPos;
		// FALLTHRU
	case 0x7F:
		m_contents=m_contents.Left(m_cursorPos) +
				m_contents.Mid(m_cursorPos+1);
		CCEditWindow::Redraw();
		return (TRUE);
	default:
		if (isprint(ch))
		{
			if (m_forbiddenChars.Find(ch) >= 0)
				return (TRUE);
			if (entered)
			{
				m_contents=_T("");
				m_cursorPos=0;
			}

			if ((size_t)m_contents.GetLength() < m_maxWidth)
			{
				m_contents=m_contents.Left(m_cursorPos) + ch
					+ m_contents.Mid(m_cursorPos);
				++m_cursorPos;
			}
			CCEditWindow::Redraw();
			return (TRUE);
		}
		break;
	}
	return (CCWindow::AcceptInput(ch));
}

void CCEditWindow::GetFocus()
{
	m_cursorPos=0;
	m_justEntered=TRUE;
	CCEditWindow::Redraw();
}

////////////////////////////////////////////////////////////////

CCButtonWindow::CCButtonWindow()
	:m_useCheckmark (FALSE), m_rightAlign(FALSE)
{
}

CCButtonWindow::~CCButtonWindow()
{
}

void CCButtonWindow::Create(int row, int col)
{
size_t	l=m_text.GetLength()+(m_useCheckmark ? 2:1);

	CCWindow::Create(row, col - (m_rightAlign ? l:0), l, 1);
	CCButtonWindow::Redraw();
}

void CCButtonWindow::Selected()
{
}

void CCButtonWindow::Redraw()
{
	if (HasInput())
		WriteAttrOn(A_REVERSE);

	Write(0, 0, (m_useCheckmark ? Value() ? _T("X "):_T("_ "):_T(" "))
			+ m_text);
	if (HasInput())
		WriteAttrOff(A_REVERSE);
	Refresh();
}

BOOL CCButtonWindow::AcceptInput(int ch)
{
	switch (ch)	{
	case ' ':
	case KEY_ENTER:
	case '\r':
	case '\n':
		Value( !Value());
		return (TRUE);
	case 'y':
	case 'Y':
		Value(TRUE);
		return (TRUE);
	case 'n':
	case 'N':
		Value(FALSE);
		return (TRUE);
	}
	return (CCWindow::AcceptInput(ch));
}

void CCButtonWindow::GetFocus()
{
	SetCursorPos(0, 0);
	CCButtonWindow::Redraw();
}

void CCButtonWindow::KillFocus()
{
	CCButtonWindow::Redraw();
}

////////////////////////////////////////////////////////////////


CCDialogWindow::CCDialogWindow() : boolTransparent(FALSE)
{
}

CCDialogWindow::~CCDialogWindow()
{
	CCDialogWindow::Destroy();
}

void CCDialogWindow::Create(int width, int height,
	CCDialogWindowInfo *info, size_t ninfo)
{
int	bw=BaseWindow->Width();
int	bh=BaseWindow->Height();

	CCDialogWindow::Create(height < bh ? (bh-height)/2:0,
		width < bw ? (bw-width)/2:0, width, height, info, ninfo);
}

void CCDialogWindow::Create(int nrow, int ncol, int width, int height,
	CCDialogWindowInfo *info, size_t ninfo)
{
size_t	i;

	CCWindow::Create(nrow, ncol, width, height);
	try
	{
		i=0;
		text.SetSize(ninfo);
		row.SetSize(ninfo);
		col.SetSize(ninfo);
		windows.SetSize(ninfo);

		for (i=0; i<ninfo; i++)
		{
			text[i]=info[i].Label();
			row[i]=info[i].Row();
			col[i]=info[i].Col();
			windows[i]=info[i].Create(nrow, ncol);
			if (!windows[i])
				AfxThrowInternalException();
		}
		windows[0]->SetInput();
	} catch (...)
	{
		text.SetSize(0);
		while (i)
		{
			--i;
			windows[i]->Destroy();
		}
		CCDialogWindow::Destroy();
		throw;
	}
	Redraw();
}

void CCDialogWindow::Redraw()
{
int	i;

	for (i=0; i<text.GetSize(); i++)
	{
	CString t=text[i];

		if ( t.GetLength())
			Write(row[i], col[i] - 2 - t.GetLength(), t);
	}
	Refresh();
}

void CCDialogWindow::Destroy()
{
int	i;

	for (i=0; i<text.GetSize(); i++)
		windows[i]->Destroy();
	text.SetSize(0);
	CCWindow::Destroy();
}

BOOL CCDialogWindow::AcceptInput(int)
{
	return (TRUE);
}

BOOL CCDialogWindow::AcceptInputNonFocus(int ch)
{
size_t	i;

	switch (ch)	{
	case ESCKEY:
		if (!Transparent())
		{
			Destroy();
			return (TRUE);
		}
		break;
	case KEY_DOWN:
	case KEY_RIGHT:
		NextField();
		return (TRUE);
	case '\t':
		NextField(TRUE);
		return (TRUE);
	case KEY_UP:
	case KEY_LEFT:
	case KEY_BTAB:
		PrevField();
		return (TRUE);
	case KEY_ENTER:
	case '\n':
	case '\r':
		for (i=0; i<(size_t)text.GetSize(); i++)
			if (!windows[i]->Validate())
			{
				beep();
				windows[i]->SetInput();
				return (TRUE);
			}
		Selected();
		return (TRUE);
	}

	if (Transparent())
		return (CCWindow::AcceptInputNonFocus(ch));
	return (TRUE);
}

void CCDialogWindow::NextField(BOOL doValidate)
{
int i;

	for (i=0; i<text.GetSize(); i++)
		if (windows[i]->HasInput())
		{
			if (doValidate && !windows[i]->Validate())
			{
				beep();
				windows[i]->SetInput();
				return;
			}
			++i;
			break;
		}
	windows[i % text.GetSize()] -> SetInput();
}

void CCDialogWindow::PrevField()
{
int	i;

	for (i=0; i<text.GetSize(); i++)
		if (windows[i]->HasInput())
		{
			i += text.GetSize()-1;
			break;
		}
	windows[i % text.GetSize()]->SetInput();
}

void CCDialogWindow::Selected()
{
}

CCTitledDialogWindow::CCTitledDialogWindow()
{
}

CCTitledDialogWindow::CCTitledDialogWindow(CString title) : m_title(title)
{
}

CCTitledDialogWindow::~CCTitledDialogWindow()
{
	Destroy();
}

void CCTitledDialogWindow::Create(int width, int height,
	CCDialogWindowInfo *info, size_t ninfo)
{
int	bw=BaseWindow->Width();
int	bh=BaseWindow->Height();

	m_title.Create(height < bh ? (bh-height)/2:0,
		width < bw ? (bw-width)/2:0, width, height);

	CCDialogWindow::Create(m_title.Row(), m_title.Col(), width, height,
		info, ninfo);
	MoveInFrontOf(&m_title);
}

void CCTitledDialogWindow::Create(int nrow, int ncol, int width, int height,
	CCDialogWindowInfo *info, size_t ninfo)
{
	m_title.Create(nrow, ncol, width, height);
	CCDialogWindow::Create(nrow, ncol, width, height, info, ninfo);
	MoveInFrontOf(&m_title);
}

void CCTitledDialogWindow::Destroy()
{
	CCDialogWindow::Destroy();
	m_title.Destroy();
}

//////////////////////////////////////////////////////////////////////////

CCLogWindow::CCLogWindow(size_t maxlog)
	: m_maxlog(maxlog), m_cursorRow(0), m_firstRow(0)
{
}

CCLogWindow::~CCLogWindow()
{
}

void CCLogWindow::AddLine(CString line)
{
	m_log.AddTail(line);

size_t	c=m_log.GetCount();

	if (m_cursorRow + 2 == c)	++m_cursorRow;
	if (m_log.GetCount() > m_maxlog)
	{
		m_log.RemoveHead();
		if (m_cursorRow)	--m_cursorRow;
	}
	CCLogWindow::Redraw();
}

void CCLogWindow::Create(int row, int col, int width, int height)
{
	CCTitledWindow::Create(row, col, width, height);
	CCLogWindow::Redraw();
}

void CCLogWindow::Destroy()
{
	m_view.Destroy();
	CCTitledWindow::Destroy();
	m_log.RemoveAll();
}

void CCLogWindow::Redraw()
{
POSITION p=NULL;
size_t nItems=m_log.GetCount();
size_t nRows=Height();
size_t	i;
CString	buf;
size_t w=Width();

	if (m_cursorRow >= nItems)
		m_cursorRow=nItems ? nItems-1:0;

	if (m_firstRow > m_cursorRow)
		m_firstRow=m_cursorRow;
	if (m_cursorRow - m_firstRow >= nRows)
		m_firstRow=m_cursorRow - nRows+1;

	if (nRows && nRows >= nItems)
		m_firstRow=0;

	if (m_firstRow < nItems)
		p=m_log.FindIndex(m_firstRow);
	SetCursorPos(m_cursorRow - m_firstRow, 0);

	for (i=0; i<nRows; i++)
	{
		if (!p)	buf=_T("");
		else	buf=m_log.GetNext(p);

	size_t	l=buf.GetLength();

		if (l < w)
		{
			memset(buf.GetBuffer(w)+l, ' ', w-l);
			buf.ReleaseBuffer(w);
		}

		Write(i, 0, buf, w);
	}
	Refresh();
}

BOOL CCLogWindow::AcceptInput(int ch)
{
	switch (ch)	{
	case KEY_PPAGE:
		if (m_cursorRow < (size_t)Height())
			m_cursorRow=0;
		else
			m_cursorRow -= Height();
		CCLogWindow::Redraw();
		return (TRUE);
	case KEY_UP:
		if (m_cursorRow > 0)	--m_cursorRow;
		CCLogWindow::Redraw();
		return (TRUE);
	case KEY_NPAGE:
		if (m_log.GetCount() - m_cursorRow <= (size_t)Height())
		{
			m_cursorRow=m_log.GetCount();
			if (m_cursorRow)	--m_cursorRow;
		}
		else
			m_cursorRow += Height();

		CCLogWindow::Redraw();
		return (TRUE);
	case KEY_DOWN:
		if (m_cursorRow + 1 < m_log.GetCount())
			++m_cursorRow;
		CCLogWindow::Redraw();
		return (TRUE);
	case '\r':
	case '\n':
		m_view.Title(Title());
		m_view.Create();
		m_view=m_log;
		m_view.MoveTo(m_view.GetSize());
		return (TRUE);
	}
	return (CCTitledWindow::AcceptInput(ch));
}
