// DSTART
//                  sFTP - a curses-based FTP client for Linux.
// 
//                            Current version is 0.81
// 
//                  Copyright 1997-1998, Double Precision, Inc.
// 
// This program is distributed under the terms of the GNU General Public
// License. See COPYING for additional information.
// DEND
#ifndef	afxwindow_h
#define	afxwindow_h

#include	"afx.h"
#include	"afxtempl.h"
#include	"afxioh.h"

#include	<curses.h>

static const char afxwindow_rcsid[]="$Id: afxwindow.h,v 1.5 1998/06/11 04:32:37 mrsam Exp $";

///////////////////////////////////////////////////////////////
//
//  Encapsulate libncurses windows
//

#define	MENUKEY	('O' & 31)
#define	MENUKEYNAME	_T("Ctrl-O")

#define	ESCKEY	27
#define	REFRESHKEY	('L' & 31)

class CCWindow;
class CCBaseWindow;

//  CCWindow - the base class from which all window classes are derived

class CCWindow {

protected:

	int m_row, m_col, m_width, m_height;		// Where I am
	int m_cursrow, m_curscol;			// Where cursor should
							// be.

	virtual BOOL IsBaseWindow();	// This is an entry for the base
					// window
static	CList<CCWindow *, CCWindow *> AllWindowList;
					// List of all current windows

//static	CList< CList<CCWindow *, CCWindow *>, CList<CCWindow *, CCWindow *> &>
//					InputWindowListStack;

static	CCBaseWindow	*BaseWindow;	// The base window
static	CCWindow	*CurrentInputWindow;	// Current window getting input
	WINDOW	*m_window;		// My curses PAD
	POSITION m_listPos;

public:
	void	MoveInFrontOf(CCWindow *);
	void	MoveBehindOf(CCWindow *);
	friend class CCTitledWindow;
	friend class CCBaseWindow;
	friend class CSigIohWinch;

	CCWindow();
	virtual ~CCWindow();

	virtual void Create(int row, int col, int width, int height);
						// Create me right here

	void Create(int width, int height);	// Create me centered.
	virtual void Create();			// Often overriden
	virtual BOOL Validate();		// Often overriden
	virtual void Destroy();
	virtual void Redraw();
	virtual void Refresh();		// Refresh my contents
private:
	void refreshpad();
public:
	virtual void Resize(int nrows, int ncols);
	virtual void Resized();
	virtual int Width();
	virtual int Height();
	virtual int Row();
	virtual int Col();
	virtual BOOL AcceptInput(int);
	virtual BOOL AcceptInputNonFocus(int);
	virtual void GetFocus();
	virtual void KillFocus();
	void	SetInput();
	BOOL	HasInput() { return (m_window && CurrentInputWindow &&
					CurrentInputWindow->m_listPos ==
						m_listPos); }
	void	MoveToFront();
	void	SetCursorPos(int row, int col)
		{
			m_cursrow=row;
			m_curscol=col;
		}
	void	Write(int row, int col, LPCTSTR str)
		{
			Write(row, col, str, _tcslen(str));
		}

	void	Write(int row, int col, LPCTSTR str, int cnt);
	void	Write(int row, int col, int theChar);
	void	WriteAttrOn(int attrs);
	void	WriteAttrOff(int attrs);

static  void	Flush();
static	void	GetInput(int &, CCWindow *&);
static	void	ProcessInput(int, CCWindow *);

static	void	CCWindow::InputLoop();	// Alternative to using ioh

static	void	QuitWindows();
static	BOOL	QuitWindowsFlag;
static	BOOL	FlushedFlag;
	} ;

//////////////////////////////////////////////////////////////////
//
//  The base window is the physical display
//  We must have ONLY one instance of this class.
//

class CCBaseWindowSUSPEND : public CSigIoh {

static	BOOL	use_subshell;

static	void	suspend();

public:
	CCBaseWindowSUSPEND();
	void	Signal();

static	void	UseSubshell();

	~CCBaseWindowSUSPEND() {}
} ;

class CCBaseWindowTERMINATE : public CSigIoh {
public:
	CCBaseWindowTERMINATE(int signum) : CSigIoh(signum) {}
	void Signal();
	~CCBaseWindowTERMINATE() {}
} ;

class CCBaseWindow : public CCWindow, public CIoh {

	CCBaseWindowSUSPEND m_sigsuspend;
	CCBaseWindowTERMINATE m_sigint, m_sigterm, m_sighup;

protected:
	BOOL	IsBaseWindow();
public:
	CCBaseWindow();
	~CCBaseWindow();

	void Create();
	void Create(int row, int col, int width, int height);
				// Ignores all params, creates for current
				// window
	void Refresh();		// Refreshes all windows
	void Destroy();

	// Implement IOH

	BOOL CanRead();
	void Read();
} ;

// CCTitledWindow is derived from CCWindow, and gives me a window with a border,
// with a title

class CCTitledWindow : public CCWindow {

	CString m_title;
	CCWindow m_title_window;	// The window used to draw the title.

public:
	CCTitledWindow();
	CCTitledWindow(CString);
	~CCTitledWindow();

	void Title(CString title)	{ m_title=title;
						CCTitledWindow::Redraw();
						m_title_window.Refresh(); }
	CString Title()			{ return (m_title); }

	// Override Create()/Destroy(), we need to do some post-processing

	void Create(int row, int col, int width, int height);
	void Destroy();

private:
	void Redraw();
} ;

// A window that displays a vertically-scrollable list
// Most parameters must be defined by a derived class.

class CCListWindow : public CCWindow {

	size_t	m_firstRow;	// First row shown
	size_t	m_firstCol;	// First column shown.
	size_t	m_cursorRow;	// Where cursor should be
	BOOL	m_alwaysSelected; // Show highlighted even without input focus
	size_t	m_biggestWidth;	// Amongst list items shown

	CArray<BOOL, BOOL> m_selected;	// Selected flags

	size_t	WidthOverhead() {
		size_t n=1;

		if (Type() != Vertical)	++n;
		return (n);
	}


	size_t	ListShown() { return (Height()); }

public:
	enum ListType	{ Vertical, Horizontal };
private:
	ListType	m_listType;
	BOOL		m_doSelects;
public:
	CCListWindow();
	CCListWindow(ListType t);
	void	DoSelects(BOOL flag);

	BOOL	DoSelects() { return (m_doSelects); }
	ListType Type() { return (m_listType); }
	void	SetVertical() { m_listType=Vertical; }
	void	SetHorizontal() { m_listType=Horizontal; }
	~CCListWindow();
	void Create(int row, int col, int width, int height);
	void Destroy();
	void Redraw();
	void GetFocus();
	void KillFocus();
	void MoveTo(size_t n)	{ m_cursorRow=n; CCListWindow::Redraw(); }
	void AlwaysSelected(BOOL flag) { m_alwaysSelected=flag; CCListWindow::Redraw(); }

	size_t CurrentItem()	{ return (m_cursorRow); }
	virtual size_t GetCount();	// # of items in list
	virtual void Render(size_t ndx, size_t hOffset,
			int row, int col, int cnt,
			BOOL selected, BOOL hascursor);
	virtual size_t RenderWidth(size_t);
	virtual BOOL IsSeparator(size_t);
	BOOL	IsSelected(size_t ndx);
	void	IsSelected(size_t ndx, BOOL flag);
private:
	void	SizeSelected();
public:
	virtual void Selected(size_t ndx);
	virtual void DeleteKey(size_t ndx);
	void Reset();
	BOOL AcceptInput(int);
} ;

// Define some parameters to have me store simple text strings :-)

class CCTextListWindow: public CCListWindow {

public:
	CStringArray	m_listStrings;
	CUIntArray	m_stringRow, m_stringCol;

	CCTextListWindow();
	~CCTextListWindow();
	void Create(int row, int col, int width, int height);
	void Destroy();

	size_t GetCount();
	virtual void Render(size_t ndx, size_t off, int row, int col, int cnt,
			BOOL selected, BOOL hascursor);
	virtual size_t RenderWidth(size_t);
	virtual BOOL IsSeparator(size_t);
	virtual CString CurrentItemText() {
		size_t n=CurrentItem();

			return ((size_t)n < (size_t)m_listStrings.GetSize() ?
				m_listStrings[n]:(CString)_T(""));
	}
} ;

template<class T> class CCTextListWindowRedirect : public CCTextListWindow {
	T *m_pSelectWindow;
public:
	CCTextListWindowRedirect() : m_pSelectWindow(NULL) {}
	~CCTextListWindowRedirect() {}
	void Selected(size_t ndx) { if (m_pSelectWindow)
						m_pSelectWindow->Selected(ndx);
				}

	void operator=(T *p) { m_pSelectWindow=p; }
} ;

/////////////////////////////////////////////////////////////////////////////
//
//  Define list button class, suitable for redirection with the above
//  above template.  Instead of deriving from CCListWindow, this way allows
//  the logic to be reused with menu lists

class CCListButtonWindow : public CCWindow {

	CCListWindow *m_plist;

	CUIntArray	m_radioButtons;

public:
	CCListButtonWindow() : m_plist(NULL)	{}
	~CCListButtonWindow()	{}
	void operator=(CCListWindow *p)	{ m_plist=p; }

	void	AddButtonListEntry(size_t n)
		{
			m_radioButtons.Add(n);
		}
	void	IsSelected(size_t i, BOOL flag);
	void	Selected(size_t i);
	virtual void	SelectedButton(size_t i);
	virtual void	SelectedNonButton(size_t i);

	// When the list is in the menu, MenuPopup class calls Create()
	// instead of Selected().  Punt

	void	Create();
} ;

/////////////////////////////////////////////////////////////////////////////
//
//  The menus:
//
//  CCMenubarWindow - the menu bar
//  CCMenuPopupWindow - the popup window

class CCMenuPopupWindow;
class CCMenuBarWindow;

class CCMenuBarWindowList : public CCTextListWindow {
public:
	CCMenuBarWindow	*m_menuBar;

	CCMenuBarWindowList() : m_menuBar(0) {}
	~CCMenuBarWindowList() {}
	void Selected(size_t ndx);
	BOOL AcceptInput(int ch)	// We want to change the 'DOWN'
		{			// key to act like ENTER.
			if (ch == KEY_DOWN)
				ch=KEY_ENTER;

			return (CCTextListWindow::AcceptInput(ch));
		}
	} ;

class CCMenuBarWindow : public CCTitledWindow {	// For the border

	CCWindow *		m_mainWindow;
	CCMenuBarWindowList	m_menuBarList;
	CList<CCMenuPopupWindow *, CCMenuPopupWindow *> m_popUps;
	CCWindow *	m_prevWindow;
public:
	CCMenuBarWindow(CCWindow *);
	~CCMenuBarWindow();

	void	Create();
	void	Destroy();
	void	Add(CString str, CCMenuPopupWindow *);
	void	SetInput();
	void	Left(BOOL doOpen)
		{
			SetInput();
			m_menuBarList.AcceptInput(KEY_LEFT);
			if (doOpen)
				m_menuBarList.AcceptInput(KEY_ENTER);
		}
	void	Right(BOOL doOpen)
		{
			SetInput();
			m_menuBarList.AcceptInput(KEY_RIGHT);
			if (doOpen)
				m_menuBarList.AcceptInput(KEY_ENTER);
		}
	BOOL	MenuInput();

	void	MenuOff();
	BOOL	AcceptInputNonFocus(int);

	void	Selected(size_t);
	} ;

inline void CCMenuBarWindowList::Selected(size_t ndx)
{
	m_menuBar->Selected(ndx);
}

class CCMenuPopupWindowList : public CCTextListWindow {
public:
	CArray<CCWindow *, CCWindow *> m_menuWindow;

	CCMenuPopupWindow *m_menuPopup;
	CCMenuPopupWindowList() : m_menuPopup(NULL)	{}
	~CCMenuPopupWindowList() {}
	BOOL AcceptInput(int ch);
	void Selected(size_t);
	} ;

class CCMenuPopupWindow : public CCTitledWindow {

public:
	CCMenuPopupWindowList m_menuList;
	CCMenuBarWindow *m_pMenubar;

	CCMenuPopupWindow();
	~CCMenuPopupWindow();

	void	Create(int row, int col, CCMenuBarWindow *mBar);
	void	Destroy();
	size_t	Add(CString str, CCWindow *wptr);
	size_t	CurrentItem();
	BOOL	MenuInput();
	} ;

/////////////////////////////////////////////////////////////////////////////
//
//  An edit box
//

class CCEditWindow : public CCWindow {

	CString	m_contents;
	CString m_forbiddenChars;
	size_t  m_cursorPos;	// Where cursor is
	size_t	m_firstPos;	// Shown
	size_t	m_maxWidth;

	BOOL	m_isPwd, m_isRequired;
	BOOL	m_justEntered;
protected:
	BOOL	m_isComboBox;	// Overriden
public:
	CCEditWindow();
	~CCEditWindow();
	BOOL	IsPassword() { return (m_isPwd); }
	BOOL	IsRequired() { return (m_isRequired); }
	void	Password(BOOL f=TRUE) { m_isPwd=f; }
	void	Required(BOOL f=TRUE) { m_isRequired=f; }
	void	Forbidden(CString s)  { m_forbiddenChars=s; }
	virtual BOOL Validate();

	void	operator=(CString s) { m_contents=s; CCEditWindow::Redraw(); }
	operator CString() { return (m_contents + _T("")); }

	void CreateEdit(int row, int col, int width, int maxwidth);
	void Destroy();

	void Redraw();
	void GetFocus();
	BOOL AcceptInput(int);
} ;

/////////////////////////////////////////////////////////////////////////////
//
//  A combo box
//

class CCEditComboBoxWindow;

	// CCEditComboTextListWindow - the actual text list

class CCEditComboTextListWindow : public CCTextListWindow {
public:
	CCEditComboBoxWindow	*m_pComboBox;
	CCEditComboTextListWindow();
	~CCEditComboTextListWindow();
	void Selected(size_t);
	BOOL AcceptInput(int ch);
	} ;

	// CCEditComboListWindow - the titled window that drops down

class CCEditComboListWindow : public CCTitledWindow {
public:
	CCEditComboTextListWindow m_list;

	CCEditComboListWindow();
	~CCEditComboListWindow();

	void Create(int row, int col, int minwidth);
	void Destroy();
	void operator=(CStringArray &array);

	} ;

	// CCEditComboBoxWindow - the combo edit box window

class CCEditComboBoxWindow : public CCEditWindow {

	CCEditComboListWindow m_listwindow;

public:
	CCEditComboBoxWindow();
	~CCEditComboBoxWindow();
	void Destroy();
	void operator=(CStringArray &array);
	void operator=(CString);
	BOOL AcceptInput(int);

	friend class CCEditComboTextListWindow;
} ;

///////////////////////////////////////////////////////////////////////////
//
//  Button

class CCButtonWindow : public CCWindow {

	CString	m_text;
	BOOL	m_value;
	BOOL	m_useCheckmark;
	BOOL	m_rightAlign;
public:
	CCButtonWindow();
	~CCButtonWindow();
	CCButtonWindow(CString &text) : m_text(text), m_value(FALSE),
			m_useCheckmark(FALSE), m_rightAlign(FALSE) {}
	void	Checkmark(BOOL flag)	{ m_useCheckmark=flag; }
	void	RightAlign(BOOL flag)	{ m_rightAlign=flag; }
	void	Title(CString text) { m_text=text; CCButtonWindow::Redraw(); }
	BOOL	Value() { return (m_value); }
	operator BOOL() { return (Value()); }
	void Value(BOOL f) { m_value=f; CCButtonWindow::Redraw(); Selected(); }
	void operator=(BOOL f) { Value(f); }
	void Create(int row, int col);
	void GetFocus();
	void KillFocus();
	BOOL AcceptInput(int);
	virtual void Selected();
private:
	void Redraw();
} ;

// Convenient templates for the OK and CANCEL buttons in a dialog

template <class T> class CCOkButtonRedirect : public CCButtonWindow {

	T *m_ptr;
public:
	CCOkButtonRedirect() : m_ptr(NULL)	{}
	~CCOkButtonRedirect()			{}
	void operator=(T *p)	{ m_ptr=p; }
	void Selected()		{ if (m_ptr)	m_ptr->Ok(); }
	} ;

template <class T> class CCCancelButtonRedirect : public CCButtonWindow {

	T *m_ptr;
public:
	CCCancelButtonRedirect() : m_ptr(NULL)	{}
	~CCCancelButtonRedirect()			{}
	void operator=(T *p)	{ m_ptr=p; }
	void Selected()		{ if (m_ptr)	m_ptr->Cancel(); }
	} ;

//////////////////////////////////////////////////////////////////////////////
//
// A view window is a window suitable for displaying a short file.

class CCViewWindow : public CCTitledWindow {

	CStringArray m_text;
	size_t	m_cursorRow;
	size_t	m_firstRow;
	size_t	m_firstCol;
public:
	CCViewWindow();
	~CCViewWindow();
	void	Create(int row, int col, int width, int height);
	void	Create();
	void	Destroy();
	void	Redraw();
	BOOL	AcceptInput(int);
	void	operator=(CStringArray &);
	void	operator=(CStringList &);
	void	MoveTo(size_t row)
	{
		m_cursorRow=row;
		CCViewWindow::Redraw();
	}
	size_t	GetSize() { return (m_text.GetSize()); }
	} ;

//////////////////////////////////////////////////////////////////////////////
//
// A log window is a window suitable for displaying some kind of a log.
// We keep a limited number of lines, and if the user presses ENTER when
// in this window, we pop up a full screen view.

class CCLogWindow : public CCTitledWindow {
	CStringList m_log;
	size_t	m_maxlog;
	size_t	m_cursorRow;
	size_t	m_firstRow;
	CCViewWindow m_view;
public:
	CCLogWindow(size_t maxlog=99);
	~CCLogWindow();

	void	AddLine(CString line);
	void	operator<<(CString line) { AddLine(line); }

	void	Create(int row, int col, int width, int height);
	void	Destroy();

	void	Redraw();
	BOOL	AcceptInput(int);
} ;

///////////////////////////////////////////////////////////////////////////
//
//  Dialog information array

class CCDialogWindowInfo {

	int m_row, m_col, m_width, m_height;	// For most windows
	CString m_label;			// Edit/button box

	// Edit box specific:

	CCEditWindow *m_edit_box;
	int m_maxwidth;				// For edit box

	// Button box specific:

	CCButtonWindow *m_button_window;

	// List window specific:

	CCListWindow *m_list_window;

	// Log window specific:

	CCLogWindow *m_log_window;

	// Combo window specific:

	CCEditComboBoxWindow *m_combo_window;

public:
	CCDialogWindowInfo(int row, int col, int width, int maxwidth,
			CCEditWindow *edit_box, CString label) :
			m_row(row), m_col(col), m_width(width), m_height(0),
			m_label(label),
			m_edit_box(edit_box),
			m_maxwidth(maxwidth),
			m_button_window(NULL),
			m_list_window(NULL),
			m_log_window(NULL),
			m_combo_window(NULL)
			{}

	CCDialogWindowInfo(int row, int col, int width, int maxwidth,
			CCEditComboBoxWindow *edit_box, CString label) :
			m_row(row), m_col(col), m_width(width), m_height(0),
			m_label(label),
			m_edit_box(NULL),
			m_maxwidth(maxwidth),
			m_button_window(NULL),
			m_list_window(NULL),
			m_log_window(NULL),
			m_combo_window(edit_box)
			{}

	CCDialogWindowInfo(int row, int col, CCButtonWindow *button_window) :
			m_row(row), m_col(col), m_width(0), m_height(0),
			m_edit_box(NULL),
			m_maxwidth(0),
			m_button_window(button_window),
			m_list_window(NULL),
			m_log_window(NULL),
			m_combo_window(NULL)
			{}

	CCDialogWindowInfo(int row, int col, int width, int height,
			CCListWindow *list_window, CString label) :
			m_row(row), m_col(col), m_width(width), m_height(height),
			m_label(label),
			m_edit_box(NULL),
			m_maxwidth(0),
			m_button_window(NULL),
			m_list_window(list_window),
			m_log_window(NULL),
			m_combo_window(NULL)
			{}

	CCDialogWindowInfo(int row, int col, int width, int height,
			CCLogWindow *log_window, CString label) :
			m_row(row), m_col(col), m_width(width), m_height(height),
			m_label(label),
			m_edit_box(NULL),
			m_maxwidth(0),
			m_button_window(NULL),
			m_list_window(NULL),
			m_log_window(log_window),
			m_combo_window(NULL)
			{}

	CCDialogWindowInfo(const CCDialogWindowInfo &o)
		: m_row(o.m_row),
		m_col(o.m_col),
		m_width(o.m_width),
		m_height(o.m_height),
		m_label(o.m_label),
		m_edit_box(o.m_edit_box),
		m_maxwidth(o.m_maxwidth),
		m_button_window(o.m_button_window),
		m_list_window(o.m_list_window),
		m_log_window(o.m_log_window),
		m_combo_window(o.m_combo_window)	{}

	int Row() { return (m_row); }
	int Col() { return (m_col); }
	CString Label() { return (m_label); }

	CCWindow *Create(int dlg_row, int dlg_col)
	{
		if (m_edit_box)
		{
			m_edit_box->CreateEdit(dlg_row+m_row,
				dlg_col+m_col,
				m_width, m_maxwidth);
			return (m_edit_box);
		}

		if (m_combo_window)
		{
			m_combo_window->CreateEdit(dlg_row+m_row,
				dlg_col+m_col,
				m_width, m_maxwidth);
			return (m_combo_window);
		}

		if (m_button_window)
		{
			m_button_window->Create(dlg_row+m_row, dlg_col+m_col);
			return (m_button_window);
		}

		if (m_list_window)
		{
			m_list_window->Create(dlg_row+m_row, dlg_col+m_col,
				m_width, m_height);
			return (m_list_window);
		}

		if (m_log_window)
		{
			m_log_window->Create(dlg_row+m_row, dlg_col+m_col,
				m_width, m_height);
			return (m_log_window);
		}
		return (NULL);
	}

	~CCDialogWindowInfo()	{}
} ;


class CCDialogWindow : public CCWindow {

	CStringArray		text;
	CArray<int, int>	row;
	CArray<int, int>	col;
	CArray<CCWindow *, CCWindow *>	windows;

	BOOL			boolTransparent;
public:
	CCDialogWindow();
	~CCDialogWindow();

	void Create(int width, int height,
		CCDialogWindowInfo *, size_t);
	void Create(int row, int col, int width, int height,
		CCDialogWindowInfo *, size_t);
	void Destroy();
	BOOL AcceptInput(int);
	BOOL AcceptInputNonFocus(int);
	void Transparent(BOOL flag)
		{
			boolTransparent=flag;
		}
	BOOL Transparent() { return (boolTransparent); }
	virtual void Selected();	// Enter key pressed
	void Redraw();
	void NextField(BOOL doValidate=FALSE);
	void PrevField();
} ;

class CCTitledDialogWindow : public CCDialogWindow {
	CCTitledWindow	m_title;
public:
	CCTitledDialogWindow();
	CCTitledDialogWindow(CString);
	~CCTitledDialogWindow();
	void Create(int width, int height,
		CCDialogWindowInfo *, size_t);
	void Create(int row, int col, int width, int height,
		CCDialogWindowInfo *, size_t);
	void Destroy();
	void Title(CString title) { m_title.Title(title); }
} ;

#endif
