// DSTART
//                  sFTP - a curses-based FTP client for Linux.
// 
//                            Current version is 0.81
// 
//                  Copyright 1997-1998, Double Precision, Inc.
// 
// This program is distributed under the terms of the GNU General Public
// License. See COPYING for additional information.
// DEND
#ifndef	afxioh_h
#define	afxioh_h

#if HAVE_CONFIG_H
#include	"autoconfig.h"
#endif

static const char afxioh_rcsid[]="$Id: afxioh.h,v 1.2 1998/06/09 02:12:11 mrsam Exp $";

#include	"afx.h"
#include	"afxtempl.h"
#if TIME_WITH_SYS_TIME
#include	<sys/time.h>
#include	<time.h>
#else
#if HAVE_SYS_TIME_H
#include	<sys/time.h>
#else
#include	<time.h>
#endif
#endif

/////////////////////////////////////////////////////////////////////////////
//
//	Generic IO multiplexing.
//
// The CIoh class provides an interface to the select() call.  The file
// descriptor must be created elsewhere.  It is usually a socket, but
// it also may be a pipe (whatever works with select() is fine here).
//
// The static Select() function selects all descriptors, then calls
// either the Read or Write functions for descriptors which were flagged
// by select.  The CanRead() and CanWrite() functions are used to
// declare which descriptors are suitable for being selected for
// Read or the Write.  The default definitions of these functions use a
// flag, which is initially cleared.  CanRead() and CanWrite() will
// indicate if the flag is clear, and Read() and Write() will set the flag,
// so once a descriptor indicates read/write availability, it will not be
// selected again for read or write until the flag is cleared, via
// ClearRead() and ClearWrite().  This is usually fine for most situations.
//
// If necessary, most of this behavior may be overriden.
//
// A timeout feature is available.  Use SetTimeout() to have Select()
// invoke virtual function Timeout() if data cannot be EITHER read or
// written for the given period of timeout.
//
/////////////////////////////////////////////////////////////////////////////

class CIoh;

class CIoh	{

static CList<CIoh *, CIoh *>	ioh_list;	// All declared instances

	POSITION	m_pos;			// My position in this list.
	int	m_fd;				// My descriptor

	BOOL	m_okread, m_okwrite;

	BOOL	m_hasTimeout;			// Timeout interval set
	time_t	m_timeout;			// When.

public:

	CIoh()	: m_fd(-1)	{}
	virtual ~CIoh();
	void	fd(int);			// Set descriptor
	int	fd() { return (m_fd); }

	void	SetTimeout(int nseconds);
	void	CancelTimeout()	{ m_hasTimeout=FALSE; }

	virtual BOOL CanRead();
	virtual BOOL CanWrite();
	virtual void ClearRead(BOOL isClear=TRUE);
	virtual void ClearWrite(BOOL isClear=TRUE);
	virtual void Timeout();
	virtual void Read();
	virtual void Write();

static int Select(struct timeval *tv=NULL);
static int Select(int tv_sec, int tv_usec=0);
static void Forked();

friend class CSigIoh;

private:
static	class CSigIoh *sigtraps[];
static  int		sigcaught[];
static	int		siginselect;
static	RETSIGTYPE sighandler(int);
	} ;

/////////////////////////////////////////////////////////////////////////////
//
// Also define signal handlers.  We may want to trap some signals, but
// don't want to process them when we're in an unknown state.  Using
// the CSigIoh class, the signal handler function will be called only when
// CIoh::Select() is running, in other words, the Select() call becomes
// the unified I/O and signal handler, and all the data structures in the
// program will be consistent.
//
// Some caveats.  There are some minor race condition that can happen,
// resulting in a signal being dropped if two signals are sent to the
// process in a row.  Since we are interested in trapping things like
// SIGTERM, or SIGHUP, or SIGTSTP, this is not very important.
//
// What is important, is that only ONE CSigIoh signal is instantiated
// for each signal.

class CSigIoh {
	int m_signum;
public:
	CSigIoh(int signum);
	virtual ~CSigIoh();
	virtual void Signal();
	} ;	

/////////////////////////////////////////////////////////////////////////
//
// Provide a generic template for a class derived from CIoh that needs
// to redirect handling to another class

template<class T> class CIohRedirect: public CIoh {

	T	*m_pClass;
public:
	void	(T::*m_pRead)();
	void	(T::*m_pWrite)();
	void	(T::*m_pTimeout)();

	CIohRedirect() : m_pClass(NULL)
		{
			m_pRead=(void (T::*)())NULL;
			m_pWrite=(void (T::*)())NULL;
			m_pTimeout=(void (T::*)())NULL;
		}

	void operator=(T *t) { m_pClass=t; }

	~CIohRedirect()	{}

	void Read()
	{
		if (m_pRead && m_pClass)
			(m_pClass->*m_pRead)();
		else
			CIoh::Read();
	}

	void Write()
	{
		if (m_pWrite && m_pClass)
			(m_pClass->*m_pWrite)();
		else
			CIoh::Write();
	}

	void Timeout()
	{
		if (m_pTimeout && m_pClass)
			(m_pClass->*m_pTimeout)();
	}

	void Read(void	(T::*pRead)())
	{
		m_pRead=pRead;
	}

	void Write(void	(T::*pWrite)())
	{
		m_pWrite=pWrite;
	}

	void Timeout(void	(T::*pTimeout)())
	{
		m_pTimeout=pTimeout;
	}
} ;

#endif
