// DSTART
//                  sFTP - a curses-based FTP client for Linux.
// 
//                            Current version is 0.81
// 
//                  Copyright 1997-1998, Double Precision, Inc.
// 
// This program is distributed under the terms of the GNU General Public
// License. See COPYING for additional information.
// DEND
#if HAVE_CONFIG_H
#include	"autoconfig.h"
#endif
#include	"afxioh.h"
#include	<signal.h>
#include	<errno.h>
#include	<time.h>
#include	<unistd.h>
#include	<stdio.h>
#include	"debug.h"

static const char rcsid[]="$Id: afxioh.C,v 1.2 1998/06/09 02:12:11 mrsam Exp $";

CList<CIoh *, CIoh *> CIoh::ioh_list;

CSigIoh *CIoh::sigtraps[NSIG];
int	CIoh::sigcaught[NSIG];
int	CIoh::siginselect=FALSE;

CIoh::~CIoh()
{
	if (m_fd >= 0)	ioh_list.RemoveAt(m_pos);
}

void CIoh::fd(int new_fd)
{
	if (new_fd < 0)
	{
		if (m_fd < 0)	return;
		ioh_list.RemoveAt(m_pos);
		m_fd= -1;
		return;
	}

	if (m_fd < 0)
		m_pos=ioh_list.AddTail(this);
	m_fd=new_fd;
	m_okread=TRUE;
	m_okwrite=TRUE;
	ClearRead(TRUE);
	ClearWrite(TRUE);
	CancelTimeout();
}

void CIoh::SetTimeout(int nseconds)
{
	time(&m_timeout);
	m_hasTimeout=TRUE;
	m_timeout += nseconds;
}

// Default select logic for this ioh

BOOL CIoh::CanRead()
{
	return (m_okread);
}

BOOL CIoh::CanWrite()
{
	return (m_okwrite);
}

void CIoh::ClearRead(BOOL ok)
{
	m_okread=ok;
	TRACE(9, "fd(" << m_fd << ") " << (ok ? "":"not ") << "readable."
			<< endl);
}

void CIoh::ClearWrite(BOOL ok)
{
	m_okwrite=ok;
	TRACE(9, "fd(" << m_fd << ") " << (ok ? "":"not ") << "writeable."
			<< endl);
}

void CIoh::Read()
{
	ClearRead(FALSE);
}

void CIoh::Write()
{
	ClearWrite(FALSE);
}

void CIoh::Timeout()
{
}

////////////////////////////////////////////////

int CIoh::Select(int tv_sec, int tv_usec)
{
struct timeval tv;

	tv.tv_sec=tv_sec;
	tv.tv_usec=tv_usec;
	return (Select(&tv));
}

int CIoh::Select(struct timeval *tv)
{
fd_set rset, wset;
POSITION p;
CIoh *ioh;
int	fdmax=0;
int	fdc;
struct	timeval my_tv;
time_t	current_time;

	FD_ZERO(&rset);
	FD_ZERO(&wset);

	time(&current_time);

	for (p=ioh_list.GetHeadPosition(); p; )
	{
		ioh=ioh_list.GetNext(p);

		fdc=ioh->fd();
		if (fdc >= 0)
		{
		BOOL canrw=FALSE;

			if (fdc > fdmax)	fdmax=fdc;

			if (ioh->CanRead())
			{
				FD_SET(fdc, &rset);
				canrw=TRUE;
			}

			if (ioh->CanWrite())
			{
				FD_SET(fdc, &wset);
				canrw=TRUE;
			}

			if (canrw && ioh->m_hasTimeout)
			{
			int	tinterval=0;

				if (ioh->m_timeout > current_time)
					tinterval=ioh->m_timeout - current_time;
				TRACE(9, DebugTimeStamp() <<
					" fd(" << fdc << ") timeout in "
					<< tinterval << " seconds." << endl);
				if (tv == NULL || tv->tv_sec >= tinterval)
				{
					tv= &my_tv;
					my_tv.tv_sec=tinterval;
					my_tv.tv_usec=0;
				}
			}
			else
			{
				TRACE(9, DebugTimeStamp() <<
					" fd(" << fdc << ") no timeout."
					<< endl);
			}
		}
	}

int	rc;
size_t	n;

	siginselect=TRUE;

	for (n=0; n < sizeof(sigcaught)/sizeof(sigcaught[0]); n++)
		if (sigcaught[n])
		{
			sigcaught[n]=FALSE;
			if (sigtraps[n])
				sigtraps[n]->Signal();
		}

	rc=::select(fdmax+1, &rset, &wset, NULL, tv);
	siginselect=FALSE;

	if (rc < 0)
	{
		if (errno == EINTR)	return (0);
		return (-1);
	}

	time(&current_time);

	for (p=ioh_list.GetHeadPosition(); p; )
	{
		ioh=ioh_list.GetNext(p);

		fdc=ioh->fd();
		if (fdc >= 0)
		{
			if (FD_ISSET(fdc, &rset))
			{
				TRACE(9, DebugTimeStamp() << " fd(" << fdc << ").Read()" << endl);
				ioh->CancelTimeout();
				ioh->Read();

// It is important to rescan the list, because the io handler could close
// some descriptors, and the list gets changed

				p=ioh_list.GetHeadPosition();
				FD_CLR(fdc, &rset);
				continue;
			}
			if (FD_ISSET(fdc, &wset))
			{
				TRACE(9, DebugTimeStamp() << " fd(" << fdc << ").Write()" << endl);
				ioh->CancelTimeout();
				ioh->Write();

				p=ioh_list.GetHeadPosition();
				FD_CLR(fdc, &wset);
				continue;
			}
			if (ioh->m_hasTimeout &&
				(ioh->CanRead() || ioh->CanWrite()) &&
				current_time >= ioh->m_timeout)
			{
				TRACE(9, DebugTimeStamp() <<
					" fd(" << fdc << ") timed out." << endl);
				ioh->CancelTimeout();
				ioh->Timeout();
				p=ioh_list.GetHeadPosition();
				continue;
			}
		}
	}
	return (rc);
}

// We need to know if we were forked, and are now about to exec.
// Let's clean up and close all files.

void CIoh::Forked()
{
CIoh *ioh;
POSITION p;
int	fd;

	for (p=ioh_list.GetHeadPosition(); p; )
	{
		ioh=ioh_list.GetNext(p);
		if ((fd=ioh->fd()) && fd > 2)	// Don't close standards
		{
			ioh->fd(-1);
			close(fd);
		}
	}
}

RETSIGTYPE CIoh::sighandler(int signum)
{
	signal(signum, &sighandler);
	if (siginselect)
	{
		if (sigtraps[signum])
			sigtraps[signum]->Signal();
	}
	else
		sigcaught[signum]=TRUE;
#if RETSIGTYPE != void
	return 0;
#endif
}

///////////////////////////////////////////////////////////////////////

CSigIoh::CSigIoh(int signum)
{
	m_signum= -1;

	if (signum >= 0 && signum <
		(int)(sizeof(CIoh::sigtraps)/sizeof(CIoh::sigtraps[0])))
	{
		m_signum=signum;
		CIoh::sigtraps[m_signum]=this;
		signal(signum, &CIoh::sighandler);
	}
}

CSigIoh::~CSigIoh()
{
	if (m_signum >= 0)
	{
		signal(m_signum, SIG_DFL);
		CIoh::sigtraps[m_signum]=NULL;
	}
}

void CSigIoh::Signal()
{
}
