/* mget.c -- v0.2
 *
 * (c) 1996 Kuno Software
 *
 * This program is free software. If you change
 * any part of it, please, send me a copy!
 * Martin Krause, mkrause@prz.tu-berlin.de
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <setjmp.h>
#include <errno.h>


#ifndef STRLEN
#define STRLEN 255
#endif

#ifndef TIMEOUT
#define TIMEOUT 300
#endif


typedef struct ftpdirs {
  char *name;
  struct ftpdirs *next;
  struct ftpdirs *prev;
} dirlist;

typedef struct ftpfiles {
  char *name;
  long size;
  struct ftpfiles *next;
  struct ftpfiles *prev;
} flelist;


int ftpopen(const char *);
int ftpsend(int, const char *);
int ftprecs(FILE *, const char *, const char *);
void ftpchdir(int, FILE *, const char *);
void ftpgetdir(int, FILE *, char *);
void ftpgetfle(int, FILE *, char *);
void newdir(const char *, dirlist **);
void newfle(const char *, long, flelist **);
void erasedlist(dirlist **, dirlist **);
void eraseflist(flelist **, flelist **);


#ifndef __hpux__
#if defined(__i386__) && !defined(__linux__)
/* FreeBSD */
#else
extern int connect(int , struct sockaddr *, int);
extern int send(int, const void *, int, unsigned int);
extern int bind(int, struct sockaddr *, int);
#endif
extern int accept(int, struct sockaddr *, int *);
extern int getsockname(int, struct sockaddr *, int *);
#endif
extern int socket(int, int, int);
extern int listen(int, int);
extern int setsockopt(int, int, int, const void *, int);
extern int fprintf(FILE *, const char *, ...);
extern int fscanf(FILE *, const char *, ...);
extern int fclose(FILE *);
extern void perror(const char *);
extern void (*signal(int, void (*)(int)))(int);


jmp_buf env;


int main(int argc, char **argv)
{
  char *server, *dir, *p;
  char strcmd[STRLEN];
  FILE *fp;
  register int s;
  extern FILE *fdopen(int, const char *);
  extern char *getenv(const char *);
  extern void timeout(int);
  extern void fatal(int);

  if (argc!=3) {
    (void) fprintf(stderr, "Syntax error.\nUsage: %s SERVER DIRECTORY\n\n", argv[0]);
    exit(-1);
  }
  server=argv[1];

  signal(SIGTERM, fatal);
  signal(SIGALRM, timeout);
  if (!setjmp(env)) {
    (void) alarm(TIMEOUT);
    s=ftpopen(server);
    (void) alarm(0);
  } else exit(-1);

  if (argv[2][0] != '/') {
    if (!(dir=(char *) malloc(strlen(argv[2])+1))) {
      perror("malloc"); exit(-1);
    }
    sprintf(dir, "/%s", argv[2]);
  } else dir=argv[2];

  if(!(fp=fdopen(s, "r"))) { perror("fdopen"); exit(-1); }
  if ((ftprecs(fp, "ready.", "error")) < 0) {
    (void) fprintf(stderr, "Connection refused.\n");
    exit(-1);
  }

  if ((ftpsend(s, "user anonymous")) < 0) { perror("send"); exit(-1); }
  if ((ftprecs(fp, "password.", "denied.")) < 0) {
    (void) fprintf(stderr, "Anonymous access denied.\n");
    exit(-1);
  }

#ifdef __linux__
  p=getenv("LOGNAME");
#else
  p=getenv("USER");
#endif
  if (!p) p="joe";
  strcpy(strcmd, "pass -");
  strcat(strcmd, p);
  strcat(strcmd, "@");
  if ((ftpsend(s, strcmd)) < 0) { perror("send(password)"); exit(-1); }
  if ((ftprecs(fp, "apply.", "failed.")) < 0) {
    (void) fprintf(stderr, "Login failed.\n");
    exit(-1);
  }

  if ((ftpsend(s, "type i")) <0) { perror("send(type)"); exit(-1); }
  if ((ftprecs(fp, "I.", " ")) < 0) {
    (void) fprintf(stderr, "Cannot change transfer mode.\n");
    exit(-1);
  }

  signal(SIGINT, fatal);
  ftpgetdir(s, fp, dir);

  if ((ftpsend(s, "quit")) < 0) { perror("send(quit)"); exit(-1); }
  (void) ftprecs(fp, "Goodbye.", " ");
  fclose(fp);
  close(s);
  exit(0);
}


void timeout(int sig) 
{
  signal(sig, SIG_IGN);
  (void) fprintf(stderr, "Timeout. Cannot connect to the server.\n");
  signal(SIGALRM, timeout);
  longjmp(env, 1);
}


int ftpopen(const char *hostname)
{
  struct hostent *hp;
  register int s;
  struct sockaddr_in sn;
#if defined(__i386__) && !defined(__linux__)
/* FreeBSD */
#else
  extern void bcopy(const void *, void *, int);
#endif
  if ((hp=gethostbyname(hostname)) == NULL) {
    (void) fprintf(stderr, "%s: unknown host\n", hostname);
    exit(-1);
  }
  if ((s=socket(AF_INET, SOCK_STREAM, 0)) < 0) { perror("socket"); exit(-1); }
  sn.sin_family=AF_INET;
  sn.sin_port=htons(21); /* ftp */
  bcopy(hp->h_addr, &sn.sin_addr, hp->h_length);
  if (connect(s, (struct sockaddr *) &sn, sizeof(sn)) < 0) { perror("connect"); exit(-1); }
  return(s);
}


int ftpsend(int s, const char *strcmd)
{
  int n, m;
  if((n=send(s, strcmd, strlen(strcmd), 0)) < 0) return(-1);
  if((m=send(s, "\n", 1, 0)) < 0) return(-1);
  return(n+m);
}


int ftprecs(FILE *fp, const char *bs, const char *er)
{
  char strcmd[STRLEN];
  int i;

  for(;;) {
    (void) fscanf(fp, "%s", strcmd);
    i=strcmp(strcmd, bs);
    if ((i==0) || (i==13)) return(0);
    if (!(strcmp(strcmd, er))) return(-1);
  }
  return(0);
}


void ftpchdir(int s, FILE *fp, const char *dir)
{
  char *strcmd;
  if (!(strcmd=(char *) malloc(5+strlen(dir)))) { perror("malloc"); exit(-1); }
  strcpy(strcmd, "cwd "); strcat(strcmd, dir);
  if ((ftpsend(s, strcmd)) < 0) { perror("send(cwd)"); exit(-1); }
  if ((ftprecs(fp, "successful.", "directory.")) < 0) {
    (void) fprintf(stderr, "%s: No such file or directory.\n", dir);
    exit(-1);
  }
  free(strcmd);
}


void ftpgetdir(int s, FILE *fp, char *dir)
{
  dirlist *d, *firstd, *lastd;
  flelist *f, *firstf, *lastf;
  struct stat fstatus;
  char temp[STRLEN];
  char file[STRLEN];
  char perm[11];
  long rfsize=0;
  short b=1;
  int c, i;

  if (dir[strlen(dir)-1] == (char) 13) dir[strlen(dir)-1] = (char) 0;
  ftpchdir(s, fp, dir);
  if ((dir[0] != (char)47) && (dir[1] != (char)0)) {
    if (mkdir(dir, 0775) < 0) (void) fprintf(stderr, "%s: directory exists.\n", dir);
    if (chdir(dir) < 0) { perror("chdir"); exit(-1); }
  }
  (void) fprintf(stdout, "Entering directory %s\n", dir);

  if (!(firstd=(dirlist *) malloc(sizeof(dirlist)))) { perror("malloc"); exit(-1); }
  firstd->next=NULL;
  firstd->prev=NULL;
  lastd=firstd;
  if (!(firstf=(flelist *) malloc(sizeof(flelist)))) { perror("malloc"); exit(-1); }
  firstf->next=NULL;
  firstf->prev=NULL;
  lastf=firstf;

  if ((ftpsend(s, "stat .")) < 0) { perror("send(dir)"); exit(-1); }
  fscanf(fp, "%s", temp);
  fscanf(fp, "%s", temp);	/* total n */
  while(b) {
    (void) fscanf(fp, "%s", temp);
    strcpy(perm, temp);
    for(c=2; c<=9; c++) {
      (void) fscanf(fp, "%s", temp);
      i=strcmp(temp, "Status");
      if ((i==0) || (i==13)) b=0;
      if (c==5) rfsize=atol(temp);
      if (!b) break;
    }
    strcpy(file, temp);
    if (b) {
      if ((perm[0]=='d') && (perm[7]=='r') && (perm[9]=='x') && (strcmp(file, ".") != 0) && (strcmp(file, "..") != 0)) {
        newdir(file, &firstd);
      }
      if ((perm[0]=='-') && (perm[7]=='r')) {
        newfle(file, rfsize, &firstf);
      }
    }
  }
  for(f=lastf->prev; f; f=f->prev) {
    if (stat(f->name, &fstatus) == 0) {
      if (f->size > (&fstatus)->st_size) {
        (void) fprintf(stdout, "Overwriting file %s: old size %ld - new size %ld\n", f->name, (long) (&fstatus)->st_size, f->size);
	ftpgetfle(s, fp, f->name);
      } else (void) fprintf(stdout, "%s: %ld bytes (not overwritten)\n", f->name, (long) (&fstatus)->st_size); 
    } else ftpgetfle(s, fp, f->name);
  }
  eraseflist(&firstf, &lastf);
  for(d=lastd->prev; d; d=d->prev) {
    ftpgetdir(s, fp, d->name);	  /* Iterate is human, recurse divine. */
  }
  erasedlist(&firstd, &lastd);
  if ((dir[0] != (char)47) && (dir[1] != (char)0)) {
    if (chdir("..") < 0) { perror("chdir"); exit(-1); }
    if ((ftpsend(s, "cdup")) < 0) { perror("send"); exit(-1); }
    if ((ftprecs(fp, "successful.", "directory.")) < 0)
      (void) fprintf(stderr, "%s: No such file or directory.\n", dir);
  }
}


void ftpgetfle(int s, FILE *fp, char *filename)
{
  register char *p, *a;
  int s1, len, data, on=1;
  char strcmd[STRLEN];
  struct sockaddr_in data_addr;
  struct sockaddr_in from;
  int fromlen = sizeof(from);
  FILE *rfp, *lfp;
  struct stat st;
  char *buffer;
  int bytes=0, rb=0, lb=0;
  extern FILE *fdopen(int, const char *);
#ifdef __linux__
  extern int fileno(FILE *);
#endif
  len=sizeof(data_addr);
  if ((getsockname(s, (struct sockaddr *) &data_addr, &len)) < 0) { perror("getsockname"); exit(-1); }
  data_addr.sin_port=0;
  if ((data=socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("data socket");
    return;
  }
  if (setsockopt(data, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof (on)) < 0) {
    perror("setsockopt (reuse address)");
    return;
  }
  if (bind(data, (struct sockaddr *) &data_addr, sizeof (data_addr)) < 0) {
    perror("bind");
    return;
  }
  len=sizeof(data_addr);
  if (getsockname(data, (struct sockaddr *) &data_addr, &len) < 0) {
    perror("getsockname");
    return;
  }
  if (listen(data, 1) < 0) {
    perror("listen");
    return;
  }
  a = (char *)&data_addr.sin_addr;
  p = (char *)&data_addr.sin_port;
  #define UC(b) (((int)b)&0xff)
  sprintf(strcmd, "PORT %d,%d,%d,%d,%d,%d", UC(a[0]), UC(a[1]), UC(a[2]), UC(a[3]), UC(p[0]), UC(p[1]));
  #undef UC
  if ((ftpsend(s, strcmd)) < 0) {
    perror("port");
    return;
  }
  sprintf(strcmd, "retr %s", filename);
  if (filename[strlen(filename)-1] == (char) 13) filename[strlen(filename)-1] = (char) 0;
  if ((ftpsend(s, strcmd)) < 0) { perror(strcmd); exit(-1); }
  if ((ftprecs(fp, "successful.", "understood.")) < 0) {
    (void) fprintf(stderr, "Port command failed: unable to receive file %s\n", filename);
    return;
  }
  s1 = accept(data, (struct sockaddr *) &from, &fromlen);
  if (s1 < 0) {
    perror("accept");
    (void) close(data);
    return;
  }
  (void) close(data);
  data = s1;
  rfp=fdopen(data, "rb");
  lfp=fopen(filename, "wb");
  if ((fstat(fileno(lfp), &st) < 0) || (st.st_blksize == 0)) st.st_blksize = BUFSIZ;
  if (!(buffer = (char *) malloc((unsigned int) st.st_blksize))) { perror("malloc"); exit(-1); }
  errno=0;
  (void) fprintf(stdout, "%s: ", filename);
  while((rb=read(fileno(rfp), buffer, st.st_blksize)) > 0) {
    if ((lb=write(fileno(lfp), buffer, rb)) != rb) break;
    bytes+=rb;
  }
  if (rb<0) {
    if (errno != EPIPE) perror("net");
    bytes=0;
  }
  if (lb<rb) {
    if (lb<0) perror(filename);
    else (void) fprintf(stderr, "short file\n");
  }
  (void) fprintf(stdout, "%d bytes.\n", bytes);
  (void) fclose(lfp);
  (void) fclose(rfp);  
  (void) close(data);
}


void newdir(const char *file, dirlist **first)
{
  dirlist *dnew, *right;
  if (!(dnew=(dirlist *) malloc(sizeof(dirlist)))) { perror("malloc"); exit(-1); }
  right=*first;
  (*first)->prev=dnew;
  *first=dnew;
  dnew->prev=NULL;
  dnew->next=right;
  if (!(dnew->name=(char *) malloc(strlen(file)+1))) { perror("malloc"); exit(-1); }
  strcpy(dnew->name, file);
}


void newfle(const char *file, long rfsize, flelist **first)
{
  flelist *fnew, *right;
  if (!(fnew=(flelist *) malloc(sizeof(flelist)))) { perror("malloc"); exit(-1); }
  right=*first;
  (*first)->prev=fnew;
  *first=fnew;
  fnew->prev=NULL;
  fnew->next=right;
  if (!(fnew->name=(char *) malloc(strlen(file)+1))) { perror("malloc"); exit(-1); }
  strcpy(fnew->name, file);
  fnew->size=rfsize;
}


void erasedlist(dirlist **first, dirlist **last)
{
  dirlist *t;
  t=*first;
  while (t!=*last) {
    t=t->next;
    free((*first)->name);
    free(*first);
    *first=t;
  }
}


void eraseflist(flelist **first, flelist **last)
{
  flelist *t;
  t=*first;
  while (t!=*last) {
    t=t->next;
    free((*first)->name);
    free(*first);
    *first=t;
  }
}


void fatal(int r)
{
  signal(SIGINT, SIG_DFL);
  (void) fprintf(stderr, "Program aborted.\n");
  exit(r);
}

