
/*

    File: sync.c

    Copyright (C) 2003  Wolfgang Zekoll  <wzk@happy-ent.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>

#include <time.h>
#include <signal.h>
#include <wait.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <sys/time.h>
#include <dirent.h>

#include "ftp.h"
#include "sync.h"
#include "ip-lib.h"
#include "lib.h"


static int getstringcode(char *string)
{
	int	c, i, code;

	code = 0;
	for (i=0; (c = string[i]) != 0; i++)
		code = code + c;

	code = code + (i << 8);
	return (code);
}

static int checkfilename(sync_t *s, char *filename)
{
	int	c, i;

	if (strcmp(filename, ".") == 0  ||  strcmp(filename, "..") == 0)
		return (0);

	if (strchr(filename, ':') != NULL)
		return (0);

	if (s->allowblanks == 0) {
		for (i=0; (c = filename[i]) != 0; i++) {
			if (c <= 32)
				return (0);
			}
		}

	if (filename[0] == '.') {
		int	c;

		if (strncmp(filename, ".sync", 5) == 0  &&
		    ((c = filename[5]) == '.'  ||  c == '-'))
			return (0);

		if (s->dotfiles == 0)
			return (0);
		}

	return (1);
}

static int comparefiles(char *file1, char *file2)
{
	int	rc, n, bytes, fd1, fd2;
	unsigned long size;
	char	b1[4096], b2[4096];
	struct stat sbuf;

	if (stat(file1, &sbuf) != 0)
		return (1);

	size = sbuf.st_size;
	if (stat(file2, &sbuf) != 0  ||  sbuf.st_size != size)
		return (1);

	rc = 0;
	fd1 = fd2 = -1;

	if ((fd1 = open(file1, O_RDONLY)) < 0  ||  (fd2 = open(file2, O_RDONLY)) < 0)
		rc = 1;

	while (rc == 0  &&  size > 0) {
		bytes = (size > 4096)? 4096: size;

		if ((n = read(fd1, b1, bytes)) != bytes)
			rc = 1;
		else if (read(fd2, b2, n) != n)
			rc = 1;
		else if (memcmp(b1, b2, n) != 0)
			rc = 1;

		size = size - n;
		}

	if (fd1 >= 0)
		close (fd1);

	if (fd2 >= 0)
		close (fd2);

	return (rc);
}

static file_t *getfile(sync_t *s, dir_t *dir, char *filename)
{
	int	i, code;
	file_t	*file;

	code = getstringcode(filename);
	for (i=0; i<dir->listc; i++) {
		file = &dir->list[i];
		if (code == file->code  &&  strcmp(filename, file->filename) == 0)
			return (file);
		}

	if (dir->listc >= dir->listmax) {
		dir->listmax += 10;
		dir->list = reallocate(dir->list, dir->listmax * sizeof(file_t));
		}

	file = &dir->list[dir->listc++];
	memset(file, 0, sizeof(file_t));
	file->code = code;
	copy_string(file->filename, filename, sizeof(file->filename));

	return (file);
}

int readprevinfo(sync_t *s, dir_t *dir, int index)
{
	char	*syncstatus;
	file_t	*file;
	FILE	*fp;

	if (index != 0)
		index = 1;

	syncstatus = dir->infofile[index];
	if ((fp = fopen(syncstatus, "r")) != NULL) {
		char	*p, filename[80], line[300];

		while (fgets(line, sizeof(line), fp) != NULL) {
			p = skip_ws(noctrl(line));
			get_word(&p, filename, sizeof(filename));

			file = getfile(s, dir, filename);
			p = skip_ws(p);
			copy_string(file->info[index].prev, p, sizeof(file->info[index].prev));
			}

		fclose (fp);
		}

	return (0);
}

int writeinfo(sync_t *s, dir_t *dir, int index)
{
	char	*syncstatus;
	FILE	*fp;

	if (index != 0)
		index = 1;

	syncstatus = dir->infofile[index];
	if ((fp = fopen(syncstatus, "w")) == NULL) {
		fprintf (stderr, "%s: can't write file: %s\n", program, syncstatus);
		return (1);
		}
	else {
		int	i;
		file_t	*file;

		for (i=0; i < dir->listc; i++) {
			file = &dir->list[i];
			if (*file->info[index].current != 0)
				fprintf (fp, "%s %s\n", file->filename, file->info[index].current);
			}

		fclose (fp);
		}

	return (0);
}


int getpeerstat(ftp_t *x, char *filename, char *stat, int bsize)
{
	char	size[80], mdtm[80];

	if (dosize(x, filename, size, sizeof(size)) != 0  ||
	    domdtm(x, filename, mdtm, sizeof(mdtm)) != 0)
		return (1);

	snprintf (stat, bsize - 2, "%s %s", size, mdtm);
	return (0);
}

int getpeerinfo(sync_t *s, dir_t *dir)
{
	int	size;
	char	*p, *nlist, filename[80], stat[80];
	file_t	*file;
	ftp_t	*x;

	x = s->server;
	nlist = NULL;
	size  = 0;
	if (dolist(x, 1, "", &nlist, &size) != 0)
		return (1);

	p = nlist;
	while (*get_quoted(&p, '\n', filename, sizeof(filename)) != 0) {
		if (checkfilename(s, filename) == 0)
			continue;

		file = getfile(s, dir, filename);
		file->info[1].exists = 1;
		if (getpeerstat(x, filename, stat, sizeof(stat)) != 0) {
			file->info[1].type = FT_DIR;	/* just guessing */
			continue;
			}

		file->info[1].type = FT_FILE;
		copy_string(file->info[1].current, stat, sizeof(file->info[1].current));
		}

	free(nlist);
	return (0);
}


static int getnodestat(sync_t *s, char *filename, char *statbuf, int bsize)
{
	char	mtime[80];
	struct tm tm;
	struct stat sbuf;

	*statbuf = 0;
	if (stat(filename, &sbuf) != 0)
		return (1);
	else if (S_ISREG(sbuf.st_mode) == 0)
		return (1);

	tm = *gmtime(&sbuf.st_mtime);
	strftime(mtime, sizeof(mtime) - 2, "%Y%m%d%H%M%S", &tm);
	snprintf (statbuf, bsize - 2, "%lu %s", sbuf.st_size, mtime);

	return (0);
}

int getnodeinfo(sync_t *s, dir_t *dir)
{
	int	i, n;
	char	*name;
	struct dirent **namelist;
	file_t	*file;

	n = scandir(".", &namelist, NULL, alphasort);
	if (n < 0) {
		fprintf (stderr, "%s: scandir error: %s", program, strerror(errno));
		return (-1);
		}

	for (i=0; i<n; i++) {
		name = (*namelist[i]).d_name;
		if (checkfilename(s, name) == 0)
			continue;

		file = getfile(s, dir, name);
		file->info[0].exists = 1;
		if (getnodestat(s, name, file->info[0].current, sizeof(file->info[0].current)) != 0) {
			struct stat sbuf;

			if (stat(name, &sbuf) == 0  &&  S_ISDIR(sbuf.st_mode))
				file->info[0].type = FT_DIR;

			continue;
			}

		file->info[0].type = FT_FILE;
		}

	for (i=0; i<n; i++)
		free (namelist[i]);

	return (0);
}

int computestatus(sync_t *s, dir_t *dir, int index)
{
	int	i;
	char	*p, *q;
	file_t	*file;

	if (index != 0)
		index = 1;

	for (i=0; i < dir->listc; i++) {
		file = &dir->list[i];
		if (file->info[index].type == FT_DIR)
			continue;

		p = file->info[index].prev;
		q = file->info[index].current;

		if (*p != 0  &&  *q == 0)
			file->info[index].status = ST_DELETED;
		else if (*p == 0  &&  *q != 0)
			file->info[index].status = ST_CHANGED;
		else if (*p == 0  &&  *q == 0)
			file->info[index].status = ST_NOSUCHFILE;
		else if (strcmp(p, q) == 0)
			file->info[index].status = ST_UNCHANGED;
		else
			file->info[index].status = ST_CHANGED;

/*		if (index == 0)
 *			printf ("%s %s %s %d\n", file->filename, p, q, file->info[index].status);
 */

		}

	return (0);
}


int computeaction(sync_t *s, dir_t *dir)
{
	int	i;
	file_t	*file;

	for (i=0; i < dir->listc; i++) {
		file = &dir->list[i];
		if (file->info[0].type == FT_DIR  ||  file->info[1].type == FT_DIR) {
			if (file->info[0].type == FT_FILE  ||  file->info[1].type == FT_FILE)
				printf ("M %s\n", file->filename);

			file->sync = 0;
			continue;
			}

		file->sync = 1;
		}


	for (i=0; i < dir->listc; i++) {
		file = &dir->list[i];
		if (file->sync == 0)
			continue;

		file->action = s->synctab[file->info[0].status][file->info[1].status];
		}

	return (0);
}


int syncfiles(sync_t *s, dir_t *dir)
{
	int	i, n;
	file_t	*file;

	n = 0;
	for (i=0; i < dir->listc; i++) {
		file = &dir->list[i];

		if (file->sync == 0) {

			/*
			 * Invalidate entry for following writeinfo().
			 */

			*file->info[0].current = 0;
			*file->info[1].current = 0;
			continue;
			}

		n++;
		if (verbose != 0  ||
		    (file->action != AC_NOTHING  &&  file->action != AC_IGNORE)) {
			printf ("%c%c%c %s\n",
				statchar(dir->list[i].info[0].status),
				statchar(dir->list[i].info[1].status),
				actionchar(dir->list[i].action),
				dir->list[i].filename);
			}

		if (s->printonly != 0)
			continue;

		if (file->action == AC_NOTHING  ||  file->action == AC_IGNORE)
			;		/* do nothing */
		else if (file->action == AC_GET) {
			if (doretr(s->server, file->filename, file->filename) != 0)
				return (1);
			else if (getnodestat(s, file->filename, file->info[0].current, sizeof(file->info[0].current)) != 0)
				return (1);
			}
		else if (file->action == AC_PUT) {
			if (dostor(s->server, file->filename, file->filename) != 0)
				return (1);
			else if (getpeerstat(s->server, file->filename, file->info[1].current, sizeof(file->info[1].current)) != 0)
				return (1);
			}
		else if (file->action == AC_DUPLICATE) {
			char	filename[120];

			snprintf (filename, sizeof(filename) - 2, "%s:%s", file->filename, s->peername);
			if (doretr(s->server, file->filename, filename) != 0)
				return (1);

			if (comparefiles(file->filename, filename) == 0) {
				printf ("uu= %s\n", dir->list[i].filename);
				unlink(filename);
				}
			else {
				snprintf (filename, sizeof(filename) - 2, "%s:%s", file->filename, s->nodename);
				if (dostor(s->server, filename, file->filename) != 0)
					return (1);
				}
			}
		else if (file->action == AC_REMOVE) {
			if (*file->info[0].current != 0)
				unlink(file->filename);

			if (*file->info[1].current != 0)
				dodele(s->server, file->filename, 0);

			*file->info[0].current = 0;
			*file->info[1].current = 0;
			}
		else
			fprintf (stderr, "%s: unhandled action: %d\n", program, file->action);
		}

	return (0);
}

int createdirs(sync_t *s, dir_t *dir)
{
	int	i, rc, ldir, rdir;
	file_t	*file;
	struct stat sbuf;

	for (i=0; i < dir->listc; i++) {
		file = &dir->list[i];
		if (file->info[0].type == FT_FILE  ||  file->info[1].type == FT_FILE)
			continue;

		if (stat(file->filename, &sbuf) != 0)
			ldir = 0;	/* doesn't exist */
		else if (S_ISDIR(sbuf.st_mode) == 0)
			continue;	/* exists but not a directory */
		else
			ldir = 1;

		if (file->info[1].exists == 0)
			rdir = 0;
		else {
			rc = docheckdir(s->server, file->filename);
			if (rc < 0) {
				fprintf (stderr, "%s: can't check directory: %s\n", program, file->filename);
				exit (1);
				}
			else if (rc != 0)
				continue;	/* exists but not a directory */
			else
				rdir = 1;
			}

		if (ldir == 1  &&  rdir == 1)
			continue;		/* nothing to do here */
		else if (ldir == 0  &&  rdir == 0)
			continue;		/* same as above */

		if (s->modeid == MODE_SYNC) {
			if (ldir == 0)
				ldir = 2;
			else
				rdir = 2;
			}
		else if (s->modeid == MODE_MASTER  ||  s->modeid == MODE_ORIGINAL) {
			if (ldir == 1  &&  rdir == 0)
				rdir = 2;
			}
		else if (s->modeid == MODE_SLAVE  ||  s->modeid == MODE_MIRROR) {
			if (ldir == 0  &&  rdir == 1)
				ldir = 2;
			}

		if (ldir == 2) {
			printf ("> creating local directory: %s\n", file->filename);

			if (s->printonly != 0)
				/* nothing */ ;
			else if (mkdir(file->filename, 0777) == 0)
				file->info[0].type = FT_DIR;
			else {
				fprintf (stderr, "%s: can't create directory: %s\n", program, file->filename);
				continue;
				}
			}

		if (rdir == 2) {
			printf ("< creating remote directory: %s\n", file->filename);

			if (s->printonly != 0)
				/* nothing */ ;
			else if (domkdir(s->server, file->filename) == 0)
				file->info[1].type = FT_DIR;
			else {
				continue;
				}
			}
		}

	return (0);
}


int syncdir(sync_t *s, char *basedir, char *dirname)
{
	int	i, rc;
	char	thisdir[200], nodedir[200], peerdir[200];
	file_t	*file;
	dir_t	*dir;

	rc = 0;
	dir = allocate(sizeof(dir_t));

	snprintf (dir->infofile[0], sizeof(dir->infofile[0]) - 2, ".sync-%s:%s", s->nodename, s->peername);
	snprintf (dir->infofile[1], sizeof(dir->infofile[1]) - 2, ".sync-%s:%s", s->peername, s->nodename);

	snprintf (thisdir, sizeof(thisdir) - 2, "%s%s%s",
			basedir, (*basedir != 0)? "/": "", dirname);

	if (*dirname != 0) {
		if (getcwd(nodedir, sizeof(nodedir)) == NULL  ||
		    dopwd(s->server, peerdir, sizeof(peerdir)) != 0)
			return (2);

		if (chdir(dirname) != 0)
			return (1);
		else if (docwd(s->server, dirname, 0, 0) != 0) {
			if (chdir(nodedir) != 0)
				return (2);

			return (1);
			}

		printf ("\n");
		printf ("* %s\n", thisdir);
		}

	readprevinfo(s, dir, 1);
	readprevinfo(s, dir, 0);

	if (getpeerinfo(s, dir) != 0)
		rc = 1;
	else {
		computestatus(s, dir, 1);

		getnodeinfo(s, dir);
		computestatus(s, dir, 0);

		computeaction(s, dir);
		syncfiles(s, dir);

		if (s->printonly == 0  ||  s->forceupdate != 0) {
			writeinfo(s, dir, 0);
			writeinfo(s, dir, 1);

			if (s->symsync != 0) {
				dostor(s->server, dir->infofile[0], dir->infofile[1]);
				dostor(s->server, dir->infofile[1], dir->infofile[0]);
				}
			}
		}

	if (s->recurse != 0) {
		if (s->createdirs != 0)
			createdirs(s, dir);

		for (i=0; i < dir->listc; i++) {
			file = &dir->list[i];

			if (file->sync == 0  &&
			    file->info[0].type == FT_DIR  &&  file->info[1].type == FT_DIR) {
				rc = syncdir(s, thisdir, file->filename);
				if (rc == 2)
					return (rc);

				rc = 0;
				}
			}
		}

	free(dir->list);
	free(dir);

	if (*dirname != 0) {
		if (chdir(nodedir) != 0  ||
		    docwd(s->server, peerdir, 0, 1) != 0)
			return (2);
		}

	return (rc);
}

