
/*

    File: main.c

    Copyright (C) 2003,2004  Wolfgang Zekoll  <wzk@happy-ent.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>

#include <signal.h>
#include <wait.h>
#include <pwd.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <sys/time.h>

#include "ftp.h"
#include "sync.h"
#include "ip-lib.h"
#include "lib.h"


char	*program =		"";
char	progname[80] =		"";

int	debug =			0;
int	verbose =		1;
int	ignorenetrc =		0;




struct _syncdata {
    struct {
    	char	name[20];
	int	id;
	int	synctab[4][4];
	} mode;
    } syncdata[] = {
    	{
	  { "sync", MODE_SYNC,
	    {
		{ AC_NOTHING,  AC_GET,       AC_REMOVE,    AC_PUT },
		{ AC_PUT,      AC_DUPLICATE, AC_PUT,       AC_PUT },
		{ AC_REMOVE,   AC_GET,       AC_IGNORE,    AC_IGNORE },
		{ AC_GET,      AC_GET,       AC_IGNORE,    AC_IGNORE }
	    }
	  }
	},

	{
	  { "master", MODE_MASTER,
	    {
		{ AC_NOTHING,  AC_GET,       AC_PUT,       AC_PUT },
		{ AC_PUT,      AC_PUT,       AC_PUT,       AC_PUT },
		{ AC_REMOVE,   AC_GET,       AC_IGNORE,    AC_IGNORE },
		{ AC_GET,      AC_GET,       AC_IGNORE,    AC_IGNORE }
	    }
	  }
	},

    	{
	  { "slave", MODE_SLAVE,
	    {
		{ AC_NOTHING,  AC_GET,       AC_REMOVE,    AC_PUT },
		{ AC_PUT,      AC_GET,       AC_PUT,       AC_PUT },
		{ AC_GET,      AC_GET,       AC_IGNORE,    AC_IGNORE },
		{ AC_GET,      AC_GET,       AC_IGNORE,    AC_IGNORE }
	    }
	  }
	},

    	{
	  { "mirror", MODE_MIRROR,
	    {
		{ AC_NOTHING,  AC_GET,       AC_REMOVE,    AC_IGNORE },
		{ AC_GET,      AC_GET,       AC_REMOVE,    AC_IGNORE },
		{ AC_GET,      AC_GET,       AC_IGNORE,    AC_IGNORE },
		{ AC_GET,      AC_GET,       AC_IGNORE,    AC_IGNORE }
	    }
	  }
	},

	{
	  { "original", MODE_ORIGINAL,
	    {
		{ AC_NOTHING,  AC_PUT,       AC_PUT,       AC_PUT },
		{ AC_PUT,      AC_PUT,       AC_PUT,       AC_PUT },
		{ AC_REMOVE,   AC_REMOVE,    AC_IGNORE,    AC_IGNORE },
		{ AC_IGNORE,   AC_IGNORE,    AC_IGNORE,    AC_IGNORE }
	    }
	  }
	},

	{
	  { "", }
	}
    };
 


int getsyncmode(char *mode)
{
	int	i;

	for (i=0; syncdata[i].mode.name[0] != 0; i++) {
		if (strcmp(syncdata[i].mode.name, mode) == 0)
			return (i);
		}

	fprintf (stderr, "%s: unknown sync mode: %s\n", program, mode);
	exit (1);

	return (0);
}


int printsynctable(sync_t *s)
{
	int	i, j;

	for (i=0; i < 4; i++) {
		for (j=0; j < 4; j++) {
			printf (" %c%c%c",
				statchar(i), statchar(j),
				actionchar(s->synctab[i][j]));
			}

		printf ("\n");
		}

	return (0);
}


int readconfig(sync_t *s, char *peer)
{
	char	*p, word[80], line[300], filename[200];
	FILE	*fp;

	if (*s->configfile == 0) {
		if (*peer == 0)
			strcpy(filename, ".sync.conf");
		else
			snprintf (filename, sizeof(filename) - 2, ".sync-%s.conf", peer);
		}
	else {
		if (*peer == 0)
			copy_string(filename, s->configfile, sizeof(filename));
		else {
			char	ext[80];

			if ((p = strrchr(s->configfile, '.')) != NULL  &&  strcmp(p, ".conf") == 0) {
				copy_string(ext, p, sizeof(ext));
				*p = 0;
				}

			snprintf (filename, sizeof(filename) - 2, "%s-%s%s", s->configfile, peer, ext);
			}
		}

	copy_string(s->configfile, filename, sizeof(s->configfile));
	if ((fp = fopen(filename, "r")) == NULL) {
		fprintf (stderr, "%s: can't open file: %s\n", program, filename);
		exit (1);
		}

	while (fgets(line, sizeof(line), fp) != NULL) {
		if ((p = strchr(line, '#')) != NULL)
			*p = 0;

		p = skip_ws(noctrl(line));
		if (*p == 0)
			continue;

		get_word(&p, word, sizeof(word));
		p = skip_ws(p);

		strlwr(word);
		if (strcmp(word, "nodename") == 0)
			get_word(&p, s->nodename, sizeof(s->nodename));
		else if (strcmp(word, "peername") == 0)
			get_word(&p, s->peername, sizeof(s->peername));
		else if (strcmp(word, "syncmode") == 0) {
			if (s->syncmode < 0)
				s->syncmode = getsyncmode(p);
			}
		else if (strcmp(word, "symsync") == 0) {
			if (s->symsync < 0) {
				get_word(&p, word, sizeof(word));
				strlwr(word);
				if (strcmp(word, "yes") == 0)
					s->symsync = 1;
				}
			}
		else if (strcmp(word, "recurse") == 0) {
			if (s->recurse < 0) {
				get_word(&p, word, sizeof(word));
				strlwr(word);
				if (strcmp(word, "yes") == 0)
					s->recurse = 1;
				}
			}
		else if (strcmp(word, "createdirs") == 0) {
			get_word(&p, word, sizeof(word));
			strlwr(word);
			if (strcmp(word, "yes") == 0)
				s->createdirs = 1;
			}
		else if (strcmp(word, "dotfiles") == 0) {
			if (s->dotfiles < 0) {
				get_word(&p, word, sizeof(word));
				strlwr(word);
				if (strcmp(word, "yes") == 0)
					s->dotfiles = 1;
				}
			}
		else if (strcmp(word, "allow-blanks") == 0) {
			if (s->allowblanks < 0) {
				get_word(&p, word, sizeof(word));
				strlwr(word);
				if (strcmp(word, "yes") == 0)
					s->allowblanks = 1;
				}
			}
		else if (strcmp(word, "server") == 0)
			get_word(&p, s->server->server, sizeof(s->server->server));
		else if (strcmp(word, "login") == 0) {
			if (*s->server->username == 0)
				get_word(&p, s->server->username, sizeof(s->server->username));
			}
		else if (strcmp(word, "password") == 0) {
			struct stat sbuf;

			if (stat(filename, &sbuf) != 0  ||  (sbuf.st_mode & 0x077) != 0) {
				fprintf (stderr, "%s: wrong permissions on %s\n", program, filename);
				exit (1);
				}

			if (*s->server->password == 0)
				get_word(&p, s->server->password, sizeof(s->server->password));
			}
		else if (strcmp(word, "dir") == 0) {
			if (*s->dir == 0)
				get_word(&p, s->dir, sizeof(s->dir));
			}
		else if (strcmp(word, "localdir") == 0)
			get_word(&p, s->localdir, sizeof(s->localdir));
		else if (strcmp(word, "passive") == 0) {
			get_word(&p, word, sizeof(word));
			strlwr(word);
			if (strcmp(word, "yes") == 0)
				s->server->usepassive = 1;
			}
		else {
			fprintf (stderr, "%s: unknown configuration option: %s\n", program, word);
			exit (1);
			}
		}

	return (0);
}


int checkconfig(sync_t *s)
{
	ftp_t	*x;

	if (*s->nodename == 0) {
		fprintf (stderr, "%s: empty or unset nodename\n", program);
		exit (1);
		}
	else if (*s->peername == 0) {
		fprintf (stderr, "%s: empty or unset peername\n", program);
		exit (1);
		}

	if (*s->localdir != 0) {
		if (chdir(s->localdir) != 0) {
			fprintf (stderr, "%s: can't change directory: %s, error= %s\n",
					program, s->localdir, strerror(errno));
			exit (1);
			}
		}

	if (s->syncmode < 0)
		s->syncmode = 0;

	memmove(s->synctab, syncdata[s->syncmode].mode.synctab, sizeof(s->synctab));
	s->modeid = syncdata[s->syncmode].mode.id;

	s->recurse     = (s->recurse < 0)? 0: s->recurse;
	s->dotfiles    = (s->dotfiles < 0)? 0: s->dotfiles;
	s->allowblanks = (s->allowblanks < 0)? 0: s->allowblanks;
	s->symsync     = (s->symsync < 0)? 0: s->symsync;

	if (s->recurse == 0)
		s->createdirs = 0;

	x = s->server;
	if (*x->server == 0) {
		fprintf (stderr, "%s: empty or unset servername\n", program);
		exit (1);
		}

	if (*x->password == 0) {
		if (ignorenetrc == 0) {
			read_netrc(x->server, x->username, sizeof(x->username),
					x->password, sizeof(x->password));
			}

		if (*x->password == 0  &&  isatty(0)) {
			char	*p;

			p = getpass("password: ");
			copy_string(x->password, p, sizeof(x->password));
			}
		}

	if (strcmp(x->password, "-") == 0)
		*x->password = 0;

	return (0);
}


void missing_arg(int c, char *string)
{
	fprintf (stderr, "%s: missing arg: -%c, %s\n", program, c, string);
	exit (-1);
}

int main(int argc, char *argv[], char *envp[])
{
	int	c, i, k;
	char	*p, option[80], peer[80];
	sync_t	*s;
	

	if ((p = strrchr(argv[0], '/')) == NULL)
		program = argv[0];
	else {
		copy_string(progname, &p[1], sizeof(progname));
		program = progname;
		}

	s = allocate(sizeof(sync_t));
	s->server = allocate(sizeof(ftp_t));
	s->syncmode = s->recurse = s->dotfiles = s->symsync = -1;

	k = 1;
	while (k < argc  &&  argv[k][0] == '-'  &&  argv[k][1] != 0) {
		copy_string(option, argv[k++], sizeof(option));
		for (i=1; (c = option[i]) != 0; i++) {
			if (c == 'd')
				debug = 1;
			else if (c == 'b')
				s->symsync = 1;
			else if (c == 'f') {
				if (k >= argc)
					missing_arg(c, "configuration file");

				copy_string(s->configfile, argv[k++], sizeof(s->configfile));
				}
			else if (c == 'i')
				ignorenetrc = 1;
			else if (c == 'l') {
				if (k >= argc)
					missing_arg(c, "username");

				p = argv[k++];
				if (*p == ':') {
					p++;
					copy_string(s->server->password, p, sizeof(s->server->password));
					}
				else {
					get_quoted(&p, ':', s->server->username, sizeof(s->server->username));
					if (*p != 0)
						copy_string(s->server->password, p, sizeof(s->server->password));
					}
				}
			else if (c == 'm') {
				if (k >= argc)
					missing_arg(c, "syncmode");

				s->syncmode = getsyncmode(argv[k++]);
				}
			else if (c == 'q') {
				if (s->printonly == 0)
					s->printonly = 1;
				else
					s->printonly = 2;
				}
			else if (c == 'Q') {
				s->printonly = 1;
				s->forceupdate = 1;
				}
			else if (c == 'r') {
				if (s->recurse == 1)
					s->createdirs = 1;

				s->recurse = 1;
				}
			else if (c == 's')
				verbose = 0;
			else if (c == 'z') {
				if (k >= argc)
					missing_arg(c, "buffer size");

				s->server->bsize = atoi(argv[k++]);
				}
			else if (c == 'V') {
				printf ("%s %s\n", program, VERSION);
				exit (0);
				}
			else {
				fprintf (stderr, "%s: unknown option: -%c\n", program, c);
				exit (-1);
				}
			}
		}



	if (s->printonly == 2) {
		if (s->syncmode < 0)
			s->syncmode = 0;

		memmove(s->synctab, syncdata[s->syncmode].mode.synctab, sizeof(s->synctab));
		printsynctable(s);
		exit (0);
		}

	*peer = 0;
	if (k < argc)
		copy_string(peer, argv[k++], sizeof(peer));

	readconfig(s, peer);
	checkconfig(s);

	if (dologin(s->server) != 0)
		exit (1);

	if (*s->dir != 0) {
		if (docwd(s->server, s->dir, s->printonly == 0, 1) != 0)
			exit (1);
		}

	syncdir(s, "", "");
	doquit(s->server);

	exit (0);
}


