
/*

    File: ftp.c

    Copyright (C) 1999,2003  Wolfgang Zekoll  <wzk@happy-ent.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <errno.h>

#include <time.h>
#include <signal.h>
#include <wait.h>
#include <pwd.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <sys/time.h>

#include "ftp.h"
#include "ip-lib.h"
#include "lib.h"



int read_netrc(char *hostname, char *username, int usize,
			char *password, int psize)
{
	int	is_user, is_server;
	char	home[200], filename[200];
	char	*p, token[80], par[80], line[200];
	struct passwd *pw;
	struct stat sbuf;
	FILE	*fp;

	if ((pw = getpwuid(getuid())) == NULL)
		return (1);

	copy_string(home, pw->pw_dir, sizeof(home));
	snprintf (filename, sizeof(filename) - 2, "%s/.netrc", home);

	*password = 0;
	if (stat(filename, &sbuf) != 0)
		return (0);
	else if ((sbuf.st_mode & 0x077) != 0) {
		fprintf (stderr, "%s: wrong permission on %s\n", program, filename);
		return (1);
		}

	if ((fp = fopen(filename, "r")) == NULL)
		return (0);


	is_server = is_user = 0;
	while (fgets(line, sizeof(line), fp) != NULL) {
		p = skip_ws(noctrl(line));
		if (*p == 0  ||  *p == '#')
			continue;

		get_word(&p, token, sizeof(token));
		get_word(&p, par, sizeof(par));
		if (strcmp(token, "machine") == 0) {
			is_server = (strcasecmp(par, hostname) == 0);
			continue;
			}

		if (is_server != 0) {
			if (strcmp(token, "login") == 0) {
				is_user = 0;
				if (*username != 0)
					is_user = (strcmp(par, username) == 0)? 1: 0;
				else {
					copy_string(username, par, usize);
					is_user = 1;
					}

				continue;
				}

			if (is_user != 0) {
				if (strcmp(token, "password") == 0) {
					copy_string(password, par, psize);
					break;
					}
				}
			}
		}

	fclose (fp);
	return (0);
}


unsigned get_interface_info(int pfd, char *ip, int max)
{
	int	size;
	unsigned int port;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getsockname(pfd, (struct sockaddr *) &saddr, &size) < 0) {
		syslog(LOG_NOTICE, "-ERR: can't get interface info: %m");
		exit (1);
		}

	copy_string(ip, (char *) inet_ntoa(saddr.sin_addr), max);
	port = ntohs(saddr.sin_port);

	return (port);
}


int waitfd(ftp_t *x, int fd, int write, int timeout)
{
	int	rc;
	fd_set	*rs, *ws, fdset, available;
	struct timeval tv;

	while (1) {
		FD_ZERO(&fdset);
		FD_SET(fd, &fdset);
		memmove(&available, &fdset, sizeof(fd_set));
		tv.tv_sec  = timeout;
		tv.tv_usec = 0;

		if (write == 0) {
			rs = NULL;
			ws = &available;
			}
		else {
			rs  = &available;
			ws = NULL;
			}

		if ((rc = select(fd + 1, rs, ws, NULL, &tv)) < 0) {
			if (errno == EINTR)
				continue;

			fprintf (stderr, "%s: select error: %s\n", program, strerror(errno));
			exit (1);
			}
		else if (rc == 0) {
			fprintf (stderr, "%s: select timeout\n", program);
			exit (1);
			}

		if (FD_ISSET(fd, &available) == 0) {
			fprintf (stderr, "%s: select fd error\n", program);
			exit (1);
			}

		break;
		}

	return (0);
}

int getc_fd(ftp_t *x, int fd)
{
	int	c;
	static int here = 0, len = 0;
	static char buffer[1024];

	if (here >= len) {
		int	bytes;

		len = here = 0;
		if (x->timeout == 0)
			x->timeout = 60;

		waitfd(x, fd, 0, x->timeout);
		if ((bytes = read(fd, buffer, sizeof(buffer) - 2)) <= 0)
			return (-1);

		len  = bytes;
		here = 0;
		}

	if (here >= len)
		return (-1);

	c = (unsigned char) buffer[here++];
	return (c);
}

char *readline_fd(ftp_t *x, int fd, char *line, int size)
{
	int	c, k;

	*line = 0;
	size = size - 2;

	c = getc_fd(x, fd);
	if (c < 0)
		return (NULL);

	k = 0;
	while (c > 0  &&  c != '\n'  &&  c != 0) {
		if (k < size)
			line[k++] = c;

		c = getc_fd(x, fd);
		}

	line[k] = 0;
	noctrl(line);

	return (line);
}


char *sfgets(ftp_t *x, char *line, int size)
{
	char *p;

	*line = 0;
	if ((p = readline_fd(x, x->control, line, size)) == NULL)
		return (NULL);
	else if (debug != 0)
		fprintf (stderr, "<<< SVR: %s\n", p);

	return (line);
}

int sfputs(ftp_t *x, char *format, ...)
{
	char	buffer[310];
	va_list	ap;

	va_start(ap, format);
	vsnprintf (buffer, sizeof(buffer) - 10, format, ap);
	va_end(ap);

	if (debug)
		fprintf (stderr, ">>> SVR: %s\n", buffer);

	strcat(buffer, "\r\n");
	write(x->control, buffer, strlen(buffer));

	return (0);
}

int sfputc(ftp_t *x, char *command, char *parameter, char *line, int size, char **here)
{
	int	rc;
	char	*p, buffer[300];

	if (command != NULL  &&  *command != 0) {
		if (parameter != NULL  &&  *parameter != 0)
			snprintf (buffer, sizeof(buffer) - 2, "%s %s", command, skip_ws(parameter));
		else
			copy_string(buffer, command, sizeof(buffer));

		sfputs(x, buffer);
		}
	
	if (sfgets(x, line, size) == NULL)
		return (-1);

	rc = strtol(line, &p, 10);
	if (*p == '-') {
		while (1) {
			if (sfgets(x, line, size) == NULL)
				return (-1);

			p = noctrl(line);
			if (*p == ' ')
				/* weiter */ ;
			else if (atoi(line) == rc) {
				if (strlen(line) == 3)
					break;
				else if (strlen(line) > 3  &&  line[3] == ' ')
					break;
				}
			}
		}

	p = skip_ws(p);
	if (here != NULL)
		*here = p;

	return (rc);
}




int dotype(ftp_t *x, char *type)
{
	int	rc;
	char	*p, line[200];

	x->ascii = (*type == 'A')? 1: 0;
	if ((rc = sfputc(x, "TYPE", type, line, sizeof(line), &p)) != 200) {
		fprintf (stderr, "%s: can't set transfer type: %s\n", program, line);
		return (rc);
		}

	return (0);
}

int doport(ftp_t *x)
{
	unsigned int port;
	int	c, i, rc;
	char	*p, ipnum[200], line[200];

	if (x->usepassive != 0) {
		if (sfputc(x, "PASV", "", line, sizeof(line), &p) != 227)
			x->usepassive = 0;
		else {
			char	*part[10];

			while ((c = *p) != 0) {
				if (isdigit(c) != 0)
					break;

				p++;
				}

			if (*p == 0  ||  split(p, part, ',', 10) != 6)
				x->usepassive = 0;
			else {
				snprintf (x->pasv.ipnum, sizeof(x->pasv.ipnum), "%s.%s.%s.%s",
					part[0], part[1], part[2], part[3]);
				x->pasv.port = (atoi(part[4]) << 8) + atoi(part[5]);
				return (0);
				}
			}
		}

	get_interface_info(x->control, ipnum, sizeof(ipnum));
	x->data.fd = bind_to_port(ipnum, 0);
	port = get_interface_info(x->data.fd, ipnum, sizeof(ipnum));
	snprintf (line, sizeof(line) - 2, "%s,%d,%d", ipnum, (port >> 8) & 0XFF, port & 0XFF);

	for (i=0; (c = line[i]) != 0; i++) {
		if (c == '.')
			line[i] = ',';
		}

	if ((rc = sfputc(x, "PORT", line, line, sizeof(line), &p)) != 200) {
		fprintf (stderr, "%s: port not accepted: %s\n", program, line);
		return (rc);
		}

	return (0);
}

int connect_server(ftp_t *x)
{
	if (x->usepassive != 0)
		x->data.fd = openip(x->pasv.ipnum, x->pasv.port);
	else {
		int	sock, adrlen;
		struct sockaddr_in adr;

		adrlen = sizeof(struct sockaddr);
		sock = accept(x->data.fd, (struct sockaddr *) &adr, &adrlen);
		if (sock < 0) {
			fprintf (stderr, "%s: accept error: %s\n", program, strerror(errno));
			return (1);
			}

		close(x->data.fd);
		x->data.fd = sock;
		}

	return (0);
}

int doretr(ftp_t *x, char *remote, char *local)
{
	int	rc, bytes;
	char	*p, line[200], buffer[MAXBSIZE+10];
	FILE	*fp;

	if ((rc = dotype(x, x->ascii != 0? "A": "I")) != 0  ||  (rc = doport(x)) != 0)
		return (rc);

	if ((fp = fopen(local, "w")) == NULL) {
		fprintf (stderr, "%s: can't open file: %s\n", program, local);
		return (-1);
		}

	if ((rc = sfputc(x, "RETR", remote, line, sizeof(line), &p)) != 150) {
		fprintf (stderr, "%s: can't retrieve data: %s\n", program, line);
		fclose (fp);
		return (rc);
		}

	if ((rc = connect_server(x)) != 0) {
		fclose (fp);
		return (rc);
		}

	while (1) {
		waitfd(x, x->data.fd, 0, x->timeout);
		if ((bytes = read(x->data.fd, buffer, x->bsize)) > 0) {
			if (x->ascii == 0)
				fwrite(buffer, sizeof(char), bytes, fp);
			else {
				int	c, i, k;

				k = 0;
				for (i=0; i<bytes; i++) {
					if ((c = buffer[i]) != '\r')
						buffer[k++] = c;
					}

				fwrite(buffer, sizeof(char), k, fp);
				}
			}
		else if (bytes == 0)
			break;
		else if (bytes < 0) {
			fprintf (stderr, "%s: broken data connection: %s\n", program, strerror(errno));
			break;
			}
		}

	close(x->data.fd);
	fclose(fp);

	sfputc(x, NULL, NULL, line, sizeof(line), NULL);
	if ((rc = atoi(line)) != 226) {
		fprintf (stderr, "%s: data transfer error: %s\n", program, line);
		return (rc);
		}

	return (0);
}

int dostor(ftp_t *x, char *remote, char *local)
{
	int	rc, bytes;
	char	*p, line[200], buffer[MAXBSIZE+10];
	fd_set	fdset, available;
	struct timeval tv;
	FILE	*fp;

	if ((rc = dotype(x, x->ascii != 0? "A": "I")) != 0  ||  (rc = doport(x)) != 0)
		return (rc);

	if ((fp = fopen(local, "r")) == NULL) {
		fprintf (stderr, "%s: can't open file: %s\n", program, local);
		return (-1);
		}

	if ((rc = sfputc(x, "STOR", remote, line, sizeof(line), &p)) != 150) {
		fprintf (stderr, "%s: can't store data: %s\n", program, line);
		fclose (fp);
		return (rc);
		}

	if ((rc = connect_server(x)) != 0) {
		fclose (fp);
		return (rc);
		}

	FD_ZERO(&fdset);
	FD_SET(x->data.fd, &fdset);
	rc = 0;
	while (1) {
		memmove(&available, &fdset, sizeof(fd_set));
		tv.tv_sec  = x->timeout;
		tv.tv_usec = 0;

		if ((rc = select(x->data.fd + 1, NULL, &available, NULL, &tv)) < 0) {
			if (errno == EINTR)
				continue;

			fprintf (stderr, "%s: select error: %s\n", program, strerror(errno));
			rc = -1;
			break;
			}
		else if (rc == 0) {
			rc = -1;
			fprintf (stderr, "%s: select timeout\n", program);
			break;
			}

		if (FD_ISSET(x->data.fd, &available)) {
			if ((bytes = fread(buffer, sizeof(char), x->bsize, fp)) <= 0) {
				if (bytes < 0) {
					fprintf (stderr, "%s: broken data connection\n", program);
					rc = -1;
					}

				break;
				}
			else if (write(x->data.fd, buffer, bytes) != bytes) {
				rc = -1;
				fprintf (stderr, "%s: data write error\n", program);
				break;
				}
			}
		}

	close (x->data.fd);
	fclose (fp);
	if (0  &&  rc != 0)
		return (rc);

	sfputc(x, NULL, NULL, line, sizeof(line), NULL);
	if ((rc = atoi(line)) != 226) {
		fprintf (stderr, "%s: data transfer error: %s\n", program, line);
		return (rc);
		}

	return (0);
}

int domkdir(ftp_t *x, char *dirname)
{
	int	rc;
	char	*p, line[200];

	if ((rc = sfputc(x, "MKD", dirname, line, sizeof(line), &p)) != 257) {
		fprintf (stderr, "%s: can't create directory: %s, dir= %s\n", program, line, dirname);
		return (rc);
		}

	return (0);
}

int dormdir(ftp_t *x, char *dirname)
{
	int	rc;
	char	*p, line[200];

	if ((rc = sfputc(x, "RMD", dirname, line, sizeof(line), &p)) != 250) {
		fprintf (stderr, "%s: can't remove directory: %s, dir= %s\n", program, line, dirname);
		return (rc);
		}

	return (0);
}

int dorename(ftp_t *x, char *rnfr, char *rnto)
{
	int	rc;
	char	*p, line[200];

	if ((rc = sfputc(x, "RNFR", rnfr, line, sizeof(line), &p)) != 350) {
		fprintf (stderr, "%s: can't rename from: %s, filename= %s\n", program, line, rnfr);
		return (rc);
		}
	else if ((rc = sfputc(x, "RNTO", rnto, line, sizeof(line), &p)) != 250) {
		fprintf (stderr, "%s: can't rename to: %s, filename= %s\n", program, line, rnto);
		return (rc);
		}

	return (0);
}

int dosize(ftp_t *x, char *filename, char *buffer, int size)
{
	int	rc;
	char	*p, line[300];

	if ((rc = sfputc(x, "SIZE", filename, line, sizeof(line), &p)) != 213)
		return (rc);

	copy_string(buffer, p, size);
	return (0);
}

int domdtm(ftp_t *x, char *filename, char *buffer, int size)
{
	int	rc;
	char	*p, line[300];

	if ((rc = sfputc(x, "MDTM", filename, line, sizeof(line), &p)) != 213)
		return (rc);

	copy_string(buffer, p, size);
	return (0);
}

int dopwd(ftp_t *x, char *buffer, int size)
{
	int	c, rc;
	char	*p, line[300];

	if ((rc = sfputc(x, "PWD", "", line, sizeof(line), &p)) != 257) {
		fprintf (stderr, "%s: protocol error, cmd= PWD: %s\n", program, line);
		return (rc);
		}

	if ((c = *p) != '"'  &&  c != '\'')
		copy_string(buffer, p, size);
	else {
		p++;
		get_quoted(&p, c, buffer, size);
		}

	return (0);
}

int docwd(ftp_t *x, char *dir, int create, int pe)
{
	int	rc;
	char	*p, line[300];

	if ((rc = sfputc(x, "CWD", dir, line, sizeof(line), &p)) != 250) {
		if (create != 0) {
			char	*r, subdir[80];

			if (*(r = dir) == '/') {
				if ((rc = sfputc(x, "CWD", "/", line, sizeof(line), &p)) != 250)
					goto end;

				r++;
				}

			while (*get_quoted(&r, '/', subdir, sizeof(subdir)) != 0) {
				if ((rc = sfputc(x, "CWD", subdir, line, sizeof(line), &p)) != 250) {
					if ((rc = sfputc(x, "MKD", subdir, line, sizeof(line), &p)) != 257)
						goto end;
					else if ((rc = sfputc(x, "CWD", subdir, line, sizeof(line), &p)) != 250)
						goto end;
					}
				}
			}
		}

end:
	if (rc != 0  &&  rc != 250) {
		if (pe != 0)
			fprintf (stderr, "%s: can't change dir, dir= %s: %s\n", program, dir, line);

		return (rc);
		}

	return (0);
}

int docheckdir(ftp_t *x, char *dir)
{
	int	rc;
	char	curdir[200];

	if ((rc = dopwd(x, curdir, sizeof(curdir))) != 0)
		return (rc);

	if (docwd(x, dir, 0, 0) != 0)
		return (1);

	if ((rc = docwd(x, curdir, 0, 0)) != 0)
		return (-rc);

	return (0);
}

int dodele(ftp_t *x, char *filename, int pe)
{
	int	rc;
	char	*p, line[300];

	if ((rc = sfputc(x, "DELE", filename, line, sizeof(line), &p)) != 250) {
		if (pe != 0)
			fprintf (stderr, "%s: can't delete file: %s error= %s\n", program, filename, line);

		return (rc);
		}

	return (0);
}

int doquit(ftp_t *x)
{
	int	rc;
	char	*p, line[200];

	if ((rc = sfputc(x, "QUIT", "", line, sizeof(line), &p)) != 221) {
		fprintf (stderr, "%s: quit error: %s\n", program, line);
		return (rc);
		}

	close (x->control);
	return (0);
}


static int addline(char **buffer, int *size, int here, char *line)
{
	int	len;

	if (*buffer == NULL  ||  *size == 0)
		*buffer = allocate(*size = 1024);

	len = strlen(line);
	if (here + len + 10 >= *size)
		*buffer = realloc(*buffer, *size = *size + len + 1024);

	strcpy(&(*buffer)[here], line);
	here += len;
	(*buffer)[here++] = '\n';
	(*buffer)[here]   = 0;

	return (here);
}

int dolist(ftp_t *x, int nlist, char *par, char **buffer, int *size)
{
	int	k, rc;
	char	*p, line[200];

	if (*buffer == NULL  ||  *size == 0)
		*buffer = allocate(*size = 1024);

	if ((rc = doport(x)) != 0)
		return (rc);

	if ((rc = sfputc(x, (nlist != 0)? "NLST": "LIST", par, line, sizeof(line), &p)) != 150) {
		fprintf (stderr, "%s: can't retrieve data: %s\n", program, line);
		return (rc);
		}

	if ((rc = connect_server(x)) != 0)
		return (rc);

	k = 0;
	while (readline_fd(x, x->data.fd, line, sizeof(line)) != NULL) {
		p = skip_ws(noctrl(line));
		k = addline(buffer, size, k, line);
		}

	close(x->data.fd);
	sfputc(x, NULL, NULL, line, sizeof(line), NULL);
	if ((rc = atoi(line)) != 226) {
		fprintf (stderr, "%s: data transfer error: %s\n", program, line);
		return (rc);
		}

	return (0);
}

int dologin(ftp_t *x)
{
	int	rc;
	char	*p, line[300];

	if (x->port == 0)
		x->port = get_port(x->server, 21);

	if (x->timeout == 0)
		x->timeout = 60;

	if (x->bsize <= 0)
		x->bsize = 2048;
	else if (x->bsize >= MAXBSIZE)
		x->bsize = MAXBSIZE;

	if ((x->control = openip(x->server, x->port)) < 0) {
		fprintf (stderr, "%s: can't connect to server: %s:%u: %s\n", program, x->server, x->port, strerror(errno));
		return (-1);
		}

	while (sfgets(x, line, sizeof(line)) != 0) {
		if (line[3] == ' ')
			break;
		}

	rc = strtoul(line, &p, 10);
	if (rc != 220) {
		fprintf (stderr, "%s: can't login: %s\n", program, line);
		exit (rc);
		}

	rc = sfputc(x, "USER", x->username, line, sizeof(line), &p);
	if (rc == 230)
		;	/* login without password */
	else if ((rc = sfputc(x, "PASS", x->password, line, sizeof(line), &p) != 230)) {
		fprintf (stderr, "%s: can't login: %s\n", program, line);
		return (rc);
		}

	return (0);
}

