/*
 *
 * telprox -- a simple telnet proxy daemon
 *
 * This software is free, go ahead and use it! It is
 * distributed under the GNU General Public License
 *
 * Original Author: faramir@drunken.scots.net  1997
 *
 * please see the files 'README.1st' and 'README'
 *
 *   5 Jan 1998 -- Modified to work with multihomed servers,
 *                 with possibly different listen and connection
 *                 IPs      EFC
 *   7 Jan 1998 -- Added IP authentication code     EFC
 * 
 */

/* telprox.c ---- the main body of the program */

#include "telprox.h"


int main( int argc, char * argv[] )
{

     char buf[80], *ptr;

    allowed = NULL;
    denied  = NULL;

    /* read the config file if there is one */
    get_parameters(configname, Paramtab, TABSIZE);

    prune_allowed();

     if( argc != 2) {
	  printf( "using default port, %d\n\r", default_port );
	  port = default_port;
     }
     else {
	  strcpy( buf, argv[1] );
	  ptr = &buf[0];
	  if( isdigit( *ptr )) {
	       port = atoi( argv[1] );
               printf("-------------------------------\n");
	       printf( "%s: using port %d\n\r", argv[0], port );
	  }
	  else {
	       if( *ptr != '-' ) {
		    printf( "%s: specify options with '-'\n\r", argv[0] );
		    exit( 0 );
	       }
	       ptr++;
	       switch ( *ptr ) {
	       case '\0':
	       case 'h':
	       case 'H':
		    printf( "%s %s\n\r",GREET_MSG, VERS_MSG );
		    printf( "%s: usage: telprox {local port}  (default = %d)\n\r", argv[0], default_port );
		    printf( "%s:        telprox -h  (this help)\n\r", argv[0] );
		    printf( "%s:        telprox -v  (version)\n\r", argv[0] );
		    break;
	       case 'v':
	       case 'V':
		    printf( "%s %s\n\r", GREET_MSG, VERS_MSG );
		    printf( "Feel free to use this software under the auspices of the GNU General Public\n\rLicense. You can find this at:\thttp://www.gnu.org\n\r" );
		    break;
	       default:
		    printf( "%s: unrecognised option -%c\n\r", argv[0],  *ptr );
		    printf( "%s: try 'telprox -h'\n\r", argv[0] );
	       }
	       return 0;
	  }
     }

     proxy_up = TRUE;
     printf( "%s: pid() == %d\n\r", argv[0], getpid() );
     if( proxy_init()) {
	  fprintf(stderr, "proxy_init() returned error\n\r" );
	  return 1;
     }
     net_main();
     exit( 0 );
}




/*
 open a socket to the specified port and listen on it.
   initialise the daemon 
   */
  
int proxy_init()
{
     pid_t pid;
     char buf[80];
     int len = 0;

     
     if(( socket_in = socket( AF_INET, SOCK_STREAM, 0 )) == -1 ) {
	  fprintf(stderr, "socket creation failed\n\r" );
	  return 1; 
     }
     if( setsockopt( socket_in, SOL_SOCKET, SO_REUSEADDR, 
		     &sockoptions, sizeof( sockoptions ))) {
	  fprintf(stderr, "setsockopt(): can't set reuse\n\r" );
	  return 1;
     }
     if( setsockopt( socket_in, SOL_SOCKET, SO_LINGER,
		     &hang_around, sizeof( hang_around ))) {
	  fprintf(stderr, "cannot set linger on socket\n\r" );
	  return 1;
     }
     memset( (void *) &s, 0, sizeof( s )); /* blankety blank */
     s.sin_family = AF_INET;
     s.sin_port = htons( port );
     /* set my listen address to the one specified */ 
     if( inet_aton( listen_host, (struct in_addr *) &s.sin_addr ) == 0 ) {
	  fprintf( stderr, "inet_aton() failed: %s\n", listen_host );
	  return 1;
     }
     
     if( bind( socket_in, (struct sockaddr *) &s, sizeof( s ))) {
	  fprintf(stderr, "cannot bind port %d\n\r", port );
	  return 1;
     }
     if( listen( socket_in, 5 )) {
	  fprintf(stderr, "cannot listen()\n\r" );
	  return 1;
     }
     if(( pid = fork()) < 0 )
	  return 1;
     else if( pid != 0 )
	  exit( 0 ); /* adios parent */
     setsid();	  
     return 0;
}

void printaddr(char* sb)
{
    fprintf(stderr,"\t%d.%d.%d.%d\n", sb[0] & 0xff, sb[1]& 0xff,
                        sb[2]&0xff, sb[3]& 0xff);
}

int notify_failed()
{
     if( send( socket_plr, (char *) NOTAUTH_MSG, sizeof(NOTAUTH_MSG), 
	       0 ) == -1 )
     {
	  fprintf( stderr, "notauthorized message failure\n\r" );
     }

     fprintf(stderr,"client host not authorized\n\r");

     return 1;
}

/* 
 * the main net handling routine
 */

int net_main()
{
     
     int numfds, socket_size;

     signal( SIGCHLD, SIG_IGN );

     do {
	  FD_ZERO( &fd_start );
	  FD_SET( socket_in, &fd_start );
	  if( select( FD_SETSIZE, &fd_start, NULL, NULL, NULL ) < 0 ) {
	       fprintf( stderr, "net_main(): select() error\n\r" );
	       return 1;
	  }
	  if( FD_ISSET( socket_in, &fd_start )) {
	       fprintf( stderr, "net_main(): connection\n\r" );
               socket_size = sizeof( s2 );
	       if(( socket_in2 = accept( socket_in, (struct sockaddr *) &s2, 
					 &socket_size )) == -1 ) {
		    fprintf( stderr, "net_main(): accept error\n\r" );
		    return 1;
	       }
	       if(( child_pid = fork()) < 0 ) {
		    fprintf( stderr, "net_main(): fork error\n\r" );
		    return 1;
	       }
	       if( !child_pid ) { /* child */
		    if(( socket_plr = dup( socket_in2 )) == -1 ) {
			 fprintf( stderr, "net_main(): dup error\n\r" );
			 return 1;
		    } 
		    if(( socket_mud = socket( AF_INET, 
					      SOCK_STREAM, 0 )) < 0 ) {
			 fprintf( stderr, "net_main(): mud socket error\n\r" );
			 return 1;
		    }

                    /* set port for binding */
                    s.sin_port = s2.sin_port;
                    /* set my connect address to the one specified */ 
                    if( inet_aton( connect_host,
				  (struct in_addr *) &s.sin_addr ) == 0 ) {
	                           fprintf( stderr,
					    "inet_aton() failed: %s\n",
					   connect_host );
	            return 1;
                    }

                    fprintf(stderr,"got contact from host: ");
                    printaddr( (char *)&s2.sin_addr.s_addr );

                    /* verify that this connection is allowed */
                    if ( allow( s2.sin_addr.s_addr ) )
		          return manage_connections();
                    else
                       return notify_failed();

	       }
	       else {  /* parent */
		    sleep( 5  );    /* hack, but should be enough */
		    close( socket_in2 ); 
	       }
	  }
	  
     } while( proxy_up );
     
}


/*
 * control the connections
 */

int manage_connections()
{
     
     char buf[256];

     fprintf( stderr, "socket_mud = %d, socket_plr = %d\n", socket_mud, socket_plr );

     memset( (void * ) &out_s, 0, sizeof( out_s));
     out_s.sin_family = AF_INET;
     out_s.sin_port = htons( proxy_port );
     if( inet_aton( proxy_host, (struct in_addr *) &out_s.sin_addr ) == 0 ) {
	  fprintf( stderr, "inet_aton() failed: %s\n", proxy_host );
	  return 1;
     }

     /* bind the remote connection to use the specified local addr */
     if( bind( socket_mud, (struct sockaddr *) &s, sizeof( s ))) {
	  fprintf(stderr, "cannot bind to mud socket (%d)\n", errno);
	  return 1;
     }

     fprintf( stderr, "connecting to %s\n", proxy_host );
     if(( connect( socket_mud, (struct sockaddr *) &out_s,
		   sizeof( out_s ))) < 0 ) {
	  fprintf( stderr, "connection failed: %s\n", sys_errlist[errno] );
	  return 1;
     }
     if( send( socket_plr, (char *) GREET_MSG, sizeof( GREET_MSG ), 
	       0 ) == -1 ){
	  fprintf( stderr, "greet failure\n\r" );
	  return 1;
     }
     send( socket_plr, (char *) VERS_MSG, sizeof( VERS_MSG ), 0 );
     sprintf( buf, "\n\rconnecting to: %s:%d\n\r", proxy_host, proxy_port );
     send( socket_plr, (char *) buf, strlen( buf ), 0 );
     do {
	  FD_ZERO( &fd_read );
	  FD_SET( socket_plr, &fd_read );
	  FD_SET( socket_mud, &fd_read );

	  if( select( FD_SETSIZE, &fd_read, NULL, NULL, NULL ) < 0 ) {
	       fprintf( stderr, "net_main(): select() failed\n" );
	       return 1;
	  }
	  if( FD_ISSET( socket_mud, &fd_read )) {    /* read from the mud */

	       if( !relay_buffer( socket_mud, socket_plr )) {   
		    /* i.e. if read failed, mud sock must have died */
		    /* so kill the player socket off... */
		    
		    clear_fds();             
	       }
	  }
	  
	  if( FD_ISSET( socket_plr, &fd_read )) {  /* read from the player */

	       if( !relay_buffer( socket_plr, socket_mud )) {    
		    /* i.e. if read failed, plr sock must have died */
		    /* player socket must have died so kill mud sock */
		    
		    clear_fds();
	       }
	  }

     } while( proxy_up );
     fprintf( stderr, "manage_connections(): finished\n\r" );
     return 0;
}


/*
 * transfer data back and forth between the two parties 
 */

bool relay_buffer( int read_d, int write_d )
{
     
     char buffer[BUFSIZE];
     int received = 0, s = 0;
     int size =  0;
     do {
	  if(( received = recv( read_d, (char *)buffer, BUFSIZE, 0 )) == -1 ) {
	       perror( "relay_buffer (recv): " );
	       return FALSE;
	  }
	  if( received == 0 ) 
	       return FALSE;
	  
	  if(( s = send( write_d, (char *)buffer, received, 0 )) == -1 ) {
	       perror( "relay_buffer (send): " );
	       return FALSE;
	  }
	  if( s == 0 ) 
	       return FALSE;
	  if( ioctl( read_d, FIONREAD, &size, sizeof( int ) )  == -1 ) 
	       perror( "relay_buffer(): ioctl():" );
/*	  fprintf( stderr, "ioctl(): %d\n\r", size ); */
     } while( size  > 0 );
     return TRUE;
}


/* close things down */

void clear_fds()
{
     
     FD_CLR( socket_plr, &fd_read );
     FD_CLR( socket_mud, &fd_read );
     close( socket_mud );
     close( socket_plr );
     fprintf( stderr, "clear_fds()\n\r" );
     proxy_up = FALSE;
}

/* add to the allowed list */
void add_allow(char* str )
{
     long int ip;

     if ( (ip = inet_addr(str)) != -1L )
           add( &allowed, ip );
}

/* add to the denied list */
void add_deny(char* str )
{
     long int ip;

     if ( (ip = inet_addr(str)) != -1L )
           add( &denied, ip );
}

/* add an element to the IP address lists */
int add(list** l, long int val)
{
    list* new_element;

    if ( (new_element = (list *)malloc( sizeof( list ))) == NULL )
                        return 1;

    new_element->addr = val;
    new_element->next = *l;

    *l = new_element;

    return 0;

}

/*
   if an entry is in both the allowed and denied list,
   take it out of the allowed
*/

void prune_allowed()
{
      list* den;
      list* al;
      list* prev = NULL;

      for (al = allowed; al; )
      {
           for (den = denied; den; den = den->next )
	   {
               if ( den->addr == al->addr )
	       {
                    if ( al == allowed )
                       allowed = al->next;
                    else
                       prev->next = al->next;

                    free( al );
                    al = prev;
                    break;
	       }

	   }

           prev = al;

           if ( al == NULL )
                al = allowed;
	   else
	        al = al->next;
      }

}

/* scan the IP address list
   return:
       0   no match
       1   masked match (same subnet)
       2   exact match
*/

int found(list* l, long int val)
{
    list* ptr;

    for (ptr = l; ptr; ptr = ptr->next)
    {
        if ( ptr->addr == val )
                 return 2;
        if ( (ptr->addr & val) == val )
                 return 1;
    }


    return 0;

}

/* scan both the allowed and denied lists to see if its okay */
int allow(long int addr)
{
    int den, al;

    den = found( denied, addr);
    al =  found( allowed, addr);

    if ( den > al )       /* denied has greater authority */
           return 0;

    if ( al > den )       /* allowed has greater authority */
           return 1;

    /* if both allowed and denied and not implict allow then deny */
    if ( al == den && den != 0 && al != 0 )
           return 0;


    /* at this point it was not found in either list */

    if ( allowed )      /* an allowed list exists, so deny */
            return 0;


    /* no allowed list and not in the denied list */

    return 1;      /* implicit allow (not in either list) */
}




