
/*
 *
 * telprox -- a simple telnet proxy daemon
 *
 * This software is free, go ahead and use it! It is
 * distributed under the GNU General Public License
 *
 * Author: faramir@drunken.scots.net  1997
 *
 * please see the file 'README'
 */

/* telprox.c ---- the main body of the program */

#include "telprox.h"


int main( int argv, char * argc[] ) {

     char buf[80], *ptr;
     if( argv != 2) {
	  printf( "using default port, %d\n\r", default_port );
	  port = default_port;
     }
     else {
	  strcpy( buf, argc[1] );
	  ptr = &buf[0];
	  if( isdigit( *ptr )) {
	       port = atoi( argc[1] );
	       printf( "%s: using port %d\n\r", argc[0], port );
	  }
	  else {
	       if( *ptr != '-' ) {
		    printf( "%s: specify options with '-'\n\r", argc[0] );
		    exit( 0 );
	       }
	       ptr++;
	       switch ( *ptr ) {
	       case '\0':
	       case 'h':
	       case 'H':
		    printf( GREET_MSG );
		    printf( "%s: usage: telprox {local port}  (default = %d)\n\r", argc[0], default_port );
		    printf( "%s:        telprox -h  (this help)\n\r", argc[0] );
		    printf( "%s:        telprox -v  (version)\n\r", argc[0] );
		    break;
	       case 'v':
	       case 'V':
		    printf( GREET_MSG );
		    printf( VERS_MSG );
		    printf( "Feel free to use this software under the auspices of the GNU General Public\n\rLicense. You can find this at:\n\rhttp://www.gnu.org or mail me at faramir@drunken.scots.net\n\r" );
		    break;
	       default:
		    printf( "%s: unrecognised option -%c\n\r", argc[0],  *ptr );
		    printf( "%s: try 'telprox -h'\n\r", argc[0] );
	       }
	       return 0;
	  }
     }
     proxy_up = TRUE;
     printf( "%s: pid() == %d\n\r", argc[0], getpid() );
     if( proxy_init()) {
	  printf( "proxy_init() returned error\n\r" );
	  return 1;
     }
     net_main();
     exit( 0 );
}



/* read the config file if there is one
   open a socket to the specified port and listen on it.
   initialise the daemon 
   */
  
int proxy_init() {
     
     pid_t pid;
     char buf[80], *ptr, *ptr2;
     int len = 0;

     if(( configfile = fopen( configname, "r" )) == NULL ) 
	  printf( "couldn't open config file 'telprox.conf', using defaults\n\r" );
     else {
	  if( fgets( buf, 80, configfile ) != NULL ) {
	       ptr = &buf[0];
	       while( isspace( *ptr ))
		    ptr++;
	       if(( ptr2 = strtok( ptr, " " )) != NULL ) {
		    strcpy( proxy_host, ptr2 );
		    if(( ptr2 = strtok( NULL, " " )) != NULL ) 
			 proxy_port = atoi( ptr2 );
		    else 
			 printf( "looks like the port number in your config file is broken\n\r" );
	       }
	       else
		    printf( "looks like the address in your config file is broken\n\r" );
	  }
	  else 
	       printf( "couldn't read from config file, hmmm\n\r" );
	  fclose( configfile );
     }
     
     if(( socket_in = socket( AF_INET, SOCK_STREAM, 0 )) == -1 ) {
	  printf( "socket creation failed\n\r" );
	  return 1; 
     }
     if( setsockopt( socket_in, SOL_SOCKET, SO_REUSEADDR, 
		     &sockoptions, sizeof( sockoptions ))) {
	  printf( "setsockopt(): can't set reuse\n\r" );
	  return 1;
     }
     if( setsockopt( socket_in, SOL_SOCKET, SO_LINGER,
		     &hang_around, sizeof( hang_around ))) {
	  printf( "cannot set linger on socket\n\r" );
	  return 1;
     }
     memset( (void *) &s, 0, sizeof( s )); /* blankety blank */
     s.sin_family = AF_INET;
     s.sin_port = htons( port );
     
     if( bind( socket_in, (struct sockaddr *) &s, sizeof( s ))) {
	  printf( "cannot bind port %d\n\r", port );
	  return 1;
     }
     if( listen( socket_in, 5 )) {
	  printf( "cannot listen()\n\r" );
	  return 1;
     }
     if(( pid = fork()) < 0 )
	  return 1;
     else if( pid != 0 )
	  exit( 0 ); /* adios parent */
     setsid();	  
     return 0;
}


/* 
 * the main net handling routine
 */

int net_main() {
     
     int numfds;

     signal( SIGCHLD, SIG_IGN );

     do {
	  FD_ZERO( &fd_start );
	  FD_SET( socket_in, &fd_start );
	  if( select( FD_SETSIZE, &fd_start, NULL, NULL, NULL ) < 0 ) {
	       fprintf( stderr, "net_main(): select() error\n\r" );
	       return 1;
	  }
	  if( FD_ISSET( socket_in, &fd_start )) {
	       fprintf( stderr, "net_main(): connection\n\r" );
	       if(( socket_in2 = accept( socket_in, (struct sockaddr *) &s2, 
					 &socket_size )) == -1 ) {
		    fprintf( stderr, "net_main(): accept error\n\r" );
		    return 1;
	       }
	       if(( child_pid = fork()) < 0 ) {
		    fprintf( stderr, "net_main(): fork error\n\r" );
		    return 1;
	       }
	       if( !child_pid ) { /* child */
		    if(( socket_plr = dup( socket_in2 )) == -1 ) {
			 fprintf( stderr, "net_main(): dup error\n\r" );
			 return 1;
		    } 
		    if(( socket_mud = socket( AF_INET, 
					      SOCK_STREAM, 0 )) < 0 ) {
			 fprintf( stderr, "net_main(): mud socket error\n\r" );
			 return 1;
		    }
		    manage_connections();
		    return 0;
	       }
	       else {  /* parent */
		    sleep( 5  );    /* hack, but should be enough */
		    close( socket_in2 ); 
	       }
	  }
	  
     } while( proxy_up );
     
}


/*
 * control the connections
 */

void manage_connections() {
     
     char buf[256];
     fprintf( stderr, "socket_mud = %d, socket_plr = %d\n", socket_mud, socket_plr );
     memset( (void * ) &out_s, 0, sizeof( out_s));
     out_s.sin_family = AF_INET;
     out_s.sin_port = htons( proxy_port );
     if( inet_aton( proxy_host, (struct in_addr *) &out_s.sin_addr ) == 0 ) {
	  fprintf( stderr, "inet_aton() failed: %s\n", proxy_host );
	  return;
     }
     fprintf( stderr, "connecting to %s\n", proxy_host );
     if(( connect( socket_mud, (struct sockaddr *) &out_s,
		   sizeof( out_s ))) < 0 ) {
	  fprintf( stderr, "connection failed: %s\n", sys_errlist[errno] );
	  return;
     }
     if( send( socket_plr, (char *) GREET_MSG, sizeof( GREET_MSG ), 
	       0 ) == -1 ){
	  fprintf( stderr, "greet failure\n\r" );
	  return;
     }
     send( socket_plr, (char *) VERS_MSG, sizeof( VERS_MSG ), 0 );
     sprintf( buf, "connecting to: %s:%d\n\r", proxy_host, proxy_port );
     send( socket_plr, (char *) buf, strlen( buf ), 0 );
     do {
	  FD_ZERO( &fd_read );
	  FD_SET( socket_plr, &fd_read );
	  FD_SET( socket_mud, &fd_read );

	  if( select( FD_SETSIZE, &fd_read, NULL, NULL, NULL ) < 0 ) {
	       fprintf( stderr, "net_main(): select() failed\n" );
	       return;
	  }
	  if( FD_ISSET( socket_mud, &fd_read )) {    /* read from the mud */
	       fprintf( stderr, "FD_ISSET( socket_mud )\n" );
	       if( !relay_buffer( socket_mud, socket_plr )) {   
		    /* i.e. if read failed, mud sock must have died */
		    /* so kill the player socket off... */
		    
		    clear_fds();             
	       }
	  }
	  
	  if( FD_ISSET( socket_plr, &fd_read )) {  /* read from the player */
	       fprintf( stderr, "FD_ISSET( socket_plr )\n" );
	       if( !relay_buffer( socket_plr, socket_mud )) {    
		    /* i.e. if read failed, plr sock must have died */
		    /* player socket must have died so kill mud sock */
		    
		    clear_fds();
	       }
	  }

     } while( proxy_up );
     fprintf( stderr, "manage_connections(): finished\n\r" );
     return;
}


/*
 * transfer data back and forth between the two parties 
 */

bool relay_buffer( int read_d, int write_d ) {
     
     char buffer[BUFSIZE];
     int received = 0, s = 0;
     int size =  0;
     do {
	  if(( received = recv( read_d, (char *)buffer, BUFSIZE, 0 )) == -1 ) {
	       perror( "relay_buffer (recv): " );
	       return FALSE;
	  }
	  if( received == 0 ) 
	       return FALSE;
	  
	  if(( s = send( write_d, (char *)buffer, received, 0 )) == -1 ) {
	       perror( "relay_buffer (send): " );
	       return FALSE;
	  }
	  if( s == 0 ) 
	       return FALSE;
	  if( ioctl( read_d, FIONREAD, &size, sizeof( int ) )  == -1 ) 
	       perror( "relay_buffer(): ioctl():" );
/*	  fprintf( stderr, "ioctl(): %d\n\r", size ); */
     } while( size  > 0 );
     return TRUE;
}


/* close things down */

void clear_fds() {
     
     FD_CLR( socket_plr, &fd_read );
     FD_CLR( socket_mud, &fd_read );
     close( socket_mud );
     close( socket_plr );
     fprintf( stderr, "clear_fds()\n\r" );
     proxy_up = FALSE;
}

