/* Ringconnect 2.1 (c) 1997 Karl Stenerud (stenerud@lightspeed.bc.ca)
 * 02-Aug-97
 *
 * See readme file and manpage for details.
 *
 * Supported platforms:
 *    Linux.  Anything else is untested.
 *
 * Copying / Liscence / Warranty:
 *    This software is released as freeware under the GNU Public Liscence.
 *    If you do any significant improvements on it, mail me a copy!
 *    
 *    As for warranty, there is none.  Use at your own risk!
 * 
 */

#include <stdio.h>
#include <fcntl.h>
#include <sys/time.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <sys/ioctl.h>
#include <linux/if.h>
#include <linux/if_ppp.h>
#include <netdb.h>
#include <signal.h>
#include <linux/serial.h>
#include <stdlib.h>
#include "opt.h"
#include "log.h"

void  wait_for_ring   (int num_rings);
int   count_rings     ();
int   is_ppp_up       ();
int   start_ppp       (char* prog, int max_tries);
int   is_prg_running  (char* proc_cmdline);
int   get_pid         (char* proc_cmdline);
int   kill_proc       (char* proc_cmdline);
void  call_prg        (char* cmdline);
void  mail_ip         (char* server, char* recipient, char* my_ip, int timeout);
int   connect_socket  (char* address, short port);
int   dialog          (int fd, char* message, char* expected, char* err_msg, int timeout, int do_or_die);
int   wait_for_input  (int fd, int timeout);
int   contains        (char* buff, char* str);
char* get_ppp_addr    ();
void  sleep_until_ring();
void  error_callback  (char* msg, int is_fatal);


int g_sockfd;				// used for ioctl functions for ppp
int g_modemfd;

opt_struct g_opts[] =			// options
{
//option        flag type      default          description
{"device",      'd', a_efile,  "/dev/modem",    "Modem device to attach to"},
{"rings",       'R', a_int,    "0",             "Wait this many rings to connect. (0=immediate)"},
{"ring_time",   'T', a_int,    "7",             "Time between rings"},
{"connect_prg", 'c', a_cmd,    "/usr/sbin/pppd","Program to connect online (full path!)"},
{"redial",      'r', a_int,    "10",            "Times to redial if line is busy"},
{"smtp_server", 's', a_str,    "",              "Mail server"},
{"mailto",      'm', a_str,    "",              "User to mail IP address to"},
{"timeout",     't', a_int,    "30",            "Timeout when waiting for mail server's response"},
{"run_prg",     'p', a_dcmd,   "",              "Run this program upon connection"},
{"onceonly",    'o', a_bool,   "0",             "Connect once, then quit"},
{"kill_after",  'k', a_int,    "0",             "Kill Connection after num seconds (0=disabled)"},
{"kill_prg",    'K', a_dcmd,   "",              "Use this prog to end connection (full path!)"},
{"nodetach",    'n', a_bool,   "0",             "Run in foreground instead of detaching"},
{"config_file", 'f', a_defile, "/etc/ringconnect.conf","Config file to read from"},
{NULL}
};


int main(int argc, char** argv)
{
   char* connect_prg;
   char* device;
   char* smtp_server;
   char* mailto;
   int redial;
   int onceonly;
   int rings;
   int timeout;
   int kill_after;
   char* kill_prg;
   char* run_prg;
   int nodetach;

   signal(SIGCHLD, SIG_IGN);		// Kills zombies.  Dead.

   opt_init(g_opts, "ringconnectd [options]", NULL, error_callback);
   parse_args(argc, argv);
   opt_load(opt_get("config_file"));

   // get our options
   connect_prg = opt_get("connect_prg");
   kill_prg = opt_get("kill_prg");
   device = opt_get("device");
   smtp_server = opt_get("smtp_server");
   mailto = opt_get("mailto");
   redial = *((int*) opt_get("redial"));
   rings = *((int*) opt_get("rings"));
   timeout = *((int*) opt_get("timeout"));
   onceonly = *((int*) opt_get("onceonly"));
   kill_after = *((int*) opt_get("kill_after"));
   run_prg = opt_get("run_prg");
   nodetach = *((int*) opt_get("nodetach"));

   if(!nodetach)
   {
      if(fork())			// go into daemon mode
         exit(0);
      if(fork())
         exit(0);

      mode_syslog(argv);		// go to syslog mode
   }

   // socket for ioctl function calls in get_ppp_addr() and is_ppp_up()
   if( (g_sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0 ) fatal("main: socket");
   // modem device for monitoring
   if( (g_modemfd = open(opt_get("device"), O_RDONLY | O_NONBLOCK | O_NOCTTY)) < 0 )
      fatal("main: open");

   for(;;)
   {
      log("Waiting for ring on %s\n", device);
      wait_for_ring(rings);
      log("RING! Running %s\n", connect_prg);
      if(start_ppp(connect_prg, redial))
      {
         log("Connected!\n");
         mail_ip(smtp_server, mailto, get_ppp_addr(), timeout);
         if(*run_prg != '\0')
         {
            log("Executing %s\n", run_prg);
            call_prg(run_prg);
         }
         if(kill_after > 0)
         {
            sleep(kill_after);
            log("%d seconds, killing connection\n", kill_after);
            if(*kill_prg != '\0')
            {
               log("Calling %s to kill connection\n", kill_prg);
               call_prg(kill_prg);
            }
            else
            {
               log("Killing %s\n", connect_prg);
               kill_proc(connect_prg);
            }
         }
         if(onceonly)
            break;
      }
      else
         log("Failed to connect.\n");
   }
   log("Exiting.\n");
   exit(0);
}

void wait_for_ring(int num_rings)
{
   int base_rings = count_rings();	// get current ring count
   int last_time;			// timer stuff
   int this_time = time(NULL) - 1000;
   int ring_time = *((int*)opt_get("ring_time"));

   if(num_rings <= 0)			// answer on first ring
   {
      sleep_until_ring();
      sleep(ring_time);
      return;
   }

   for(;;)				// answer on num_rings
   {
      while(count_rings() < base_rings + num_rings)
      {
         last_time = this_time;
         sleep_until_ring();
         if((this_time=time(NULL)) - last_time > ring_time)
            base_rings = count_rings() - 1;
      }
      // got initial pattern.  Now make sure no more rings come
      sleep(ring_time);
      if(count_rings() == base_rings + num_rings)
         return;

      // there were too many rings.  Flush out all subsequent rings
      // and start over

      last_time = this_time = time(NULL);

      while((this_time=time(NULL)) - last_time <= ring_time)
      {
         last_time = this_time;
         sleep_until_ring();
      }
      base_rings = count_rings() - 1;
   }
}

void sleep_until_ring()
{
   int num_rings = count_rings();

   // wait for 1 ring
   while(count_rings() == num_rings)
      ioctl(g_modemfd, TIOCMIWAIT, TIOCM_RNG);
}

int count_rings()
{
   struct serial_icounter_struct c;
   int tries = 0;

   // query serial device
   while(ioctl(g_modemfd, TIOCGICOUNT, &c) < 0)
   {
      if(errno != EIO)
         fatal("ioctl");
      if(tries++ >= 5)		// try 5 times, then abort
         write_error(1, "Problem with serial port.\n");
      close(g_modemfd);
      if( (g_modemfd = open(opt_get("device"), O_RDONLY | O_NONBLOCK | O_NOCTTY)) < 0 )
         write_perror("count_rings: open");
   }
   return c.rng;
}

int is_ppp_up()
{
   struct ifreq ifr;

   // get ppp0 flags from the kernel
   sprintf(ifr.ifr_name, "ppp0");
   if(ioctl(g_sockfd, SIOCGIFFLAGS, &ifr) < 0)
   {
      if(errno == ENODEV)
         return 0;
      fatal("is_ppp_up: ioctl(SIOCGIFFLAGS)");
   }
   return (ifr.ifr_flags & IFF_UP);
}

int start_ppp(char* prog, int max_tries)
{
   int tries = 0;

   while(!is_ppp_up())			// keep trying
   {
      if(!is_prg_running(prog))
      {
         call_prg(prog);		// call the program
         if(max_tries >= 0 && ++tries > max_tries)
            return 0;
      }
      sleep(1);
   }
   return 1;
}

int kill_proc (char* proc_cmdline)
{
   int pid = get_pid(proc_cmdline);

   if(pid == -1)
   {
      write_error(0, "Could not kill %s: no such process\n", proc_cmdline);
      return 0;
   }   
   if(kill(pid, SIGKILL) < 0)
   {
      write_perror("Could not kill %s", proc_cmdline);
      return 0;
   }
   return 1;
}

int is_prg_running(char* proc_cmdline)
{
   return get_pid(proc_cmdline) != -1;
}

int get_pid(char* proc_cmdline)
{
   DIR* dir;
   struct dirent* dent;
   char buff[1000];
   int fd;
   int len;

   // open the /proc dir and look for pids
   if( (dir=opendir("/proc")) == NULL) fatal("find_pid: opendir");

   while( (dent = readdir(dir)) != NULL)
   {
      if(is_integer(dent->d_name))
      {
         // for each entry in /proc, try to use it as a directory and read
         // the file "cmdline" file from within it.
         sprintf(buff, "/proc/%s/cmdline", dent->d_name);
         if( (fd=open(buff, O_RDONLY)) >0 )
         {
            if( (len=read(fd, buff, 1000)) < 0 ) fatal("find_pid: read");
            close(fd);
            buff[len] = 0;
            // if it matches what we're looking for, return true
            if(strcmp(buff, proc_cmdline) == 0)
            {
               closedir(dir);
               return atoi(dent->d_name);
            }
         }
      }
   }
   closedir(dir);
   return -1;
}

void call_prg(char* cmdline)
{
   char argv[25][100];
   char* argvp[25];
   char* ptr = cmdline;
   int i;

   if(cmdline == NULL)
      return;

   for(i=0;i<25;i++)
   {
      argvp[i] = argv[i];
      ptr=copy_arg(argvp[i], ptr);

      if(*ptr == '\0')	// last arg
         break;
   }
   if(++i>24)		// get off last arg
      i = 24;

   argvp[i] = NULL;	// null terminate

   switch(fork())
   {
      case 0:
         if(execv(*argvp, argvp) < 0)
            perror("execv");
      case -1:
         fatal("call_prg: fork");
   }
}

void mail_ip(char* server, char* recipient, char* my_ip, int timeout)
{
   char  Rcpt[100];
   char  Mesg[100];
   char  ip[16];
   char* Helo = "HELO localhost\r\n";
   char* From = "MAIL FROM:ringconnect@localhost\r\n";
   char* Data = "DATA\r\n";
   char* Quit = "QUIT\r\n";
   int fd;

   if(server == NULL || recipient == NULL || my_ip == NULL
   || *server == '\0' || *recipient == '\0')
      return;


   strcpy(ip, my_ip);
   sprintf(Rcpt, "RCPT TO:%s\r\n", recipient);
   sprintf(Mesg, "Subject: IP is %s\r\nMy IP is %s\r\n.\r\n", ip, ip);

   log("Mailing IP %s to %s via %s\n", ip, recipient, server);

   // connect to the server
   if((fd=connect_socket(server, 25)) < 0)
      fatal("mail_ip: connect");

   // start a dialog and send the mail
   if(dialog(fd, NULL, "220", "Failed on CONNECT\n", timeout, 0))
   {
      dialog(fd, Helo, "250", NULL, timeout, 0);
      if(dialog(fd, From, "250", "Failed on MAIL\n", timeout, 0))
         if(dialog(fd, Rcpt, "250", "Failed on RCPT\n", timeout, 0))
            if(dialog(fd, Data, "354", "Failed on DATA\n", timeout, 0))
               if(dialog(fd, Mesg, "250", "Failed on EOF\n", timeout, 0))
                  dialog(fd, Quit, NULL, "", timeout, 0);
   }
   close(fd);
}


int connect_socket(char* address, short port)
{
   struct sockaddr_in saddr;
   struct hostent* host_info;		// returned by gethostbyname()
   int fd;

   memset(&saddr, 0, sizeof(saddr));
   saddr.sin_family = AF_INET;
   saddr.sin_port = htons(port);

   // try to decode dotted quad notation
   if(!inet_aton(address, &saddr.sin_addr))
   {
      // failing that, look up the name
      if((host_info = gethostbyname(address)) == NULL)
         fatal("connect_socket: gethostbyname");
      memcpy(&saddr.sin_addr, host_info->h_addr, host_info->h_length);
   }
   if((fd=socket(AF_INET, SOCK_STREAM, 0)) < 0) fatal("connect_socket: socket");
   if(connect(fd, (struct sockaddr*) &saddr, sizeof(saddr)) < 0)
      fatal("connect_socket: connect");
   return fd;
}

int dialog(int fd, char* message, char* expected, char* err_msg, int timeout, int do_or_die)
{
   char buff[1000];
   int len = 0;

   if(message != NULL)
      if(write(fd, message, strlen(message)) < 0) fatal("dialog: write");

   if(expected != NULL)
   {
      // wait for some data then compare it with what we're expecting
      if(!wait_for_input(fd, timeout))
         write_error(0, "Timed out on read\n");
      else
         if((len=read(fd, buff, 1000)) < 0) fatal("dialog: read");

      buff[len] = 0;
      if(!contains(buff, expected))
      {
         if(err_msg != NULL)
            write_error(do_or_die, err_msg);
         return 0;
      }
   }
   return 1;
}

int wait_for_input(int fd, int timeout)
{
   struct timeval  tval = {timeout, 0};
   struct timeval* wait_val = (timeout >= 0 ? &tval : NULL); // < 0 = forever
   fd_set in_set;
   fd_set out_set;
   fd_set exc_set;

   FD_ZERO(&in_set);
   FD_ZERO(&out_set);
   FD_ZERO(&exc_set);
   FD_SET(fd, &in_set);

   // use select to wait on this file descriptor
   switch(select(fd+1, &in_set, &out_set, &exc_set, wait_val))
   {
      case 0:
         return 0;			// we timed out
      case 1:
         if(!FD_ISSET(fd, &in_set))	// big problem
            write_error(1, "select returned with no read\n");
         return 1;			// everything a-ok
      default:
         fatal("wait_for_input: select"); // error
   }
   return 0;
}

int contains(char* buff, char* str)
{
   int len = strlen(str);
   char* ptr = buff;

   for(;ptr != NULL;ptr++)
      if(strncmp(buff, str, len) == 0)
         return 1;
   return 0;
}

char* get_ppp_addr()
{
   struct ifreq ifr;
   struct sockaddr_in saddr;

   // get ppp0 address from kernel
   sprintf(ifr.ifr_name, "ppp0");
   if(ioctl(g_sockfd, SIOCGIFADDR, &ifr) < 0)
   {
      if(errno == ENODEV)
         memset(&ifr, 0, sizeof(ifr));
      else
         fatal("get_ppp_addr: ioctl(SIOCGIFADDR)");
   }
   memcpy(&saddr, &(ifr.ifr_dstaddr), sizeof(struct sockaddr));
   return inet_ntoa(saddr.sin_addr);
}

void error_callback(char* msg, int is_fatal)	// error callback for opt.c
{
   write_error(is_fatal, msg);
}
