/*
** This program is in the public domain and may be used freely by anyone
** who wants to.
**
** OpenBSD support by Slawomir Piotrowski <slawek@telsatgp.com.pl>
** Now it includes masquerading support
**
** $Id: openbsd.c,v 1.8 2000/08/22 06:07:48 odin Exp $
*/

#include <config.h>

#include <oidentd.h>
#include <oidentd_util.h>
#include <oidentd_masq.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <fcntl.h>
#include <stdlib.h>
#include <kvm.h>
#include <nlist.h>
#include <limits.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <net/if.h>
#include <net/route.h>
#include <netinet/ip.h>
#include <netinet/in_pcb.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <netinet/ip_fil_compat.h>
#include <netinet/ip_fil.h>
#include <netinet/ip_proxy.h>
#include <netinet/ip_nat.h>

#ifdef MASQ_SUPPORT
extern u_long proxy;
#endif

extern u_char *charset;
extern u_int flags;

struct nlist nl[] = {
#define N_TCBTABLE 0
	{"_tcbtable"},
#define N_NATLIST 1
	{"_nat_instances"},
	{""}
};

static kvm_t *kd;

/* taken from Peter Eriksson <pen@lysator.liu.se> */
int k_open(void) {
	char errbuf[_POSIX2_LINE_MAX];

	/*
	** Open the kernel memory device
	*/

	kd = kvm_openfiles(NULL, "/dev/mem", NULL, O_RDONLY, errbuf);

	if (kd == NULL) {
		o_log(DPRI, "Error: kvm_openfiles: %s", strerror(errno));
		return (-1);
	}

	/*
	** Extract offsets to the needed variables in the kernel
	*/

	if (kvm_nlist(kd, nl) < 0) {
		o_log(DPRI, "Error: kvm_nlist: %s", strerror(errno));
		return (-1);
	}

	return (0);
}

/*
** Get a piece of kernel memory with error handling.
** Returns 1 if call succeeded, else 0 (zero).
** taken from Peter Eriksson <pen@lysator.liu.se>
*/

static int getbuf(long addr, char *buf, int len, char *what) {

	if (kvm_read(kd, addr, buf, len) < 0) {
		o_log(DPRI, "getbuf: kvm_read(%08lx, %d) - %s : %m", addr, len, what);
		return (0);
	}

	return (1);
}

/*
** Traverse the inpcb list until a match is found.
** Returns NULL if no match.
*/

static struct socket *getlist(struct inpcbtable *tcbtablep,
	struct inpcbtable *ktcbtablep, const struct in_addr *faddr, int fport,
	const struct in_addr *laddr, int lport)
{
	struct inpcb *kpcbp, pcb;

	if (tcbtablep == NULL)
		return (NULL);

	for (kpcbp = tcbtablep->inpt_queue.cqh_first;
		kpcbp != (struct inpcb *) ktcbtablep;
		kpcbp = pcb.inp_queue.cqe_next)
	{
		if (!getbuf((long) kpcbp, (char *) &pcb, sizeof(struct inpcb), "tcb"))
			break;
#ifdef MASQ_SUPPORT
		if (flags & PROXY) {
			if (faddr->s_addr == proxy && laddr->s_addr != proxy
				&& pcb.inp_fport == fport && pcb.inp_lport == lport)
			{
				return (pcb.inp_socket);
			} else if (pcb.inp_faddr.s_addr == faddr->s_addr &&
						pcb.inp_laddr.s_addr == laddr->s_addr &&
		 				pcb.inp_fport == fport && pcb.inp_lport == lport)
			{
				return (pcb.inp_socket);
			}
		} else
#endif
		if (pcb.inp_faddr.s_addr == faddr->s_addr &&
			pcb.inp_laddr.s_addr == laddr->s_addr &&
			pcb.inp_fport == fport && pcb.inp_lport == lport)

			return (pcb.inp_socket);
	}

	return (NULL);
}

/*
** Return the user number for the connection owner
** taken from Peter Eriksson <pen@lysator.liu.se>
*/

int get_user(int lport, int fport, const struct in_addr *laddr,
			 const struct in_addr *faddr)
{
	struct socket *sockp, sock;
	static struct inpcbtable tcbtable;

	if (!getbuf(nl[N_TCBTABLE].n_value, (char *) &tcbtable, sizeof(tcbtable), "tcbtable"))
		return (-1);

	sockp = getlist(&tcbtable, (struct inpcbtable *) nl[N_TCBTABLE].n_value,
		faddr, fport, laddr, lport);

	if (!sockp)
		return (-1);

	if (!getbuf((long) sockp, (char *) &sock, sizeof sock, "socket"))
		return (-1);

	if (!(sock.so_state & SS_CONNECTOUT))
		return (-1);

	return (sock.so_ruid);
}

#ifdef MASQ_SUPPORT

/*
** Check ident requests for NAT connection
*/

int masq(	int sock, int lport, const struct in_addr *local,
			int rport, const struct in_addr *remote)
{
	nat_t *np,nat;
	u_char os[24],user[16];

	getbuf(nl[N_NATLIST].n_value, (char *) &np, sizeof(np), "nat_instances");

	for (; np != NULL ; np = nat.nat_next) {
		if (!getbuf((long) np,(char *) &nat, sizeof(nat), "nat_entry"))
			break;
		/*
		** Note: Supports multiple output IPs for masquerading
		** Note: Can support both -P and -m at the same time
		*/
		if (nat.nat_p == IPPROTO_TCP &&
			lport == nat.nat_outport && rport == nat.nat_oport &&
			local->s_addr == nat.nat_outip.s_addr &&
			((remote->s_addr == nat.nat_oip.s_addr) ||
			((flags & PROXY) && remote->s_addr == proxy && local->s_addr != proxy)))
		{
			lport = htons(lport);
			rport = htons(rport);

			if (flags & FWD) {
				if (fwd_request(sock, rport, lport, nat.nat_inport, &nat.nat_inip)) {
					o_log(DPRI, "Forward to %s (%d %d) failed.",
						inet_ntoa(nat.nat_inip), lport, nat.nat_inport);
				} else
					return (0);
			}

			if (!find_entry(nat.nat_inip.s_addr, user, sizeof(user) - 1, os, sizeof(os) - 1)) {
				dprintf(sock, "%d , %d : USERID : %s%s%s : %s\r\n",
					lport, rport, OS(os),
					(charset ? " , " : ""),
					(charset ? charset : (u_char *) ""),
					user);
				o_log(PRIORITY,
					"[%s] (NAT) Successful lookup (by IP): %d : %d : USERID : %s : %s",
					inet_ntoa(*remote), lport, rport, OS(os), user);
				return (0);
			}
		}
	}

	return (-1);
}

#endif
