/*
** oidentd - ojnk ident daemon
** Copyright (C) 1998,1999,2000 Ryan McCabe <odin@numb.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
**
** $Id: oidentd_util.c,v 1.3 2000/08/22 06:07:48 odin Exp $
*/

#define _GNU_SOURCE
#include <config.h>
#include <oidentd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <signal.h>
#include <sys/types.h>

#include <oidentd_util.h>

extern u_int flags;

/*
** Setup a listening socket.
*/

int setup_listen(int listen_port, u_long listen_addr) {
	const char one = 1;
	int listenfd, ret;
	struct sockaddr_in sl_sin;

	memset(&sl_sin, 0, sizeof(sl_sin));

	sl_sin.sin_family = AF_INET;
	sl_sin.sin_port = htons(listen_port);
	sl_sin.sin_addr.s_addr = listen_addr;

	listenfd = socket(PF_INET, SOCK_STREAM, 0);

	if (listenfd == -1) {
		o_log(DPRI, "Error: socket: %s", strerror(errno));
		return (-1);
	}

	ret = setsockopt(listenfd, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one));

	if (ret == -1) {
		o_log(DPRI, "Error: setsockopt: %s", strerror(errno));
		return (-1);
	}

	ret = bind(listenfd, (struct sockaddr *) &sl_sin, sizeof(struct sockaddr));
	
	if (ret == -1) {
		o_log(DPRI, "Error: bind: %s", strerror(errno));
		return (-1);
	}

	if (listen(listenfd, 3) == -1) {
		o_log(DPRI, "Error: listen: %s", strerror(errno));
		return (-1);
	}

	return (listenfd);
}

/*
** Drop privileges and run with specified uid/gid.
*/

int drop_privs(uid_t new_uid, gid_t new_gid) {
	int ret;

	if (flags & GID) {
		if (setgid(new_gid) == -1) {
			o_log(DPRI, "Error: setgid(%u): %s", new_gid, strerror(errno));
			return (-1);
		}
	}

	if (flags & UID) {
		struct passwd *pw = getpwuid(new_uid);
		gid_t my_gid;

		if (pw == NULL) {
			o_log(DPRI, "Error: getpwuid(%u): %s", new_uid, strerror(errno));
			return (-1);
		}

		if (flags & GID)
			my_gid = new_gid;
		else
			my_gid = pw->pw_gid;

		ret = initgroups(pw->pw_name, my_gid);

		if (ret == -1) {
			o_log(DPRI, "Error: initgroups(%s, %u): %s", pw->pw_name, my_gid,
				strerror(errno));
			return (-1);
		}

		if (setuid(new_uid) == -1) {
			o_log(DPRI, "Error: setuid(%u): %s", new_uid, strerror(errno));
			return (-1);
		}
	}
	
	return (0);
}

#ifdef HAVE_LIBUDB
/*
** This function will look up the connection in the UDB shared memory tables.
** 
** Return values:
** 
**	If an entry is found which matches a local username, the function will
**	return the matching UID for further processing.
**
**	Otherwise, if a non-local match is found, the reply will be sent directly 
**	to the client and -2 returned.
**
**	If no match is found, -1 is returned.
*/

int get_udb_user(int lport, 
				 int fport, 
				 const struct in_addr *laddr, 
				 const struct in_addr *faddr, 
				 int sock)
{
	struct udb_connection conn;
	struct udb_conn_user buf;
	struct passwd *pw;

	memset(&conn, 0, sizeof(conn));
	conn.from.sin_family = AF_INET;
	conn.from.sin_addr = *laddr;
	conn.from.sin_port = htons(lport);
	conn.to.sin_family = AF_INET;
	conn.to.sin_addr = *faddr;
	conn.to.sin_port = htons(fport);

	o_log(DPRI, "UDB lookup: %s:%d->%s:%d",
		inet_ntoa(conn.from.sin_addr), ntohs(conn.from.sin_port), 
		inet_ntoa(conn.to.sin_addr), ntohs(conn.to.sin_port));

	if (!udb_conn_get(&conn, &buf))
		return (-1);

	/* If the user is local, return his UID */
	if (NULL != (pw = getpwnam(buf.username)))
		return (pw->pw_uid);

	/* User not local, reply with string from UDB table. */
	dprintf(sock, "%d , %d : USERID : %s%s%s : %s\r\n", lport, fport,
			OS("UNIX"), (charset != NULL ? " , " : ""),
			(charset != NULL ? charset : (u_char *) ""), buf.username);
	
	o_log(PRIORITY, "[%s] UDB lookup: %d , %d : (returned %s)",
		inet_ntoa(*faddr), lport, fport, buf.username);

	return (-2);
}
#endif

/*
** Read a line from the identd.spoof file.  If too long, return empty
** Strips leading and trailing whitespace
** Returns EOF or 0
*/

int read_permline(FILE *f, u_char *buf, int bufl) {
	int c,i;
	bufl--;		/* Reserve space for \0 */

	for (i = 0 ; ;) {
		c = getc(f);

		switch (c) {
			case EOF:
				if (!i)
					return EOF;
				/* FALLTHROUGH */
			case '\n':
				if (i >= bufl)
					i = 0;
				while (i > 0 && isspace(buf[i - 1]))
					i--;
				buf[i] = '\0';

				return (0);
			default:
				if (i == 0 && isspace(c))
					break;
				if (i < bufl)
					buf[i++] = c;
		}
	}

	/* NOTREACHED */
	return (0);
}

/*
** Read a socket. (UNP v2)
*/

ssize_t sockread(int fd, u_char *buf, size_t len) {
	u_char c;
	u_int i;
	ssize_t ret;

	if (buf == NULL)
		return (-1);

	for (i = 1; i < len ; i++) {
		ojnk:
			if ((ret = read(fd, &c, 1)) == 1) {
				*buf++ = c;
				if (c == '\n')
					break;
			} else if (ret == 0) {
				if (i == 1)
					return (0);
				else
					break;
			} else {
				if (errno == EINTR)
					goto ojnk;

				return (-1);
			}
	}

	*buf = '\0';
	return (i);
}

/*
** Go background.
*/

int go_background(int ignore) {
	int fd;
	
	switch (fork()) {
		case -1:
			return (-1);
		case 0:
			break;
		default:
			_exit(0);
	}

	if (setsid() == -1)
		return (-1);

	chdir("/");
	umask(DEFAULT_UMASK);

	fd = open("/dev/null", O_RDWR);
	
	if (fd == -1)
		return (-1);

	dup2(fd, 0);
	dup2(fd, 1);
	dup2(fd, 2);

	for (fd = 3; fd < 64 ; fd++) {
		if (fd != ignore)
			close(fd);
	}
	
	return (0);
}

void *xmalloc(size_t size) {
	void *temp = malloc(size);

	if (temp == NULL) {
		o_log(DPRI, "Fatal: malloc: Out of memory.");
		exit(-1);
	}

	return (temp);
}

/*
** Copy at most n-1 characters from src to dest and nul-terminate dest.
** Returns a pointer to the destination string.
*/

char *xstrncpy(char *dest, const char *src, size_t n) {
	u_char *ret = dest;

	if (n == 0)
		return (dest);

	while (--n > 0 && (*dest++ = *src++) != '\0')
		;
	*dest = '\0';

	return (ret);
}


int o_log(int priority, char *fmt, ...) {
	va_list ap;
	ssize_t ret;
#ifdef HAVE_VASPRINTF
	u_char *buf;

	if (priority != LOG_CRIT && (flags & QUIET))
		return (0);

	va_start(ap, fmt);
	ret = vasprintf((char **) &buf, fmt, ap);
	va_end(ap);

	syslog(priority, "%s", buf);
	free(buf);

#else
	u_char buf[4096];

	if (priority != LOG_CRIT && (flags & QUIET))
		return (0);

	va_start(ap, fmt);
	ret = vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);

	syslog(priority, "%s", buf);
#endif

	return (ret);
}

#ifndef HAVE_DPRINTF
int dprintf(int fd, char *fmt, ...) {
	va_list ap;
#ifdef HAVE_VASPRINTF
	u_char *buf;
	ssize_t ret;

	va_start(ap, fmt);
	ret = vasprintf((char **) &buf, fmt, ap);
	va_end(ap);

	ret = write(fd, buf, ret);
	free(buf);

	return (ret);
#else /* ! HAVE_VASPRINTF */
	u_char buf[4096];

	va_start(ap, fmt);
	vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);

	return (write(fd, buf, strlen(buf)));
#endif /* HAVE_VASPRINTF */
}
#endif /* HAVE_DPRINTF */

/*
** Return an IP address and/or hostname.
*/

u_char *hostlookup(u_long lookup_addr, u_char *hostname, size_t len) {
	struct hostent *host = NULL;
	struct in_addr in;

	in.s_addr = lookup_addr;

	host = gethostbyaddr((char *) &in, sizeof(struct in_addr), AF_INET);

	if (host != NULL) {
		if (strlen(host->h_name) + sizeof(" (123.123.123.123)") < len) {
			snprintf(hostname, len, "%s (%s)", host->h_name, inet_ntoa(in));
			return (hostname);
		}
	}

	xstrncpy(hostname, inet_ntoa(in), len);

	return (hostname);
}

/*
** Get the port associated with a tcp service name.
*/

int get_port(const u_char *name) {
	struct servent *servent;

	if (valid_number(name))
		return (atoi(name));

	servent = getservbyname(name, "tcp");
	if (servent != NULL)
		return (ntohs(servent->s_port));

	return (-1);
}

/*
** Return a 32-bit, network byte ordered ipv4 address.
*/

int get_addr(const u_char *const hostname, u_long *g_addr) {
#ifdef HAVE_INET_ATON
	struct in_addr in;

	if (inet_aton(hostname, &in)) {
		*g_addr = in.s_addr;

		return (0);
#else
	*g_addr = inet_addr(hostname);
	if (*g_addr != -1UL) {

		return (0);
#endif
	} else {
		struct hostent *host = gethostbyname(hostname);
		if (host) {
			*g_addr = *((u_long *) host->h_addr);
			return (0);
		}
	}

	return (-1);
}

/*
** Returns non-zero if p points to a valid decimal number
*/

int valid_number(const u_char *p) {
	/*
	** Make sure that the user isn't really trying to specify an octal
	** number... the only valid number that starts with zero is zero
	** itself
	*/
	if (*p == '0')
		return (p[1] == '\0');
	if (*p == '-')
		p++;
	if (*p == '\0' || *p == '0')
		return (0);
	for (; *p != '\0' ; p++) {
		if (!isdigit(*p))
			return (0);
	}

	return (1);
}
