/*
** oidentd - ojnk ident daemon
** Copyright (C) 1998,1999,2000 Ryan McCabe <odin@numb.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
**
** $Id: oidentd_masq.c,v 1.3 2000/10/03 06:13:54 odin Exp $
*/

#include <config.h>
#include <oidentd.h>

#ifdef MASQ_SUPPORT

#include <oidentd_util.h>
#include <oidentd_masq.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>

static void fwdalrm(int sig);

static sigjmp_buf timebuf;

extern u_char *charset;
extern u_char *ret_os;
extern u_int flags;
extern int fwdport;

/*
** Parse the masquerading map.
*/

int find_entry(u_long host, u_char *user, size_t ursz, u_char *os, size_t osz) {
	int c = 0;
	u_int i = 0;
	u_long addr = 0, mask = 0, mask2 = 0;
	FILE *fp = NULL;
	u_char buf[1024], *temp;
#ifdef HAVE_LIBUDB
	struct udb_ip_user ibuf;
	struct in_addr hostaddr;
	
	if (flags & USEUDB) {
		hostaddr.s_addr = host;

		o_log(DPRI, "[%s] UDB lookup...", inet_ntoa(hostaddr));
		
		if (udb_ip_get(&hostaddr, &ibuf)) {
			xstrncpy(user, ibuf.username, ursz);
			xstrncpy(os, "UNIX", ursz);

			o_log(PRIORITY, "Successful UDB lookup: %s : %s",
				inet_ntoa(hostaddr), user);
			
			return (0);
		}
	}
#endif

	fp = fopen(MAP, "r");

	if (fp == NULL) {
		o_log(DPRI, "Error: fopen: %s: %s", MAP, strerror(errno));
		return (-1);
	}

	for (;; i = 0) {
		while (i < sizeof(buf) - 1) {
			c = getc(fp);
			if (c == EOF)
				goto eof;
			if (c == '\n') {
				i = 0;
				continue;
			}
			if (c == '#' && i == 0)
				goto newline;
			if (isblank(c)) {
				if (i > 0)
					break;
				continue;
			}
			buf[i++] = c;
		}

		buf[i] = '\0';

		if (strlen(buf) == sizeof(buf)) {
			o_log(DPRI, "Error: Long line in %s.", MAP);
			while (!isspace(getc(fp)))
				;
		}
		
		temp = strchr(buf, '/');
		if (temp != NULL) {
			*temp++ = '\0';
			if (strchr(temp, '.') || !isdigit(*temp)) {
				if (get_addr(temp, &mask2) == -1) {
					o_log(DPRI, "%s: Invalid mask: %s/%s", MAP, buf, temp);
					goto newline;
				}
			} else {
				mask = strtoul(temp, NULL, 10);
				if (mask < 1 || mask > 32) {
					o_log(DPRI, "%s: Invalid mask: %s/%s", MAP, buf, temp);
					goto newline;
				}
			}
		}

		if (get_addr(buf, &addr) == -1)
			goto newline;

		if (mask)
			mask2 = htonl(~((1 << (32 - mask)) - 1));

		if (mask2) {
			addr &= mask2;
			host &= mask2;
		}

		if (host != addr)
			goto newline;

		for (i = 0 ; i < ursz ;) {
			c = getc(fp);
			if (c == EOF)
				goto eof;
			if (c == '\n') {
				i = 0;
				goto newline;
			}
			if (isblank(c)) {
				if (i > 0)
					break;
				continue;
			}
			user[i++] = c;
		}

		user[i] = '\0';

		if (strlen(user) == ursz) {
			o_log(DPRI, "Error: Long line in %s.", MAP);
			while (!isspace(getc(fp)))
				;
		}

		for (i = 0 ; i < osz ;) {
			c = getc(fp);
			if (c == EOF)
				goto eof;
			if (c == '\n') {
				if (i)
					break;
				goto newline;
			}
			if (isblank(c)) {
				if (i)
					break;
				continue;
			}
			os[i++] = c;
		}

		os[i] = '\0';
		return (0);

newline:
		for (;;) {
			c = getc(fp);
			if (c == EOF)
				goto eof;
			if (c == '\n')
				break;
		}
	}

eof:
	fclose(fp);
	return (-1);
}

/*
** Forward an ident request to another machine, return the response to the
** client that has connected to us and requested it.
*/

int fwd_request(int sock, int rport, int lport, int mport, const struct in_addr *mrelay) {
	extern int fsock;
	u_char buf[128], user[128];
	struct sockaddr_in fr_sin;

	fsock = socket(PF_INET, SOCK_STREAM, 0);
	if (fsock == -1) {
		o_log(DPRI, "Forward error: socket: %s", strerror(errno));
		return (-1);
	}

	memset(&fr_sin, 0, sizeof(fr_sin));
	fr_sin.sin_family = AF_INET;
	fr_sin.sin_port = htons(fwdport);
	fr_sin.sin_addr.s_addr = mrelay->s_addr;

	if (sigsetjmp(timebuf, 1) != 0)
		return (-1);

	signal(SIGALRM, fwdalrm);

	/*
	** Five seconds should be plenty, seeing as we're forwarding to a machine
	** on a local network.
	*/
	alarm(5);
	
	if (connect(fsock, (struct sockaddr *) &fr_sin, sizeof(fr_sin)) == -1) {
		o_log(DPRI, "Forward error: connect: %s", strerror(errno));
		close(fsock);
		return (-1);
	}

	if (dprintf(fsock, "%d , %d\r\n", lport, mport) < 1) {
		o_log(DPRI, "Forward error: write: %s", strerror(errno));
		close(fsock);
		return (-1);
	}

	if (sockread(fsock, buf, sizeof(buf)) < 1) {
		close(fsock);
		return (-1);
	}

	/*
	** Don't want timeout when finished processing request
	*/
	alarm(0);
	close(fsock);

	if (sscanf(buf, "%*d , %*d : USERID :%*[^:]:%127s", user) != 1) {
		o_log(DPRI, "[%s] Forward response error: %s", inet_ntoa(*mrelay), buf);
		return (-1);
	}

	dprintf(sock, "%d , %d : USERID : %s%s%s : %s\r\n",
		rport, mport, OS(ret_os), (charset != NULL ? " , " : ""),
		(charset != NULL ? charset : (u_char *) ""), user);

	o_log(PRIORITY,
		"[%s] Successful lookup (by forward): %d , %d : USERID : %s : %s",
		inet_ntoa(*mrelay), rport, mport, OS(ret_os), user);

	free(user);
	return (0);
}

/*
** Handle the timeout of a forward request.
*/

static void fwdalrm(int sig) {
	extern int fsock;

	o_log(PRIORITY, "Forward timed out.");
	close(fsock);
	siglongjmp(timebuf, sig);
}

#endif
