/*
** freebsd.c	   Low level kernel access functions for FreeBSD 2.x
**
** This program is in the public domain and may be used freely by anyone
** who wants to.
**
** Please send bug fixes/bug reports to: Peter Eriksson <pen@lysator.liu.se>
**
** $Id: freebsd.c,v 1.4 2000/08/21 06:39:27 odin Exp $
*/

#include <config.h>

#include <oidentd.h>
#include <oidentd_util.h>

#include <nlist.h>
#include <kvm.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/sysctl.h>
#include <sys/queue.h>
#include <sys/uio.h>
#include <sys/socketvar.h>
#define KERNEL
#include <sys/file.h>
#undef KERNEL
#include <sys/user.h>
#include <sys/filedesc.h>
#include <sys/proc.h>

#include <net/if.h>
#include <net/route.h>

#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/in_pcb.h>
#include <netinet/tcp.h>
#include <netinet/ip_var.h>
#include <netinet/tcp_timer.h>
#include <netinet/tcp_var.h>

#ifdef INPLOOKUP_SETLOCAL
#define	_HAVE_OLD_INPCB
#endif

extern void *calloc();
extern void *malloc();
extern u_int flags;

struct nlist nl[] =
{
#define N_TCB 0
  { "_tcb" },
#define N_BTEXT 1
  { "_btext" },
  { "" }
};

static kvm_t *kd;
static int nfile;

#ifdef _HAVE_OLD_INPCB
static struct inpcb tcb;
#else
static struct inpcbhead tcb;
#endif

/*
** Open the kernel memory device
*/

int k_open(void) {
	kd = (kvm_t *) kvm_openfiles(NULL, "/dev/mem", NULL, O_RDONLY, NULL);

	if (kd == NULL) {
		o_log(DPRI, "Error: k_open: %s", strerror(errno));	
		return (-1);
	}

	/*
	** Extract offsets to the needed variables in the kernel
	*/

	if (kvm_nlist(kd, nl) < 0) {
		o_log(DPRI, "Error: kvm_nlist: %s", strerror(errno));
		return (-1);
	}

	return (0);
}

/*
** Get a piece of kernel memory with error handling.
** Returns 1 if call succeeded, else 0 (zero).
*/
static int getbuf(u_long addr, char *buf, u_int len, char *what) {
	if (addr < nl[N_BTEXT].n_value ||		/* Overkill.. */
	addr >= (unsigned long) 0xFFC00000 ||
	(addr + len) < nl[N_BTEXT].n_value ||
	(addr + len) >= (unsigned long) 0xFFC00000)
	{

		o_log(DPRI, "getbuf: bad address (%08x not in %08x-0xFFC00000) - %s", addr, nl[N_BTEXT].n_value, what);
		return (0);
    }
    
    if (kvm_read(kd, addr, buf, len) < 0) {
		o_log(DPRI, "getbuf: kvm_read(%08x, %d) - %s : %m", addr, len, what);
		return (0);
	}
    
	return (1);
}

/*
** Traverse the inpcb list until a match is found.
** Returns NULL if no match.
*/
static struct socket *
#ifdef _HAVE_OLD_INPCB
getlist(struct inpcb *pcbp, const struct in_addr *faddr, int fport, const struct in_addr *laddr, int lport)
#else
getlist(struct inpcbhead *pcbhead, const struct in_addr *faddr, int fport, const struct in_addr *laddr, int lport)
#endif
{
#ifdef _HAVE_OLD_INPCB
  struct inpcb *head;
#else
  struct inpcb *head, pcbp;
#endif

#ifdef _HAVE_OLD_INPCB
	if (!pcbp)
		return (NULL);
#else
  head = pcbhead->lh_first;
  if (!head)
    return (NULL);
#endif

#ifdef _HAVE_OLD_INPCB
	head = pcbp->inp_prev;
#endif
#ifdef _HAVE_OLD_INPCB
	do {
	    if (pcbp->inp_faddr.s_addr == faddr->s_addr &&
			pcbp->inp_laddr.s_addr == laddr->s_addr &&
			pcbp->inp_fport == fport && pcbp->inp_lport == lport)
				return (pcbp->inp_socket);
	} while (pcbp->inp_next != head &&
	   getbuf((long) pcbp->inp_next, (char *) pcbp, sizeof(struct inpcb), "tcblist"));
#else
	do {
		if (!getbuf((long) head, (char *) &pcbp, sizeof(struct inpcb), "tcblist"))
			break;
		if (pcbp.inp_faddr.s_addr == faddr->s_addr &&
			pcbp.inp_fport == fport && pcbp.inp_lport == lport)
			return (pcbp.inp_socket);
		head = pcbp.inp_list.le_next;
	} while (head != NULL);
#endif

	return (NULL);
}

/*
** Return the user number for the connection owner
*/

int get_user(int lport, int fport, const struct in_addr *laddr,
			const struct in_addr *faddr) {
	long addr;
	struct socket *sockp;
	struct kinfo_proc *kp;
	int i, nentries;

	if ((kp = kvm_getprocs(kd, KERN_PROC_ALL, 0, &nentries)) == NULL) {
		o_log(DPRI, "Error: kvm_getprocs: %s", strerror(errno));
    	return (-1);
	}

  /* -------------------- TCP PCB LIST -------------------- */

	if (!getbuf(nl[N_TCB].n_value, (char *) &tcb, sizeof(tcb), "tcb"))
		return (-1);

#ifdef _HAVE_OLD_INPCB
	tcb.inp_prev = (struct inpcb *) nl[N_TCB].n_value;
#endif
	sockp = getlist(&tcb, faddr, fport, laddr, lport);

	if (!sockp)
		return (-1);

/*
** Locate the file descriptor that has the socket in question
** open so that we can get the 'ucred' information
*/
	for (i = 0; i < nentries; i++) {
		if (kp[i].kp_proc.p_fd != NULL) {
			int j;
			struct filedesc pfd;
			struct file **ofiles, ofile;

			if (!getbuf((u_long) kp[i].kp_proc.p_fd, (char *) &pfd, sizeof(pfd), "pfd"))
				return (-1);

			ofiles = (struct file **) xmalloc(pfd.fd_nfiles * sizeof(struct file *));
			if (!ofiles)
				return(-1);
      
			if (!getbuf((u_long) pfd.fd_ofiles, (char *) ofiles,
				pfd.fd_nfiles * sizeof(struct file *), "ofiles"))
			{
				free(ofiles);
				return (-1);
			}

			for (j = 0; j < pfd.fd_nfiles; j ++) {
				if (!ofiles[j])	/* might be sparse */
					continue;

	  			if (!getbuf((u_long) ofiles[j], (char *) &ofile, sizeof(struct file), "ofile")) {
					free(ofiles);
					return (-1);
				}

				if (ofile.f_count == 0)
					continue;

				if (ofile.f_type == DTYPE_SOCKET &&
					(struct socket *) ofile.f_data == sockp)
				{
					struct pcred pc;

					if (!getbuf((u_long) kp[i].kp_proc.p_cred, (char *) &pc, sizeof(pc), "pcred"))
					{
						free(ofiles);
						return (-1);
					}

					free(ofiles);
					return (pc.p_ruid);
				}
			}

			free(ofiles);
		}
	}
	return (-1);
}
