
/*
    LinuxWare daemon - Netware like server for Linux

    Copyright (C) 1994, 1995  Ales Dryak <e-mail: A.Dryak@sh.cvut.cz>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include "global.h"
#include "handlers.h"
#include "bindauth.h"
#include "logging.h"
#include "ncp.h"

void ncp_build_reply(struct ncp_reply* rpl,connection* c,int compl,int con)
{
	rpl->p_type=IPX_NCP_REPLY;
	if (server_state==SS_SHUTDOWN || server_state==SS_DOWN) 
		con|=CSTAT_DOWN;
	if (c==NULL)
	{
		rpl->seq=0;
		rpl->c_low=0xFF;
		rpl->c_high=0xFF;
		rpl->task=1;
		rpl->f_stat=compl;
		rpl->c_stat=con;
		return;
	}
	rpl->seq=c->out_seq;
	rpl->c_low=get_conn_id(c);
	rpl->c_high=(get_conn_id(c))/256;
	rpl->task=1;
	rpl->f_stat=compl;
	rpl->c_stat=con;
}

void ncp_raw_send(struct sockaddr_ipx* sipx,char* buf,int size)
{
	LOG_START(LL_DEBUG)
	log_printf("NCP size: %d send to ",size);
	ipx_fprint_saddr(log_file,sipx);
	log_printf("\n");
	ipx_ncp_fdump(log_file,buf,size);
	LOG_END

#if 0
	if (random()%10==0)
	{
		LPRINF(LL_INFO,("LOST!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n"));
		return;
	}
#endif
	
	if (sendto(ncp_sock,(void*)buf,size,0,(struct sockaddr*)sipx,sizeof(*sipx))==-1)
	{
		if (errno==ENETUNREACH)
		{
			notify_unreachable(sipx->sipx_network);
			return;
		}
		LPRINTF(LL_ERROR,("NCP sendto: %s\n",strerror(errno)));
	}
}

void ncp_compl_send(connection* c,int size,int compl,int con)
{
	ncp_build_reply(c->out_buf,c,compl,con);
	ncp_raw_send(&(c->peer_addr),(char*)c->out_buf,
		c->out_size=sizeof(struct ncp_reply)+size);
}

void ncp_send(connection* c,int size)
{
	ncp_compl_send(c,size,COMPL_OK,c->msg_hold?CSTAT_MSG:CSTAT_OK);
}

void ncp_send_failure(connection* c,int compl)
{
	ncp_compl_send(c,0,compl,c->msg_hold?CSTAT_MSG:CSTAT_OK);
}

void ncp_send_ok(connection* c)
{
	ncp_send_failure(c,COMPL_OK);
}

void ipx_ncp_open_dump(struct ncp_request* pkt,int size)
{
	ipx_ncp_open_fdump(stdout,pkt,size);
}

void ipx_ncp_close_dump(struct ncp_request* pkt,int size)
{
	ipx_ncp_close_fdump(stdout,pkt,size);
}

void ipx_ncp_request_dump(struct ncp_request* pkt,int size)
{
	ipx_ncp_request_fdump(stdout,pkt,size);
}

void ipx_ncp_reply_dump(struct ncp_reply* pkt,int size)
{
	ipx_ncp_reply_fdump(stdout,pkt,size);
}

void ipx_ncp_open_fdump(FILE* file,struct ncp_request* pkt,int size)
{
	fprintf(file,"NCP Open request\n");
	fprintf(file,"	sequence #:  %d\n",pkt->seq);
	fprintf(file,"	connection:  %04X\n",pkt->c_low+pkt->c_high*256);
	fprintf(file,"	client task: %d\n",pkt->task);
	fprintf(file,"	fuction:     %d\n",pkt->func);
}

void ipx_ncp_close_fdump(FILE* file,struct ncp_request* pkt,int size)
{
	fprintf(file,"NCP Close request\n");
	fprintf(file,"	sequence #:  %d\n",pkt->seq);
	fprintf(file,"	connection:  %04X\n",pkt->c_low+pkt->c_high*256);
	fprintf(file,"	client task: %d\n",pkt->task);
	fprintf(file,"	fuction:     %d\n",pkt->func);
}

void ipx_ncp_request_fdump(FILE* file,struct ncp_request* pkt,int size)
{
	fprintf(file,"NCP Request\n");
	fprintf(file,"	sequence #:  %d\n",pkt->seq);
	fprintf(file,"	connection:  %04X\n",pkt->c_low+pkt->c_high*256);
	fprintf(file,"	client task: %d\n",pkt->task);
	fprintf(file,"	fuction:     %d\n",pkt->func);
}

void ipx_ncp_reply_fdump(FILE* file,struct ncp_reply* pkt,int size)
{
	fprintf(file,"NCP Reply\n");
	fprintf(file,"	sequence #:  %d\n",pkt->seq);
	fprintf(file,"	connection:  %04X\n",pkt->c_low+pkt->c_high*256);
	fprintf(file,"	client task: %d\n",pkt->task);
	fprintf(file,"	completion:  %d\n",pkt->f_stat);
	fprintf(file,"	conn state:  %d\n",pkt->c_stat);
}

void ipx_ncp_fdump(FILE* file,char* buf,int size)
{
	switch(((struct ncp_request*)buf)->p_type)
	{
	case IPX_NCP_OPEN: ipx_ncp_open_fdump(file,(struct ncp_request*)buf,size);break;
	case IPX_NCP_CLOSE: ipx_ncp_close_fdump(file,(struct ncp_request*)buf,size);break;
	case IPX_NCP_REQUEST: ipx_ncp_request_fdump(file,(struct ncp_request*)buf,size);break;
	case IPX_NCP_REPLY: ipx_ncp_reply_fdump(file,(struct ncp_reply*)buf,size);break;
	default: fprintf(file,"Unknown NCP packet type (%d)",
	          ((struct ncp_request*)buf)->p_type);
	}
}

static void handle_ncp_request(struct ncp_request* pkt,int size,connection* c)
{
	int uid;
	int gid;
	
	if (c==NULL) return;
	c->wdog_timer=c->timer=c->msg_timer=0;
	if (pkt->seq==c->out_seq) /* lost reply, send again */
	{
		ncp_raw_send(&(c->peer_addr),(char*)c->out_buf,c->out_size);
		return;
	}
	if (pkt->seq!=((c->out_seq+1)%256)) return;
	c->out_seq++;
	
	uid=NOBODY_UID;
	gid=NOBODY_GID;
	if (c->logged_obj!=NULL && c->logged_obj->type==BTYPE_USER)
	{
		uid=c->logged_obj->id;
		gid=c->logged_obj->group_id;
	}
	if (setid(uid,gid))
		handle_ncp_request_main(pkt,size,c);
	else
		ncp_send_failure(c,COMPL_FAILURE); /* can't setid */
}

static char output_buf[BUF_SIZE];

static void handle_ncp_open(struct ncp_request* pkt,int size,struct sockaddr_ipx* sipx,connection* c)
{
	int cid;
	if (c!=NULL) return; /* connection is opened (strange)*/
	cid=alloc_conn();
	if (cid==-1)
	/* no free connect slot */
	{
		struct ncp_reply* rpl=(struct ncp_reply*)output_buf;
		ncp_build_reply(rpl,c,COMPL_FAILURE,0);
		ncp_raw_send(sipx,(char*)rpl,sizeof(*rpl));
		return;
	}
	c=get_conn(cid);
	c->peer_addr=*sipx;
	c->timer=0;
	c->wdog_timer=0;
	c->out_seq=0;
	c->msg_hold=0;
	ncp_send_ok(c);
}

static void handle_ncp_close(struct ncp_request* pkt,int size,connection* c)
{
	if (c==NULL) /* already closed or not opened */
	{
		return;
	}
	if (pkt->seq!=(c->out_seq+1)%256) return; /* request out of sequence */
	c->out_seq++;
	ncp_send_ok(c);
	free_conn(c);
}

static void handle_ncp_reply(struct ncp_reply* pkt,int size,struct sockaddr_ipx* sipx,connection* c)
{
}

void handle_ncp(char* buf,int size,struct sockaddr_ipx* sipx)
{
	struct ncp_request* rq=(struct ncp_request*)buf;
	connection* c=NULL;
	int cid;
	cid=rq->c_low+256*rq->c_high;
	if (cid<MAX_CONNECT && get_conn(cid)->alloc && 
	    sipx->sipx_network==get_conn(cid)->peer_addr.sipx_network &&
	    ipx_node_equal(sipx->sipx_node,get_conn(cid)->peer_addr.sipx_node) &&
	    sipx->sipx_port==get_conn(cid)->peer_addr.sipx_port)
		c=get_conn(cid);
	switch(rq->p_type)
	{
	case IPX_NCP_OPEN: handle_ncp_open(rq,size,sipx,c);break;
	case IPX_NCP_CLOSE: handle_ncp_close(rq,size,c);break;
	case IPX_NCP_REQUEST: handle_ncp_request(rq,size,c);break;
	case IPX_NCP_REPLY: handle_ncp_reply((struct ncp_reply*)buf,size,sipx,c);break;
	default: LPRINTF(LL_FULL,("Unknown NCP packet type (%d)",
	          ((struct ncp_request*)buf)->p_type));
	}
}

void init_ncp()
{
	ncp_sock=socket(AF_IPX,SOCK_DGRAM,PF_IPX);
	if(ncp_sock==-1)
	{
		LPRINTF(LL_FATAL,("NCP socket: %s\n",strerror(errno)));
		exit(1);
	}
	if(bind(ncp_sock,(struct sockaddr*)&my_addr,sizeof(my_addr))==-1)
	{
		LPRINTF(LL_FATAL,("bind - NCP: %s\n",strerror(errno)));
		exit(1);
	}
	LPRINTF(LL_INFO,("NCP initialized\n"));
}

void done_ncp()
{
	close(ncp_sock);
	LPRINTF(LL_INFO,("NCP shutdown\n"));
}
