
/*
    IPX service advertising daemon 

    Copyright (C) 1994, 1995  Ales Dryak <e-mail: A.Dryak@sh.cvut.cz>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <linux/ipx.h>
#include "ipxutil.h"
#include "ipxkern.h"
#include "ipxsap.h"

#define TIMER_RATE	(30)
#define EXPIRE_TIME	(180)
#define BROADCAST_TIME	(60)
#define MAX_IFACE	(32)

typedef unsigned char ifc_timer;

typedef struct tagSapTable
{
	ser_name_t name;
	ser_type_t type;
	hop_t hops;
	struct sockaddr_ipx addr;
	ifc_timer timers[MAX_IFACE]; /* 0<timer<EXPIRE_TIME => net is reachable via iface */
				     /* timer>=EXPIPE_TIME => net is not r. */
	struct tagSapTable* next;
} SapTable;

typedef struct tagSapInterface
{
	IPXNode ifnode;
	SapOutput output;
} SapInterface;

#define ifnet output.dest_addr.sipx_network

int sock;
SapPacket* in_buf;
SapTable* stable;
int n_of_ifaces;
SapInterface ifaces[MAX_IFACE];
int time_since_last_bcast=0;

volatile int dump_request=0;
volatile int timer_request=0;
volatile int terminate=0;

int new_log_entry=1;
int debug_option=0;
FILE* log_file=stderr;

#define LOG_ENTRY	{new_log_entry=1;}
#define LOG_START	{if (new_log_entry) {print_time(); new_log_entry=0;}
#define LOG_END		fflush(log_file);}
#define DL_ENTRY	{if (debug_option) LOG_ENTRY}
#define DL_START	{if (debug_option) LOG_START
#define DL_END		LOG_END }
#define DEFAULT_LOGNAME "/dev/null"

void print_time()
{
	time_t t;
	time(&t);
	fprintf(log_file,"\n%s",ctime(&t));
}

SapInterface* get_iface(IPXNet net)
{
  SapInterface* ifc;
  for(ifc=ifaces;ifc<ifaces+n_of_ifaces;ifc++)
  {
    if (ifc->ifnet==net) return ifc;
  }
  return NULL;
}

int ifc_get_index(SapInterface* ifc)
{
  if (ifc==NULL) return -1;
  return ifc-ifaces;
}

int is_expired(ifc_timer* tm)
{
	return *tm>=EXPIRE_TIME;
}

void setup_timers(SapTable* rt,SapInterface* ifc)
{
	ifc_timer* tm;
	
	for(tm=rt->timers;tm<rt->timers+MAX_IFACE;tm++) *tm=EXPIRE_TIME;
	rt->timers[ifc_get_index(ifc)]=0;
}

SapOutput* get_output(int index)
{
  return &(ifaces[index].output);
}

void output_flushall()
{
	int index;
	for(index=0;index<n_of_ifaces;index++)
	{
		ipx_sap_output_flush(get_output(index));
	}
}

void output_set_destination(IPXNode dest_node,IPXPort dest_port)
{
	int index;
	for(index=0;index<n_of_ifaces;index++)
	{
		ipx_sap_output_set_destination(get_output(index),dest_node,dest_port);
	}
}

void output_broadcast(SapTable* st,int down_allow)
{
	SapInterface* ifc;
	ifc_timer* tm;
	for(ifc=ifaces,tm=st->timers;ifc<ifaces+n_of_ifaces;ifc++,tm++)
	{
		if (is_expired(tm))
			ipx_sap_output_response(&(ifc->output),st->type,st->name,&(st->addr),st->hops,down_allow);
		
	}
}

static void output_sendto(void* buffer,int size,struct sockaddr_ipx* daddr)
{
	int res;
	
	DL_ENTRY
	DL_START
	fprintf(log_file,"Sending SAP to ");
	ipx_fprint_saddr(log_file,daddr);
	fprintf(log_file,"\n");
	ipx_sap_fdump(log_file,buffer,size);
	DL_END

	res=sendto(sock,(void*)buffer,size,0,
		(struct sockaddr*)daddr,sizeof(*daddr));
#if 0		
	{
		unsigned long outq;
		int olen;
		int opt;
		
		LOG_START
		if (ioctl(sock,TIOCOUTQ,&outq)<0)
		{
			fprintf(log_file,"ioctl TIOCOUTQ: %s\n",strerror(errno)); 
		}
		else
		{
			fprintf(log_file,"out queue: %li\n",outq);
		}
		if (ioctl(sock,TIOCINQ,&outq)<0)
		{
			fprintf(log_file,"ioctl TIOCINQ: %s\n",strerror(errno)); 
		}
		else
		{
			fprintf(log_file,"in queue: %li\n",outq);
		}
		if (getsockopt(sock,SOL_SOCKET,SO_SNDBUF,&opt,&olen)<0)
		{
			fprintf(log_file,"getsockopt SO_SNDBUF: %s\n",strerror(errno)); 
		}
		else
		{
			fprintf(log_file,"optlen: %i sndbuf: %i\n",olen,opt);
		}
		LOG_END
	}
#endif
	if (res==-1)
	{
		LOG_START
		fprintf(log_file,"sendto: %s\n",strerror(errno));
		LOG_END
	}
}

void fprint_server(FILE* file,SapTable* st)
{
	ifc_timer* tm;
	
	LOG_START
	fprintf(file,"SAP: type: %04X name: ",st->type);
	ipx_sap_fprint_name(file,st->name);
	fprintf(file," \nhops: %i addr: ",st->hops);
	ipx_fprint_saddr(file,&(st->addr));
	fprintf(file," ");
	for(tm=st->timers;tm<st->timers+n_of_ifaces;tm++)
	{
		fprintf(file,"%i",is_expired(tm)?0:1);
		DL_START
		fprintf(file,"(%i)",(int)*tm);
		DL_END
	}
	LOG_END
}

void fdump_servers(FILE* file)
{
	SapTable* st;
	
	LOG_START
	fprintf(file,"IPX server database:\n");
	for(st=stable;st!=NULL;st=st->next)
	{
		fprint_server(file,st);
		fprintf(file,"\n");
	}
	LOG_END
}


SapTable* add_server(ser_type_t type,ser_name_t name,struct sockaddr_ipx* addr,hop_t hops,SapInterface* ifc)
{
	SapTable* st;
	
	if ((st=(SapTable*)malloc(sizeof(SapTable)))==NULL)
	{
		LOG_START
		fprintf(log_file,"ipxsapd: out of memory in add_server\n");
		LOG_END
		return NULL;
	}
	st->type=type;
	ipx_sap_assign_ser_name(st->name,name);
	st->addr=*addr;
	st->hops=hops;
	setup_timers(st,ifc);
	st->next=stable;
	stable=st;
	return st;
}

void delete_server(SapTable* d)
{
	SapTable** s;
	for(s=&stable;*s!=NULL;s=&((*s)->next))
	{
		if (*s==d)
		{
			*s=d->next;
			free(d);
			return;
		}
	}
}

void handle_sap_gns_request(SapEntry* se,SapInterface* src_ifc)
{
	SapTable* cur;
	SapTable* nearest=NULL;
	
	if (ntohs(se->ser_type)==IPX_SAP_GENERAL_RQ) return;
	for(cur=stable;cur!=NULL;cur=cur->next)
	{
		if (ipx_sap_type_equal(cur->type,ntohs(se->ser_type)))
		{
			if (nearest==NULL)
			{
				nearest=cur;
				continue;
			}
			if (nearest->hops>cur->hops)
				nearest=cur;
		}
	}
	if (nearest!=NULL)
	{
		ipx_sap_output_gns_response(&(src_ifc->output),nearest->type,nearest->name,&(nearest->addr),nearest->hops);
	}
	else
	{
		DL_START
		fprintf(log_file,"No servers of type %04X was found\n",ntohs(se->ser_type));
		DL_END
	}
}	

void handle_sap_request(SapEntry* se,SapInterface* src_ifc)
{
	SapTable* cur;
	for(cur=stable;cur!=NULL;cur=cur->next)
	{
		if (ipx_sap_type_equal(cur->type,ntohs(se->ser_type)))
			ipx_sap_output_response(&(src_ifc->output),cur->type,cur->name,&(cur->addr),cur->hops,0);
	}
}

void handle_sap_response(SapEntry* se,SapInterface* src_ifc,IPXNode src_node)
{
	SapTable* cur;
	se->hops=ntohs(se->hops)+1;
	for(cur=stable;cur!=NULL;cur=cur->next)
	{
		if (ipx_sap_name_equal(cur->name,se->ser_name) && 
		    ipx_sap_type_equal(cur->type,ntohs(se->ser_type)))
		{
			/* entry found */
			if (se->hops<=IPX_SAP_SERVER_DOWN)
			{
				/* server ok */
				if (se->hops>cur->hops)
					return;
				if (se->hops==cur->hops)
				{
					/* server has equal 'route' */
					/* update info (near not neccesary) */
					cur->addr.sipx_network=se->network;
					ipx_assign_node(cur->addr.sipx_node,se->node);
					cur->addr.sipx_port=se->port;
				
					/* update timer for iface */
					cur->timers[ifc_get_index(src_ifc)]=0;
 					return;
				}
				
				/* server has better 'route' */
				LOG_START
				fprintf(log_file,"CHANGE ");
				fprint_server(log_file,cur);
				LOG_END
					
				/* update table */
				cur->hops=se->hops;
				cur->addr.sipx_network=se->network;
				ipx_assign_node(cur->addr.sipx_node,se->node);
				cur->addr.sipx_port=se->port;
				setup_timers(cur,src_ifc);
											
				LOG_START
				fprintf(log_file,"\nto     ");
				fprint_server(log_file,cur);
				fprintf(log_file,"\n");
				LOG_END

				/* send info bcast */
				output_broadcast(cur,0);
			}
			else
			{
				/* server down through iface */
				ifc_timer* tm;
				int src_ifc_idx=ifc_get_index(src_ifc);
				
				cur->timers[src_ifc_idx]=EXPIRE_TIME;
				for(tm=cur->timers;tm<cur->timers+n_of_ifaces;tm++)
				{
					if (!is_expired(tm)) return;
				}
				cur->timers[src_ifc_idx]=0;
				cur->hops=IPX_SAP_SERVER_DOWN;
				output_broadcast(cur,1);
				LOG_START
				fprintf(log_file,"DELETE ");
				fprint_server(log_file,cur);
				fprintf(log_file," (service down)\n");
				LOG_END;
				delete_server(cur);
			}
			return;
		}
	}
	/* entry not found */
	if (se->hops<=IPX_SAP_SERVER_DOWN)
	{
		SapTable* st;
		struct sockaddr_ipx addr;
		
		addr.sipx_network=se->network;
		ipx_assign_node(addr.sipx_node,se->node);
		addr.sipx_port=se->port;
		addr.sipx_type=0;
		st=add_server(ntohs(se->ser_type),se->ser_name,&addr,
			se->hops,src_ifc);
		if (st!=NULL)
		{
			/* send info bcast */
			output_broadcast(st,0);
			LOG_START
			fprintf(log_file,"ADD ");	
			fprint_server(log_file,st);
			fprintf(log_file,"\n");
			LOG_END
		}
	}
}

void handle_sap(SapPacket* pkt,int len,struct sockaddr_ipx* sipx)
{
	SapEntry* se=pkt->sap_entries;
	int nent=(len-2)/sizeof(SapEntry);
	SapInterface* src_ifc;

	if ((src_ifc=get_iface(sipx->sipx_network))==NULL)
	{
		LOG_START
		fprintf(log_file,"SAP from non-local net ");
		ipx_fprint_network(log_file,sipx->sipx_network);
		fprintf(log_file," (ignored)\n");
		LOG_END
		return;
	}
	if (len<2)
	{
		LOG_START
		fprintf(log_file,"SAP packet too small len=%i (ignored)\n",len);
		LOG_END
		return;
	}
	if (ipx_node_equal(src_ifc->ifnode,sipx->sipx_node) && 
		(unsigned short)sipx->sipx_port==htons(IPX_SAP_PORT))
	{
		DL_START
		fprintf(log_file,"My packet (ignored)\n");
		DL_END
		return;
	}
	switch (ntohs(pkt->operation))
	{
	case IPX_SAP_OP_REQUEST:
		if (len!=ipx_sap_size(1,IPX_SAP_OP_REQUEST))
		{
			LOG_START
			fprintf(log_file,"SAP packet invalid size (ignored)\n");
			LOG_END
			return;
		}
		output_set_destination(sipx->sipx_node,ntohs(sipx->sipx_port));
		handle_sap_request(se,src_ifc);
		output_flushall();
		break;
	case IPX_SAP_OP_GNS_REQUEST:
		if (len!=ipx_sap_size(1,IPX_SAP_OP_REQUEST))
		{
			LOG_START
			fprintf(log_file,"SAP packet invalid size (ignored)\n");
			LOG_END
			return;
		}
		output_set_destination(sipx->sipx_node,ntohs(sipx->sipx_port));
		handle_sap_gns_request(se,src_ifc);
		output_flushall();
		break;
	case IPX_SAP_OP_RESPONSE:
	/* option: ignore responses from non SAP ports
		if (sipx->sipx_port!=htons(IPX_SAP_PORT)) return;
	*/
		output_set_destination(IPX_BROADCAST,IPX_SAP_PORT);
		for(;nent--;se++) handle_sap_response(se,src_ifc,sipx->sipx_node);
		output_flushall();
		break;
	case IPX_SAP_OP_GNS_RESPONSE:
		LOG_START
		fprintf(log_file,"GNS response should never be received (ignored)\n");
		LOG_END
		break;
	default:
		LOG_START
		fprintf(log_file,"Unknown RIP operation\n");
		LOG_END
		break;
	} 
}

void do_aging()
{
	SapTable* cur;
	int do_broadcast=0;
	
	DL_START
	fprintf(log_file,"DO AGING\n");
	DL_END
	
	time_since_last_bcast+=TIMER_RATE;
	if (time_since_last_bcast>=BROADCAST_TIME)
	{
		do_broadcast=1;
		time_since_last_bcast-=BROADCAST_TIME;
	}
	output_set_destination(IPX_BROADCAST,IPX_SAP_PORT);
	for(cur=stable;cur!=NULL;cur=cur->next)
	{
		ifc_timer* tm;
		int down=1;
		
		for(tm=cur->timers;tm<cur->timers+n_of_ifaces;tm++)
		{
			if (!is_expired(tm)) 
			{
				(*tm)+=TIMER_RATE;
				if (!is_expired(tm)) down=0;
			}
		}
		if (down)
		{
			/* server is down */
			LOG_START
			fprintf(log_file,"DELETE ");
			fprint_server(log_file,cur);
			fprintf(log_file," (timed out)\n");
			LOG_END
			/* send info bcast */
			cur->hops=IPX_SAP_SERVER_DOWN;
			output_broadcast(cur,1);
			/* update table */
			delete_server(cur);
		}
		else
		{
			if (do_broadcast) output_broadcast(cur,0);
		}
	}
	output_flushall();
}

struct ipx_sap_scan_info
{
	SapInterface* ifc;
};

static int ipx_sap_scan_ifc(IPXNet net,IPXNode node,int type,void* data)
{
	struct ipx_sap_scan_info* si=(struct ipx_sap_scan_info*)data;
	
	if (!(si->ifc<ifaces+MAX_IFACE))
	{
		LOG_START
		fprintf(log_file,"too many interfaces (max. %i)\n",(int)MAX_IFACE);
		LOG_END
		exit(1);
	}
	ipx_assign_node(si->ifc->ifnode,node);
	if (!ipx_sap_output_sap_output(&(si->ifc->output),net))
	{
		LOG_START
		fprintf(log_file,"out of memory allocating output buffer\n");
		LOG_END
		exit(1);
	}
	if (net==0)
	{
		LOG_START
		fprintf(log_file,"interface bound to net 0\n");
		LOG_END
		exit(1);
	}
	n_of_ifaces++;
	LOG_START
	ipx_fprint_network(log_file,net);
	fprintf(log_file,":");
	ipx_fprint_node(log_file,node);
	fprintf(log_file,"\n");
	LOG_END
	si->ifc++;
	return 1;	
}

void ipx_sap_get_ifaces()
{
	struct ipx_sap_scan_info si;

	si.ifc=ifaces;
	LOG_START
	fprintf(log_file,"Interface list\n");
	LOG_END
	if (ipx_kern_scan_ifaces(ipx_sap_scan_ifc,(void*)&si)<0)
	{
		LOG_START
		fprintf(log_file,"Cannot read interface information from kernel\n");
		fprintf(log_file,"%s\n",ipx_err_string);
		LOG_END
		exit(1);
	}
  	LOG_START
	fprintf(log_file,"total %d interfaces\n",n_of_ifaces);
	LOG_END
}

void int_handler()
{
	signal(SIGINT,int_handler);
	dump_request=1;
}

void timer_handler()
{
	signal(SIGALRM,timer_handler);
	timer_request=1;
}

void terminate_handler()
{
	signal(SIGTERM,terminate_handler);
	terminate=1;
}

void hup_handler()
{
	signal(SIGHUP,hup_handler);
	terminate=2;
}

void init()
{
	struct itimerval itval;
	struct sockaddr_ipx my_addr;
	
	LOG_ENTRY
	LOG_START
	fprintf(log_file,"Init start\n");
	LOG_END

	ipx_sap_output_func=output_sendto;
	stable=NULL;
	in_buf=ipx_sap_alloc(IPX_SAP_MAX_ENTRIES);
	if (in_buf==NULL) 
	{
		LOG_START
		fprintf(log_file,"FATAL ERROR: out of memory\n");
		LOG_END
		exit(1);
	}

	sock=socket(AF_IPX,SOCK_DGRAM,IPXPROTO_IPXSPECIAL);
	if(sock==-1)
	{
		LOG_START
		fprintf(log_file,"FATAL ERROR: can't create socket: %s\n",strerror(errno));
		LOG_END
		exit(1);
	}

	my_addr.sipx_family=AF_IPX;
	my_addr.sipx_network=IPX_THIS_NET;
	ipx_assign_node(my_addr.sipx_node,IPX_THIS_NODE);
	my_addr.sipx_port=htons(IPX_SAP_PORT);
	my_addr.sipx_type=IPX_SAP_PTYPE;

	/* Permit broadcast output */
	if(ipx_kern_enable_broadcast(sock)!=0)
	{
		LOG_START
		fprintf(log_file,"FATAL ERROR: can't enable broadcasting: %s\n",ipx_err_string);
		LOG_END
		exit(1);
	}

	/* Disable routing and set src address generation by iface */
	if(ipx_kern_dont_route(sock)!=0)
	{
		LOG_START
		fprintf(log_file,"FATAL ERROR: can't disable routing: %s\n",ipx_err_string);
		LOG_END
		exit(1);
	}

	if(bind(sock,(struct sockaddr*)&my_addr,sizeof(my_addr))==-1)
	{
		LOG_START
		fprintf(log_file,"FATAL ERROR: can't bind socket: %s\n",strerror(errno));
		LOG_END
		exit(1);
	}

	ipx_sap_get_ifaces();

	itval.it_interval.tv_sec = TIMER_RATE;
	itval.it_value.tv_sec = TIMER_RATE;
	itval.it_interval.tv_usec = 0;
	itval.it_value.tv_usec = 0;
	if (setitimer(ITIMER_REAL, &itval, (struct itimerval *)NULL) < 0)
	{
		LOG_START
		fprintf(log_file,"FATAL ERROR: can't set itimer: %s\n",strerror(errno));
		LOG_END
		exit(1);
	}
	
	signal(SIGINT,int_handler);
	signal(SIGALRM,timer_handler);
	signal(SIGTERM,terminate_handler);
	signal(SIGHUP,hup_handler);

	{
		SapInterface* ifc;
                output_set_destination(IPX_BROADCAST,IPX_SAP_PORT);
		for(ifc=ifaces;ifc<ifaces+n_of_ifaces;ifc++)
		{
			ipx_sap_output_request(&(ifc->output),IPX_SAP_GENERAL_RQ);
		}
		output_flushall();
	}

	LOG_ENTRY
	LOG_START
	fprintf(log_file,"Init end\n");
	LOG_END
}

void run()
{
	struct sockaddr_ipx sipx;
	while(1)
	{
		int addr_len=sizeof(sipx);
		int size;

		LOG_ENTRY
		size=recvfrom(sock,(void*)in_buf,ipx_sap_size(IPX_SAP_MAX_ENTRIES,IPX_SAP_OP_RESPONSE),
				0,(struct sockaddr*)&sipx,&addr_len);
		if(size==-1 && errno!=EINTR)
		{
			LOG_START
			fprintf(log_file,"recvfrom error: %s\n",strerror(errno));
			LOG_END
			continue;
		}
		if (dump_request)
		{
			fdump_servers(log_file);
			dump_request=0;
		}
		if (timer_request)
		{
			do_aging();
			timer_request=0;
		}
		if (terminate)
		{
			LOG_START
			if (terminate==1)
				fprintf(log_file,"exiting on signal 15 (SIGTERM)\n");
			else
				fprintf(log_file,"exiting on signal 1 (SIGHUP)\n");
			LOG_END
			return;
		}
		if (size==-1) continue;
		DL_START
		fprintf(log_file,"SAP from: ");
		ipx_fprint_saddr(log_file,&sipx);
		fprintf(log_file,"\n");
		ipx_sap_fdump(log_file,in_buf,size);
		DL_END
		handle_sap(in_buf,size,&sipx);
	}
}

void done()
{
	SapTable* cur;
	
	LOG_ENTRY
	LOG_START
	fprintf(log_file,"Shutdown start\n");
	LOG_END

	output_set_destination(IPX_BROADCAST,IPX_SAP_PORT);
	for(cur=stable;cur!=NULL;cur=cur->next)
	{
		LOG_START
		fprintf(log_file,"DELETE ");
		fprint_server(log_file,cur);
		fprintf(log_file," (shutdown)\n");
		LOG_END
		/* send info bcast */
		cur->hops=IPX_SAP_SERVER_DOWN;
		output_broadcast(cur,1);
		/* update table */
		delete_server(cur);
	}
	output_flushall();

	if (close(sock)!=0)
	{
		LOG_START
		fprintf(log_file,"close error: %s\n",strerror(errno));
		LOG_END
	}
	
	LOG_ENTRY
	LOG_START
	fprintf(log_file,"Shutdown end\n");
	LOG_END
}

void help()
{
	fprintf(stderr,
		"IPX service advertising daemon v0.91, (c) Ales Dryak, 1995\n"
		"Usage: ipxsapd [-d] [<path to log file>]\n");
}

int parse_cmdline(int argc,char** argv)
{
	char* log_name;
	argc--,argv++;
	for(;*argv && **argv=='-';argc--,argv++)
	{
		if (*(*argv+1)=='d')
			debug_option=1;
		else
		{
			fprintf(stderr,"Unknown command line option %s\n",*argv);
			help();
			exit(1);
		}
	}
	if (argc>1)
	{
		fprintf(stderr,"Too many parameters");
		help();
		exit(1);
	}
	if (argc==0 && debug_option)
	{
		log_file=stdout;
		return 0;
	}
	log_name=DEFAULT_LOGNAME;
	if (argc==1) log_name=*argv;
	if ((log_file=fopen(log_name,"a"))==NULL)
	{
		perror(log_name);
		exit(1);
	}
	return 1;
}

static void daemonize()
{
	int fd,c;
	
	if ((c = fork()) > 0) exit(0);
  	if (c < 0)
  	{
		fprintf(stderr, "ipxripd: can't fork: %s\n",strerror(errno));
		exit(1);
	}

	close(0);
	close(1);
	close(2);
	if ((fd = open("/dev/tty", O_RDWR)) >= 0) 
	{
		ioctl(fd, TIOCNOTTY, NULL);
		close(fd);
	}
}

int main(int argc,char** argv)
{
	if (parse_cmdline(argc,argv)) daemonize();
	init();
	run();
	done();
	return 0;
}
