
/*
    IPX support library - kernel dependent functions

    Copyright (C) 1994, 1995  Ales Dryak <e-mail: A.Dryak@sh.cvut.cz>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include <errno.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <net/route.h>
#include "ipxkern.h"

#define MAX_IFC (256)

int ipx_kern_scan_rtable(IPXrtScanFunc f,void* data)
{
	int sock;
	struct ifreq itf_table[MAX_IFC];
	struct ifconf itf_info;
	struct ifreq* ifr;
	struct sockaddr_ipx* sipx;
	int bytes_left;

	sock=socket(AF_IPX,SOCK_DGRAM,PF_IPX);
	if (sock==-1)
	{
		sprintf(ipx_err_string,"open socket: %s",strerror(errno));
		return -1;
	}
	
	itf_info.ifc_len=sizeof(itf_table);
	itf_info.ifc_buf=(char*)itf_table;
#if 0
	printf("itf_table: %p itf_info: %p ifc len: %i\n",itf_table,&itf_info,itf_info.ifc_len);
#endif	
	
	if(ioctl(sock,SIOCGIFCONF,(void *)&itf_info)==-1)
	{
                sprintf(ipx_err_string,"ioctl SIOGIFCONF: %s",strerror(errno));
		close(sock);
		return -1;
	}

#if 0
	printf("ifc len: %i\n",itf_info.ifc_len);
#endif
	bytes_left=itf_info.ifc_len;
	for(ifr=itf_info.ifc_req;
		bytes_left>=sizeof(*ifr);ifr++,bytes_left-=sizeof(*ifr))
	{
		int type=0;
		
		sipx=(struct sockaddr_ipx*)&(ifr->ifr_addr);
#if 0
		printf("dev: %s ",ifr->ifr_name);
		printf("addr: ");
		ipx_print_saddr(sipx);
		printf("\n");
#endif
		
		if (sipx->sipx_special&IPX_INTERNAL) type|=IPX_KRT_INTERNAL;
		if (!f(sipx->sipx_network,sipx->sipx_network,
		       sipx->sipx_node,type,data))
		{
			close(sock);
			return 0;
		}
  	}
  	close(sock);
  	return 1;
}

struct scan_ifc_info
{
	IPXifcScanFunc f;
	void* data;
};

static int ipx_kern_scan_iface(IPXNet net,IPXNet rt_net,IPXNode rt_node,int type,void* data)
{
	struct scan_ifc_info* p=(struct scan_ifc_info*) data;
	if ((type & IPX_KRT_ROUTE)==0)
		return (p->f)(net,rt_node,type,p->data);
	else
		return 1;
}

int ipx_kern_scan_ifaces(IPXifcScanFunc f,void* data)
{
	struct scan_ifc_info sii;
	sii.f=f;
	sii.data=data;
	return ipx_kern_scan_rtable(ipx_kern_scan_iface,(void*)&sii);
}

static int ipx_kern_scan_addr(IPXNet net,IPXNode node,int type,void* data)
{
	struct sockaddr_ipx* addr=(struct sockaddr_ipx*)data;
	
	addr->sipx_network=net;
	ipx_assign_node(addr->sipx_node,node);
	if (type & IPX_KRT_INTERNAL) return 0;
	return 1;
}

int ipx_kern_get_internet_addr(struct sockaddr_ipx* addr)
{
	addr->sipx_family=AF_IPX;
	addr->sipx_type=IPX_USER_PTYPE;
	addr->sipx_port=0;
	addr->sipx_network=IPX_THIS_NET;
	ipx_assign_node(addr->sipx_node,IPX_THIS_NODE);
	return ipx_kern_scan_ifaces(ipx_kern_scan_addr,(void*)addr)<0?-1:0;	
}

int ipx_kern_route_add(int sock,IPXNet net,IPXNet rt_net,IPXNode rt_node)
{
	struct rtentry rt;
	struct sockaddr_ipx* sr;
	struct sockaddr_ipx* st;
	
	sr=(struct sockaddr_ipx*)&rt.rt_gateway;
	st=(struct sockaddr_ipx*)&rt.rt_dst;
	
	sr->sipx_family=st->sipx_family=AF_IPX;
	st->sipx_network=net;
	sr->sipx_network=rt_net;
	ipx_assign_node(sr->sipx_node,rt_node);
	rt.rt_flags=RTF_GATEWAY;

	if(ioctl(sock,SIOCADDRT,(void *)&rt)!=0)
	{
		sprintf(ipx_err_string,"ioctl SIOCADDRT: %s",strerror(errno));
		return -1;
	}
	return 0;
}

int ipx_kern_route_delete(int sock,IPXNet net)
{
	struct rtentry rt;
	struct sockaddr_ipx* sr;
	struct sockaddr_ipx* st;
	
	sr=(struct sockaddr_ipx*)&rt.rt_gateway;
	st=(struct sockaddr_ipx*)&rt.rt_dst;
	
	sr->sipx_family=st->sipx_family=AF_IPX;
	st->sipx_network=net;
	rt.rt_flags=RTF_GATEWAY;

	if(ioctl(sock,SIOCDELRT,(void *)&rt)!=0)
	{
		sprintf(ipx_err_string,"ioctl SIOCDELRT: %s",strerror(errno));
		return -1;
	}
	return 0;
}

int ipx_kern_enable_broadcast(int sock)
{
	int opt=1;
	/* Permit broadcast output */
	if(setsockopt(sock,SOL_SOCKET,SO_BROADCAST, &opt,sizeof(opt))==-1)
	{
		sprintf(ipx_err_string,"setsockopt SO_BROADCAST: %s",strerror(errno));
		return -1;
	}
	return 0;
}

int ipx_kern_dont_route(int sock)
{
	int opt=1;
	/* Disable routing and set src address generation by iface */
	if(setsockopt(sock,SOL_SOCKET,SO_DONTROUTE, &opt,sizeof(opt))==-1)
	{
		sprintf(ipx_err_string,"setsockopt SO_DONTROUTE: %s",strerror(errno));
		return -1;
	}
	return 0;
}
