/**************************************************************************
  Message Daemon 
    by Matt Hartley   hartlw@rpi.edu
  taken over from Sammy's Message Daemon v0.3
    by Sam Creasey, sammy@freenet.akron.oh.us
  with his permission.

  Please read the COPYRIGHT and README files for more information.
  Any changes to this code are documented in CHANGES.
  Any known bugs are documented in BUGS.
***************************************************************************/
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <ctype.h>
#include <stdio.h>

#define MAX_STRINGS 128
#define VERSION 1.00
#define MAINTAINER "hartlw@rpi.edu (Matt Hartley)"

typedef struct daemon_struct {
  char hostname[64],mesg[MAX_STRINGS][256],c;
  FILE *msgfp,*fp;
  int fromlen,port,pos;
  short debugmode,inputlock,portadd,noportgiven;
  struct hostent *hp;
  struct sockaddr_in sin,fsin;
} daemon_struct;

extern int errno;

/******************** Usage functions ***********************************/

void usage(void) {
/* how to call the program */
  fprintf(stderr,"Message Daemon version %0.2f          %s\n",VERSION,MAINTAINER);
  fprintf(stderr,"Usage: msgdaemon [-d] [-s] [-f <filename>] [-c <command>] port\n");
  fprintf(stderr,"       msgdaemon --help\n");
  fprintf(stderr,"Please consult the man page for information on each option.\n\n");
  exit(1);
} /* void usage */

void more_on_options(void) {
/* the --help option returns this */
  fprintf(stderr,"Message Daemon version %0.2f          %s\n",VERSION,MAINTAINER);
  fprintf(stderr,"Usage: msgdaemon [-d] [-s] [-f <filename>] [-c <command>] port\n");
  fprintf(stderr,"       msgdaemon --help\n\n");
  fprintf(stderr,"-d:\t\tEnable debug messages\n\n");
  fprintf(stderr,"-s:\t\tMessage taken from user input\n");
  fprintf(stderr,"-f <filename>:\tMessage taken from file <filename>\n");
  fprintf(stderr,"-c <command>:\tMessage taken from output of <command>\n");
  fprintf(stderr,"\n--help:\t\tThis message\n\n");
  fprintf(stderr,"port:\t\tThe port for the message daemon\n\n");
  exit(1);
} /* void more_on_options */

/************** Setup and checking functions *****************************/

void setup_daemon(daemon_struct *daemon) {
/* Setup the daemon structure */
  daemon->debugmode=0;
  daemon->inputlock=0;
  daemon->portadd=0;
  daemon->noportgiven=1;
  daemon->pos=0;
} /* void setup_daemon */

void inputlock_warning(char c) {
  fprintf(stderr,"-%c has been ignored for an earlier input!\n",c);
} /* void inputlock_warning */

void check_for_important_data(daemon_struct *daemon) { 
/* Check to make sure we have an input and a port */
 if (!daemon->inputlock) {
   fprintf(stderr,"Error, no input provided.\n");
   usage();
 }    
 if (daemon->noportgiven) {
   fprintf(stderr,"Error, no port given.\n");
   usage();
 }
} /* void check_for_important_data */

/*************** Message input functions ********************************/

FILE *pipe_input_command(daemon_struct *daemon,char *command) {
/* get message fileptr using stdio pipe commands to get message*/
  FILE *cmdpipe;
  
  if (daemon->debugmode)
    printf("Opening pipe to command: \"%s\"....",command);
  if ((cmdpipe=popen(command,"r"))==NULL) {
    perror("popen");
    exit(1);
  }  
  daemon->msgfp=cmdpipe;
  if (daemon->debugmode)
    printf("done.\n");
  
  return(cmdpipe);
} /* FILE *pipe_input_command */

void file_input_command(daemon_struct *daemon,char *filename) {
/* get message fileptr by reading in message in from a file */
  if ((daemon->msgfp=fopen(filename,"r"))==NULL) {
         printf("File open error.\n");
         exit(2);
  }
  printf("File %s opened and processed successfully.\n",filename);        
} /* void file_input_command */

/*************** Wait for a connection function ***************************/

void wfc(daemon_struct daemon,register int i,register int s,register int ns) {
/* Wait for connections */
 register int temp;

 while (1) {    
    /* Accept connections.  When accepted, ns will be connected to the 
       client.  fsin will contain the address of the client. */
 if (daemon.debugmode)
   printf("Accepting queries...\n");
 if ((ns=accept(s,&daemon.fsin,&daemon.fromlen))<0) {
   perror("server: accept");
   exit(1);
 }
 printf("Connection accepted.\n");
 
 /* Output the message to the client */
 if (daemon.debugmode)   
   printf("Sending the message...\n");
 for (i=0; i<daemon.pos+1; i++) 
   send(ns, daemon.mesg[i], strlen(daemon.mesg[i]), 0);  
 
/* Terminate the connection with close(), since we're done with both sides. */
 temp=close(ns);
 if (!temp) {
   if (daemon.debugmode)
     printf("Successfully closed connection.\n");
 }
 else 
   if (daemon.debugmode)
     printf("Error closing connection\n");  
 }
} /* void wfc */  

/*********************** Main function *************************************/

void main(int ac, char *av[]) {
 daemon_struct daemon;
 register int i, s, ns;
 int params;
 FILE *cmdpipe=NULL;

 /* Initialize the daemon's settings */
 setup_daemon(&daemon);

 /* Command line parsing */
 if (ac>=2) {
  for (params=1;params<ac;params++) {
   if (av[params][0]=='-') {
     switch(toupper(av[params][1])) {

       /* File input */
       case 'F':  if (!daemon.inputlock) {
                    file_input_command(&daemon,av[++params]);
                    daemon.inputlock=1;
                  }
                  else inputlock_warning(av[params++][1]);
                  break;

       /* command input */
       case 'C': if (!daemon.inputlock) {
                   cmdpipe=pipe_input_command(&daemon,av[++params]);
                   daemon.inputlock=1;
                 }
                 else inputlock_warning(av[params++][1]);
                 break;

       /* enable debug mode */
       case 'D': daemon.debugmode=1;
                 break;

       /* standard input */ 
       case 'S': if (!daemon.inputlock) {
                  printf("Enter your message followed by Enter and Ctrl-D:\n");
                  daemon.msgfp=stdin;
                  daemon.inputlock=1;
                 }
                 else inputlock_warning(av[params][1]);       
                 break;

       /* any -- options */
       case '-': if (toupper(av[params][2])=='H')
                   more_on_options();
                 else
                   usage();
                 break;

       /* otherwise incorrect option */
       default:  fprintf(stderr,"%c is not a valid option.\n",av[params][1]);
                 usage();
                 break;
     }
   }
   else
   /* if it is a number it must be the port number */
   if (isdigit(av[params][0])) {
     daemon.portadd=params-1;
     daemon.noportgiven=0;
   }
   else {
     fprintf(stderr,"%s is not a valid command line option.\n",av[params]);
     usage();
   }
  }
 }
 else
   usage(); 

 /* Check to make sure we have an input and a port */
 check_for_important_data(&daemon);

 /* Read in the message from its input FILE pointer to an array of strings */
 while ((daemon.pos<MAX_STRINGS)&&(fgets(daemon.mesg[daemon.pos++],256,daemon.msgfp)!=NULL));

 if (cmdpipe!=NULL)
   pclose(cmdpipe);

 if (daemon.debugmode) {
   printf("Reading in strings...done.\n");
   printf("Converting port number...");
 }
 daemon.port = atoi(av[1+daemon.portadd]);
 if (daemon.debugmode) {
   printf("done.\n");

   printf("Message: \n");
   for (i=0; i<daemon.pos+1; i++)
     printf("%s",daemon.mesg[i]);
 
  /* Get a stream socket to use that will be in the Internet domain. */
  printf("Getting a socket...");
 }
 if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
   perror("server: socket");
   exit(1);
 }
 if (daemon.debugmode) 
   printf("done.\n");
 
 /* Create the address that we will be binding to and put the port number 
    into network byte order. */
 daemon.sin.sin_family=AF_INET;
 daemon.sin.sin_port=htons(daemon.port);
 daemon.sin.sin_addr.s_addr=htonl(inet_addr("0.0.0.0"));

 /*Bind the address to the socket.*/
 if (daemon.debugmode) 
   printf("Binding to a socket...");
 if (bind(s, &daemon.sin, sizeof(daemon.sin)) < 0) {
   perror("server: bind");  
   exit(1);
 }
 if (daemon.debugmode)
   printf("done.\n");
 printf("Daemon successfully bound to socket.\n");

 /* Listen on the socket. */
 if (daemon.debugmode) 
   printf("Listening for queries...\n");
 if (listen(s, 5) < 0) {
   perror("server: listen");
   exit(1);
 }

 /* Wait for connections */
 wfc(daemon,i,s,ns);
}

