#include <qbttngrp.h>
#include <qchkbox.h>
#include <qpushbt.h>
#include <qlistbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qdatetm.h>
#include <kapp.h>

#include "talkwidget.h"
#include "simedit.h"
#include "options.h"
#include "filetransfer.h"

TalkWidget::TalkWidget (QWidget *parent=0, const char *name=0)
  : QWidget (parent, name) {

  setCaption (i18n ("Talking..."));

  strangerName = new QLabel (i18n ("nobody"), this);
  strangerName->setMinimumHeight (strangerName->sizeHint().height());
  fullName = "";

  disconn = new KEasyButton (i18n ("Disconnect"), this, 0, -1,
                             this, SLOT (disconnectPressed ()),
                             i18n ("Closes the connection, but leaves "
                                   "window open"));
  disconn->setEnabled (FALSE);
  disconn->setFocusPolicy (QWidget::NoFocus);
  close = new KEasyButton (i18n ("Close"), this, 0, -1,
                           this, SLOT (closePressed ()),
                           i18n ("Closes connection and window"));
  close->setFocusPolicy (QWidget::NoFocus);

  text = new SimpleEdit (this);
  text->setMinimumHeight (3 * text->fontMetrics().height ());
  text->setReadOnly (TRUE);
  text->setFocusPolicy (QWidget::NoFocus);

  QBoxLayout *topLayout = new QVBoxLayout (this, 5);
  QBoxLayout *topLineLayout = new QHBoxLayout ();

  topLayout->addLayout (topLineLayout);
  topLineLayout->addWidget (strangerName, 10);
  topLineLayout->addWidget (disconn);
  topLineLayout->addWidget (close);
  topLayout->addWidget (text, 10);

  topLayout->activate ();

  doConnect = new Connect (this);
  doCommunication = new Communication (this);

  connect (doConnect, SIGNAL (statusMessage (const char *)),
                      SLOT (sendMessage (const char *)));
  connect (doConnect, SIGNAL (failed (const char *)),
                      SLOT (connectFailed (const char *)));
  connect (doConnect, SIGNAL (succeeded (int)),
                      SLOT (connectSucceeded (int)));
  connect (doConnect, SIGNAL (newUserName ()),
                      SLOT (nameChanged ()));

  connect (doCommunication, SIGNAL (connectionClosed (bool)),
                            SLOT (disconnect (bool)));
  connect (doCommunication, SIGNAL (dataReceived (QString)),
                            SLOT (dataReceived (QString)));
  connect (doCommunication, SIGNAL (clientTypeChanged ()),
                            SLOT (nameChanged ()));
  connect (doCommunication, SIGNAL (commandReceived (QString, QString)),
                            SLOT (commandReceived (QString, QString)));
  status = st_fresh;
}

TalkWidget::~TalkWidget () {

  disconnect (FALSE);
}

void TalkWidget::startConnection (const char *user) {
  if (status != st_fresh) return;   // connect only once
  status = st_connecting;
  doConnect->establish (user);
}

void TalkWidget::sendData (QString data) {
  if (status == st_connected)
    doCommunication->sendData (data);
}

void TalkWidget::sendCommand (QString command, QString params) {
  if (status == st_connected || status == st_mute)
    doCommunication->sendCommand (command, params);
}

void TalkWidget::setTalkFont (const QFont &font) {
  text->setFont (font);
}

void TalkWidget::closeEvent (QCloseEvent *event) {
  event->ignore ();
  emit closeRequest (this);
} 

void TalkWidget::closePressed () {
  emit closeRequest (this);
}

void TalkWidget::disconnectPressed () {
  disconnect (FALSE);
}

void TalkWidget::connectFailed (const char *msg) {
  sendMessage (QString (msg) + i18n (" - Unable to establish connection!"));
  status = st_disconnected;  
}

void TalkWidget::connectSucceeded (int connectionSocket) {

  doCommunication->startCommunication (connectionSocket);
  status = st_connected;
  if (options->getBeepOnConnect ())
    KApplication::beep ();
  disconn->setEnabled (TRUE);
  sendMessage (i18n ("Connection established"));
  emit newStatusRequest ();
}

void TalkWidget::dataReceived (QString data) {
  text->insertData (data);
}

void TalkWidget::disconnect (bool fromRemote) {

  if (status != st_connected && status != st_connecting) return;
  QString now = QTime::currentTime ().toString ();
  if (fromRemote)
    sendMessage (i18n ("Connection closed by your partner"));
  else
    sendMessage (i18n ("Connection closed by you"));
  text->insertLine (i18n ("... connection closed at ") + now);
  text->setCursorPosition (text->numLines () - 1, 0);
  status = st_disconnected;
  delete doConnect;
  delete doCommunication;
  disconn->setEnabled (FALSE);
  emit newStatusRequest ();
  if (options->getCloseOnDisconnect ())
    emit closeRequest (this);
}

void TalkWidget::nameChanged () {
  QString client;
  int main, sub, subsub;

  switch (doCommunication->getTalkClient (main, sub, subsub)) {
    case Communication::client_unknown:
      client = i18n (" [unknown]");
      break;
    case Communication::client_talk:
      client = " [talk]";
      break;
    case Communication::client_ytalk:
      client = " [yalk]";
      break;
    case Communication::client_chitchat:
      client = " [ChitChat]";
      break;
    case Communication::client_ktalk:
      client.sprintf (" [ktalk %d.%d.%d]", main, sub, subsub);
      break;
  }
  if (fullName.isEmpty ())
    strangerName->setText (doConnect->getUserName () + client);
  else
    strangerName->setText (fullName + " (" + doConnect->getUserName () + ")" +
                           client);
}

void TalkWidget::commandReceived (QString command, QString params) {
  if (command == "FullName") {
    fullName = params;
    emit nameChanged ();
  } else if (command == "PingReq") {
    // send Ping Acknowledge
    doCommunication->sendCommand ("PingAck", params);
  } else if (command == "PingAck") {
    emit receivedPingAckCommand (params);
  } else if (command == "FileTransfer") {
    (void) new FileTransferReceive (params, getName ());
  } else
    KDEBUG1 (KDEBUG_WARN, 3900, "Received an unknown command: %s", 
                                params.data ());
}

void TalkWidget::sendMessage (const char *msg) {
  emit message (QString ("[") + doConnect->getUserName () + "] " + 
                QTime::currentTime ().toString () + ": " + msg);
}
