#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <kapp.h>
#include <qmlined.h> 
#include <qpainter.h>
#include <qkeycode.h>
#include <qclipbrd.h>

#include "simedit.h"
#include "options.h"

SimpleEdit::SimpleEdit (QWidget *parent=0, const char *name=0) 
  : QMultiLineEdit( parent, name ) {

  // allways have a vertical scrollbar
  clearTableFlags (Tbl_autoVScrollBar);
  setTableFlags (Tbl_vScrollBar);
}

void SimpleEdit::keyPressEvent (QKeyEvent *k) {

  // key bindings for cursor keys, just the same as in a ReadOnly QMultiLineEdit
  // are used here also for non ReadOnly (since Cursor remains at the end of
  // the text all the time
  // (see QMultiLine::KeyPressEvent in qmlined.cpp) 
  int pageSize = viewHeight() / cellHeight();

  switch ( k->key() ) {
    case Key_Left:
       setXOffset( xOffset() - viewWidth()/10 );
       break;
    case Key_Right:
       setXOffset( xOffset() + viewWidth()/10 );
       break;
    case Key_Up:
       setTopCell( topCell() - 1 );
       break;
    case Key_Down:
       setTopCell( topCell() + 1 );
       break;
    case Key_Next:
       setTopCell( topCell() + pageSize );
       break;
    case Key_Prior:
       setTopCell( QMAX( topCell() - pageSize, 0 ) );
       break;
  }

  // key bindings for ASCII keys, only if NOT ReadOnly
  if (isReadOnly ()) {
    k->ignore();
    return;
  }

  char ch = (char) k->ascii();
  if (ch == '\r') ch = '\n';
  QString data = translateKey (ch);
  insertData (data);
  emit keyPressed (data);
}

void SimpleEdit::mousePressEvent( QMouseEvent *m )
{
  if ( m->button() == LeftButton ) {
    QMultiLineEdit::mousePressEvent (m);
  } else if ( m->button() == MidButton && !isReadOnly ()) {
    const char *t = QApplication::clipboard()->text();
    QString translated;
    for (unsigned int i = 0; i < strlen (t); i++)
      translated += translateKey (t [i]);
    if (!translated.isEmpty()) {
      insertData (translated);
      emit keyPressed (translated);
    }
  }
}

void SimpleEdit::reset() {
  clear ();
}

void SimpleEdit::insertData (QString &data) {
  QString sendingString, partString;
  for (unsigned int i = 0; i < data.length (); i++) {
    char ch = data [i];
    if (ch == 127 || ch == -2) ch = 8;
    if (ch == '\r') ch = '\n';
    if (ch >= 32 || ch == '\t' || ch < -2)
      partString += ch;
    else {
      if (!partString.isEmpty ()) {
        appendText (partString);
        partString = "";
      }
      int y = numLines () - 1, x = lineLength (y);
      setCursorPosition (y, x);
      switch (ch) {
        case '\a':   // BELL
          kapp->beep ();
          break;
        case '\n':   // CR
          newLine ();
          break;
        case '\b':   // BACKSPACE
          if (x == 0) continue; // ignore backspace if line is empty 
          else backspace ();
          break;
        case 23:       // word erase
          if (x == 0)
            continue;   // ignore word erase if line is empty
          else {
            QString t = textLine (y);
            int n;
            for (n = t.length () - 1; n >= 0 && isspace (t[n]); n--);
            for (; n >= 0 && !isspace (t[n]); n--);
            setCursorPosition (y, n + 1);
            killLine ();
          }
          break;
        case 21:       // line erase
          if (lineLength (numLines () - 1) == 0)
            continue;   // ignore word erase if line is empty
          else {
             setCursorPosition (y, 0);
             killLine ();
          }
          break;
        default:
          continue;
      }
    }
    sendingString += ch;
  }
  if (!partString.isEmpty ())
    appendText (partString);
  data = sendingString;
}

QString SimpleEdit::translateKey (char ch) {
  const char *westeurope [95] =
    {"","","","","","","","","","","","","","","","",
     "","","","","","","","","","","","","","","","A",
     "A","A","","Ae","Aa","Ae","","E","E","E","","I","I","I","","",
     "","O","O", "O","","Oe","","Oe","U","U","U","Ue","","","ss","a",
     "a","a","","ae","aa","ae","","e","e","e","","i","i","i","","",
     "","o","o","o","","oe","","oe","u","u","u","ue","","",""};
  const char *easteurope [95] =
    {"","","","","L","","","","S","","T","","","Z","","",
     "","","","","l","","","","s","","t","","","z","","R",
     "A","","","A","L","","","C","E","","","E","I","","D","",
     "","N","O","O","","O","","R","U","U","U","U","Y","","","r",
     "a","","","a","l","","","c","e","","","e","i","","d","d",
     "","n","o","o","","","","r","u","u","u","u","y","",""};

  if (options->getTranslate () != Options::OPT_notrans && 
      (unsigned char) ch >= 161) {
    QString result;
    if (options->getTranslate () == Options::OPT_westeurope)
      result = westeurope [(unsigned char) ch - 161];
    else if (options->getTranslate () == Options::OPT_easteurope)
      result = easteurope [(unsigned char) ch - 161];
    if (!result.isEmpty ())
      return result;
  }
  return QString (&ch, 2);
}

void SimpleEdit::appendText (QString s) {
  int y = numLines () - 1;

  QString line = textLine (y) + s;
  if (textWidth (&line) + 2 < viewWidth ())
    insertAt (s, y, lineLength (y));
  else {
    setUpdatesEnabled (FALSE);
    removeLine (y);
    while (!line.isEmpty () && textWidth (&line) + 2 >= viewWidth ()) {
      unsigned int currPos = 0; 
      // look for the end of the first word
      while (currPos < line.length () && isspace (line [currPos])) currPos++;
      while (currPos < line.length () && !isspace (line [currPos])) currPos++;

      QString part = line.left (currPos);
      if (textWidth (&part) + 2 >= viewWidth ()) {
        // first word doesn't fit in line, break it
        currPos = 1;
        do {
          currPos++;
          part = line.left (currPos);
        } while (textWidth (&part) + 2 < viewWidth ());
        currPos--;
      } else {
        // first word fits, so look for more words to fit
        int lastPos;
        do {
          lastPos = currPos;
          while (currPos < line.length () && isspace (line [currPos])) 
            currPos++;
          while (currPos < line.length () && !isspace (line [currPos]))
            currPos++;
          part = line.left (currPos);
        } while (textWidth (&part) + 2 < viewWidth ());
        currPos = lastPos;
      }
      insertLine (line.left (currPos));
      // delete the following spaces
      while (currPos < line.length () && isspace (line [currPos])) currPos++;
      line.remove (0, currPos);
    }
    insertLine (line);
    setUpdatesEnabled (TRUE);
    repaint ();
  }
  setCursorPosition (numLines () - 1, lineLength (numLines () - 1));
}
