// options.cpp

#include <qdialog.h>
#include <qchkbox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qcombo.h>
#include <qdir.h>
#include <qfileinf.h>
#include <kconfig.h>
#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>
#include "global.h"
#include "options.h"

#include <kapp.h>
#include <keasybutton.h>
#include <kmsgbox.h>
#include <kfontdialog.h>
#include <kfiledialog.h>
#include <kfileinfo.h>

Options::Options (QObject *parent, const char *name)
    : QObject (parent, name)
{
  audio = new KAudio ();
  KConfig *conf = kapp->getConfig ();
  conf->setGroup ("General");
  mesgFont = conf->readFontEntry ("MessageFont", 0);
  talkFont = conf->readFontEntry ("TalkFont", 0);
  translate = (TranslateType) conf->readNumEntry ("translateUmlaute", 1);
  singleW = conf->readBoolEntry ("singleWindow", TRUE);
  beepOC = conf->readBoolEntry ("beepOnConnect", TRUE);
  closeOD = conf->readBoolEntry ("closeOnDisconnect", FALSE);
  exitONW = conf->readBoolEntry ("exitOnNoWindow", FALSE);

  struct passwd *pw = getpwuid (getuid ());
  endpwent ();
  if (pw)
    unixName = QString (pw->pw_gecos);
  else
    unixName = localLoginName;

  sendFN = conf->readBoolEntry ("sendFullName", TRUE);
  fn = conf->readEntry ("FullName");
  if (fn.isEmpty ())
    fn = unixName;

  announceConf = new KSimpleConfig (kapp->localconfigdir () + 
                                    "/ktalkannouncerc");
  announceConf->setGroup ("ktalkannounce");
  playS = announceConf->readBoolEntry ("Sound", FALSE);
  sf = announceConf->readEntry ("SoundFile", "ktalkd.wav");
  if (!sf.isEmpty () && sf [0] != '/')
    sf = KApplication::kde_sounddir () + "/" + sf;

  // create the options dialog
  d = new QDialog ();
  createDialog ();
}

Options::~Options ()
{
  delete d;
  delete announceConf;
  delete audio;
}

void Options::playAnnounceSound () {
  if (!playS || !playWAV (sf))
    kapp->beep ();
}

QString Options::getFileSaveDir () {
  KConfig *conf = kapp->getConfig ();
  conf->setGroup ("FileTransfer");
  return conf->readEntry ("SaveDir", QDir::homeDirPath ());
}

void Options::setFileSaveDir (QString dir) {
  KConfig *conf = kapp->getConfig ();
  conf->setGroup ("FileTransfer");
  conf->writeEntry ("SaveDir", dir);
}

QString Options::getFileOpenDir () {
  KConfig *conf = kapp->getConfig ();
  conf->setGroup ("FileTransfer");
  return conf->readEntry ("OpenDir", QDir::homeDirPath ());
}

void Options::setFileOpenDir (QString dir) {
  KConfig *conf = kapp->getConfig ();
  conf->setGroup ("FileTransfer");
  conf->writeEntry ("OpenDir", dir);
}

void Options::writeOptions () {
  KConfig *conf = kapp->getConfig ();
  conf->setGroup ("General");
  conf->writeEntry ("MessageFont", mesgFont);
  conf->writeEntry ("TalkFont", talkFont);
  conf->writeEntry ("translateUmlaute", translate);
  conf->writeEntry ("singleWindow", singleW);
  conf->writeEntry ("beepOnConnect", beepOC);
  conf->writeEntry ("closeOnDisconnect", closeOD);
  conf->writeEntry ("exitOnNoWindow", exitONW);
  conf->writeEntry ("sendFullName", sendFN);
  conf->writeEntry ("FullName", fn);
  conf->sync ();

  announceConf->setGroup ("ktalkannounce");
  announceConf->writeEntry ("Sound", playS);
  announceConf->writeEntry ("SoundFile", sf);
  announceConf->sync ();
}

void Options::selectMesgFont () {
  KFontDialog fd;
  fd.setCaption (i18n ("Select Message Font"));
  if (fd.getFont (mesgFont)) {
    emit mesgFontChanged (mesgFont);
    writeOptions ();
  }
}

void Options::selectTalkFont () {
  KFontDialog fd;
  fd.setCaption (i18n ("Select Talk Font"));
  if (fd.getFont (talkFont)) {
    emit talkFontChanged (talkFont);
    writeOptions ();
  }
}

void Options::doDialog () {
  // fill in the values into the fields
  fnCheck->setChecked (sendFN);
  fnLine->setText (fn);
  transCombo->setCurrentItem (translate);
  beepRadio->setChecked (!playS);
  soundRadio->setChecked (playS);
  soundLine->setText (sf);
  soundToggled ();
  singleWCheck->setChecked (singleW);
  beepOCCheck->setChecked (beepOC);
  closeODCheck->setChecked (closeOD);
  exitONWCheck->setChecked (exitONW);
  d->show ();
  d->raise ();
}

void Options::createDialog () {
  // the toplevel dialog widget
  d->setCaption (i18n ("ktalk Options"));

  // the widgets:
  // full name transmit
  fnCheck = new QCheckBox (i18n ("Transmit &Full Name"), d);
  fnCheck->setMinimumSize (fnCheck->sizeHint ());
  connect (fnCheck, SIGNAL (toggled (bool)), SLOT (sendFNToggled (bool)));
  fnLine = new QLineEdit (d);
  fnLine->setMinimumSize (fnLine->sizeHint ());
  fnReset = new QPushButton (i18n ("Reset"), d);
  fnReset->setMinimumSize (fnReset->sizeHint ());
  connect (fnReset, SIGNAL (clicked ()), SLOT (resetFN ()));
  sendFNToggled (sendFN);

  // translate special characters
  QLabel *transLabel = new QLabel (i18n ("&Translate Special Characters:"), d);
  transLabel->setMinimumSize (transLabel->sizeHint ());
  transCombo = new QComboBox (d);
  transCombo->setMinimumSize (transCombo->sizeHint ());
  transCombo->insertItem (i18n ("no translation"));
  transCombo->insertItem (i18n ("western europe"));
  transCombo->insertItem (i18n ("eastern europe"));
  transLabel->setBuddy (transCombo);

  // announcement sound
  QButtonGroup *soundGroup = 
       new QButtonGroup (i18n ("Sound on incoming announcements"), d);
  beepRadio = new QRadioButton (i18n ("&Beep"), soundGroup);
  beepRadio->setMinimumSize (beepRadio->sizeHint ());
  connect (beepRadio, SIGNAL (clicked ()), SLOT (soundToggled ()));
  soundRadio = new QRadioButton (i18n ("&Play File:"), soundGroup);
  soundRadio->setMinimumSize (soundRadio->sizeHint ());
  connect (soundRadio, SIGNAL (clicked ()), SLOT (soundToggled ()));
  soundLine = new QLineEdit (soundGroup);
  soundLine->setMinimumSize (soundLine->sizeHint ());
  browseButton = new QPushButton (i18n ("Browse"), soundGroup);
  browseButton->setMinimumSize (browseButton->sizeHint ());
  connect (browseButton, SIGNAL (clicked ()), SLOT (browseSound ()));
  testSoundButton = new QPushButton (i18n ("Test"), soundGroup);
  testSoundButton->setMinimumSize (testSoundButton->sizeHint ());
  connect (testSoundButton, SIGNAL (clicked ()), SLOT (testSound ()));

  // boolean options
  singleWCheck = new QCheckBox (i18n ("Only &One Window"), d);
  singleWCheck->setMinimumSize (singleWCheck->sizeHint ());
  beepOCCheck = new QCheckBox (i18n ("Beep on &New Connection"), d);
  beepOCCheck->setMinimumSize (beepOCCheck->sizeHint ());
  closeODCheck = new QCheckBox (i18n ("&Close on Disconnect"), d);
  closeODCheck->setMinimumSize (closeODCheck->sizeHint ());
  exitONWCheck = new QCheckBox (i18n ("&Exit on no Window"), d);
  exitONWCheck->setMinimumSize (exitONWCheck->sizeHint ());
  exitONWCheck->setChecked (exitONW);

  KEasyButton *ok = new KEasyButton (i18n ("OK"), d);
  ok->setDefault (TRUE);
  connect (ok, SIGNAL (clicked ()), SLOT (okClicked ()));
  KEasyButton *cancel = new KEasyButton (i18n ("Cancel"), d);
  connect (cancel, SIGNAL (clicked ()), SLOT (cancelClicked ()));

  // the layout
  QVBoxLayout *soundLayout = new QVBoxLayout (soundGroup, 10, 0);
  QHBoxLayout *browseLayout = new QHBoxLayout ();
  soundLayout->addSpacing (10);
  soundLayout->addWidget (beepRadio);
  soundLayout->addWidget (soundRadio);
  soundLayout->addLayout (browseLayout);
  browseLayout->addSpacing (30);
  browseLayout->addWidget (soundLine, 10);
  browseLayout->addSpacing (5);
  browseLayout->addWidget (browseButton);
  browseLayout->addSpacing (5);
  browseLayout->addWidget (testSoundButton);
  soundLayout->activate ();

  QVBoxLayout *topLayout = new QVBoxLayout (d, 10, 0);
  QHBoxLayout *fnLayout = new QHBoxLayout ();
  QHBoxLayout *buttonLayout = new QHBoxLayout ();

  topLayout->addWidget (fnCheck);
  topLayout->addLayout (fnLayout);
  topLayout->addStretch (10);
  topLayout->addWidget (transLabel);
  topLayout->addWidget (transCombo);
  topLayout->addStretch (10);
  topLayout->addWidget (soundGroup);
  topLayout->addStretch (10);
  topLayout->addWidget (singleWCheck);
  topLayout->addStretch (5);
  topLayout->addWidget (beepOCCheck);
  topLayout->addStretch (5);
  topLayout->addWidget (closeODCheck);
  topLayout->addStretch (5);
  topLayout->addWidget (exitONWCheck);
  topLayout->addStretch (10);
  topLayout->addLayout (buttonLayout);

  fnLayout->addSpacing (30);
  fnLayout->addWidget (fnLine, 10);
  fnLayout->addSpacing (5);
  fnLayout->addWidget (fnReset);

  buttonLayout->addStretch (10);
  buttonLayout->addWidget (ok);
  buttonLayout->addStretch (10);
  buttonLayout->addWidget (cancel);
  buttonLayout->addStretch (10);

  topLayout->activate ();
  d->resize (420, 420);
}

void Options::okClicked () {
  bool resendFullName = (fnCheck->isChecked () && !sendFN) ||
                        fnLine->text () != fn; 
  sendFN = fnCheck->isChecked ();
  fn = fnLine->text ();
  if (resendFullName)
    emit fullNameChanged ();
  beepOC = beepOCCheck->isChecked ();
  if (singleW != singleWCheck->isChecked ()) {
    singleW = singleWCheck->isChecked ();
    emit singleWindowChanged ();
  }
  closeOD = closeODCheck->isChecked ();
  exitONW = exitONWCheck->isChecked ();
  translate = (TranslateType) transCombo->currentItem ();
  playS = soundRadio->isChecked ();
  sf = soundLine->text ();
  writeOptions ();
  d->hide ();
}

void Options::cancelClicked () {
  d->hide ();
}

void Options::sendFNToggled (bool newValue) {
  fnLine->setEnabled (newValue);
  fnReset->setEnabled (newValue);
}

void Options::soundToggled () {
  browseButton->setEnabled (soundRadio->isChecked ());
  testSoundButton->setEnabled (soundRadio->isChecked ());
  soundLine->setEnabled (soundRadio->isChecked ());
}

void Options::browseSound () {
  QString startDir;
  if (QString (soundLine->text ()).isEmpty ())
    startDir = QDir::currentDirPath ();
  else
    startDir = QFileInfo (soundLine->text ()).dirPath ();

  QString newName = KFileDialog::getOpenFileName (startDir, 
                                      "*.wav|Sound Files (*.wav)\n*|Any File");
  if (!newName.isEmpty ())
    soundLine->setText (newName);
}

void Options::testSound () {
  if (!playWAV (soundLine->text ()))
    KMsgBox::message (0, i18n ("Error"), 
                      i18n ("An error occured while playing sound file!\n"
                            "Maybe the sound server isn't running or\n"
                            "the file is not a valid WAV file!"),
                      KMsgBox::EXCLAMATION);  
}

bool Options::playWAV (QString fileName) {
  if (audio->serverStatus () != 0)
    return FALSE;
  if (!audio->play (fileName))
    return FALSE;
  return TRUE;
}

void Options::resetFN () {
  fnLine->setText (unixName);
}

Options *options;
