// ktalk.cpp

#include <kapp.h>
#include <qpopmenu.h>
#include <qpushbt.h>
#include <qmsgbox.h>
#include <qfontmet.h>
#include <qmlined.h>
#include <qaccel.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <kmsgbox.h>
#include <kdebug.h>
#include <drag.h>

#include "ktalk.h"
#include "addrbook.h"
#include "keasybutton.h"
#include "options.h"
#include "ping.h"
#include "filetransfer.h"

KTalk::KTalk (QStrList &initial, QWidget *, const char *name)
        : KTMainWindow (name) {

  char *aboutText= "KTalk " VERSION "\n\n"
                   "Versions 0.1.0 to 0.2.0 by Robert Cimrman\n"
                   "cimrman3@students.zcu.cz\n"
                   "Versions since 0.2.1 by Burkhard Lehner\n"
                   "b_lehner@informatik.uni-kl.de";

  initial_talks = initial;
  setCaption ("ktalk " VERSION);

  msgWindow = new QWidget (0);  // Top Level Widget for messages
  msgWindow->setCaption (i18n ("ktalk Messages"));
  msgList = new QMultiLineEdit (msgWindow);
  msgList->setFont (options->getMesgFont ());
  msgList->setMinimumHeight (3 * msgList->fontMetrics().height ());
  msgList->setAutoUpdate (TRUE);
  msgList->setReadOnly (TRUE);

  KEasyButton *clearButton = 
    new KEasyButton (i18n ("Clear"), msgWindow, 0, -1,
                     msgList, SLOT (clear ()),
                     i18n ("Clears all messages from the message list"));
  KEasyButton *closeButton = 
    new KEasyButton (i18n ("Close"), msgWindow, 0, -1,
                     this, SLOT (hideMessages ()),
                     i18n ("Hides the message window"));

  QVBoxLayout *vbox = new QVBoxLayout (msgWindow, 10);
  QHBoxLayout *hbox = new QHBoxLayout ();
  vbox->addWidget (msgList, 10);
  vbox->addLayout (hbox);
  hbox->addStretch (10);
  hbox->addWidget (clearButton);
  hbox->addStretch (10);
  hbox->addWidget (closeButton);
  hbox->addStretch (10);
  vbox->activate ();

  popups [P_File] = new QPopupMenu();
  CHECK_PTR (popups [P_File]);
  popups [P_File]->insertItem (i18n ("E&xit"), this, 
                               SLOT (quit()), CTRL + Key_Q);

  popups [P_Talk] = new QPopupMenu();
  CHECK_PTR (popups [P_Talk]);
  popups [P_Talk]->insertItem (i18n ("&Talk to..."), this, 
                               SLOT (talkDlg ()), CTRL + Key_T);
  popups [P_Talk]->insertItem (i18n ("&Addressbook..."), this, 
                               SLOT (addrbook ()), CTRL + Key_A);
  popups [P_Talk]->insertItem (i18n ("Close all connections"), 
                               this, SLOT (closeAll ()));
  popups [P_Talk]->insertSeparator ();
  popups [P_Talk]->insertItem (i18n ("&Ping..."), this, SLOT (doPing ()));
  popups [P_Talk]->insertItem (i18n ("&File Transfer..."), this, 
                               SLOT (doTransferFile ()));

  popups [P_Talk]->insertItem (i18n ("Show message..."), 
                               msgWindow, SLOT (show ())); 

  popups [P_Options] = new QPopupMenu();
  CHECK_PTR (popups [P_Options]);
  popups [P_Options]->setCheckable (TRUE);
  popups [P_Options]->insertItem (i18n ("&Message Font..."), 
                                  options, SLOT (selectMesgFont()));
  popups [P_Options]->insertItem (i18n ("&Talk Font..."), options, 
                                  SLOT (selectTalkFont()));
  popups [P_Options]->insertSeparator ();
  popups [P_Options]->insertItem (i18n ("&Options..."), options,
                                  SLOT (doDialog ()));

  popups [P_Help] = KApplication::getKApplication()->
                      getHelpMenu (TRUE, aboutText);

  menuBar ()->insertItem (i18n ("&File"),       popups [P_File]);
  menuBar ()->insertItem (i18n ("&Connection"), popups [P_Talk]);
  menuBar ()->insertItem (i18n ("&Options"),    popups [P_Options]);
  menuBar ()->insertSeparator();
  menuBar ()->insertItem (i18n ("&Help"),       popups [P_Help]);

  statusBar ()->insertItem (
                   i18n ("Click here to view message window"), 1);
  connect (statusBar (), SIGNAL (pressed (int)), SLOT (showMessages (int)));
  QToolTip::add (statusBar (), i18n ("last status message\n"
                              "Click here to view window with all messages"));

  ab = new AddressBook ();
  td = new TalkDialog ();
  connect (td, SIGNAL (startTalk (const char *)), SLOT (talk (const char *)));
  tw.setAutoDelete (TRUE);

  myMultiWidget = new MultiWidget (this);
  myMainWidget = new MainWidget (myMultiWidget);
  myMultiWidget->setSingleWidget (myMainWidget);
  setView (myMultiWidget);
  updateRects ();
  borderWidth = width () - myMultiWidget->width ();
  borderHeight = height () - myMultiWidget->height ();
  myMainWidget->setTalkFont (options->getTalkFont ());
  connect (myMainWidget, SIGNAL (closeAllReq ()), SLOT (closeAll ()));
  connect (options, SIGNAL (talkFontChanged (const QFont &)), 
           myMainWidget, SLOT (setTalkFont (const QFont &)));

  atLeastOneConnected = FALSE;

  checker = new CheckProtocol (this);
  connect (checker, SIGNAL (timeout ()), SLOT (protocolTimeout ()));
  connect (checker, SIGNAL (found (ProtocolType)), 
                    SLOT (protocolFound (ProtocolType)));
  if (!checker->CheckHost (localhostAddr)) {
    KDEBUG (KDEBUG_ERROR, 3900, 
            "No talk demon found on your host! Using ntalk as default!");
    localProtocol = ntalkProtocol;
  }
  setMinimumSize (myMultiWidget->minimumSize ().width () + borderWidth,
                  myMultiWidget->minimumSize ().height () + borderHeight);
  connect (options, SIGNAL (singleWindowChanged ()),
                    SLOT (toggleSingleWindow ()));

  KDNDDropZone *FileDropZone = new KDNDDropZone (this, DndURL);
  connect (FileDropZone, SIGNAL (dropAction (KDNDDropZone *)),
                         SLOT (fileDropped (KDNDDropZone *)));
}

KTalk::~KTalk () {

  while (tw.count () > 0)
    tw.remove (0U);

  delete ab;
  delete td;
  delete msgWindow;

}

void KTalk::talk (const char *stranger ) {
  if (strlen (stranger) == 0) {
    show ();
    raise ();
    return;
  }
  printMessage (i18n ("Starting connection to ") + QString (stranger));
  ab->insertRecentEntry (QString (stranger));
  QWidget *my_parent = (options->getSingleWindow ()) ? myMultiWidget : 0;
  TalkWidget *newWidget = new TalkWidget (my_parent);
  tw.append (newWidget);
  connect (newWidget, SIGNAL (closeRequest (TalkWidget *)),
                      SLOT (closeTalk (TalkWidget *)));
  connect (newWidget, SIGNAL (newStatusRequest ()),
                      SLOT (connectedStatusChanged ()));
  connect (newWidget, SIGNAL (message (const char *)), 
                      SLOT (printMessage (const char *)));
  connect (options, SIGNAL (talkFontChanged (const QFont &)), 
           newWidget, SLOT (setTalkFont (const QFont &)));
  connect (myMainWidget, SIGNAL (spreadData (QString)), 
           newWidget,    SLOT (sendData (QString)));
  newWidget->setTalkFont (options->getTalkFont ());
  newWidget->show();
  newWidget->startConnection (stranger);
  reorganizeWindows ();
}

void KTalk::printMessage (const char *msg) {
  statusBar ()->changeItem (msg, 1);
  msgList->insertLine ((const char *) msg);
  msgList->setCursorPosition (msgList->numLines () - 1, 0);
}

void KTalk::talkDlg() {

  td->execTalkDialog (ab);
}

void KTalk::addrbook() {

  ab->execAddressBook ();
}

void KTalk::closeAll () {
  while (tw.count () > 0)
    tw.remove (0U);
  reorganizeWindows ();
}

void KTalk::doPing () {
  (void) new PingDialog (tw);
}

void KTalk::doTransferFile () {
  (void) new FileTransferSend (tw);
}

void KTalk::fileDropped (KDNDDropZone *zone) {
  QStrList fileList = zone->getURLList ();
  for (char *name = fileList.first (); name; name = fileList.next ())
    (void) new FileTransferSend (tw, name);
}

void KTalk::closeTalk (TalkWidget *widget) {
  tw.removeRef (widget);
  reorganizeWindows ();
}

void KTalk::connectedStatusChanged () {
  
  bool oneConnected = FALSE;
  QListIterator <TalkWidget> iter (tw);
  while (iter.current () && !oneConnected) {
    if (iter.current ()->getStatus () == TalkWidget::st_connected)
      oneConnected = TRUE;
    ++iter;
  }
  if (oneConnected != atLeastOneConnected) {
    myMainWidget->setInteractiveMode (oneConnected);
    atLeastOneConnected = oneConnected;
  }
}

void KTalk::quit () {

//  closeAll ();
  KApplication::getKApplication()->quit();
}

void KTalk::setMesgFont (const QFont &font) {
  msgList->setFont (font);
}

void KTalk::toggleSingleWindow () {
  TalkWidget *help;
  if (options->getSingleWindow ()) {
    for (help = tw.first (); help; help = tw.next ())
      if (help->parent () != myMultiWidget)
        help->recreate (myMultiWidget, 0, help->mapToGlobal (QPoint (0, 0)), 
                        TRUE);
  } else {
    for (help = tw.first (); help; help = tw.next ())
      if (!help->isTopLevel ())
        help->recreate (0, 0, help->mapToGlobal (QPoint (0, 0)), TRUE);
  }
  reorganizeWindows ();
}

void KTalk::protocolFound (ProtocolType protocol) {
  localProtocol = protocol;
  while (!initial_talks.isEmpty ()) {
    talk (initial_talks.getFirst ());
    initial_talks.removeFirst ();
  }  
}

void KTalk::protocolTimeout () {
  KMsgBox::message (0, 0, i18n ("No talk daemon was found on your host!\n"
                                "Probably talking will not be possible!\n"
                                "Using ntalk protocol as a default!"));
  localProtocol = ntalkProtocol;
  while (!initial_talks.isEmpty ()) {
    talk (initial_talks.getFirst ());
    initial_talks.removeFirst ();
  }  
}

void KTalk::reorganizeWindows () {
  if (options->getSingleWindow ())
    myMultiWidget->setWidgets (myMainWidget, tw);
  else
    myMultiWidget->setSingleWidget (myMainWidget);
  setMinimumSize (myMultiWidget->minimumSize ().width () + borderWidth,
                  myMultiWidget->minimumSize ().height () + borderHeight);
}

//----------------------------------------------------------------------------

MultiWidget::MultiWidget (QWidget *parent, const char *name) :
   QWidget (parent, name) 
{
  actualWidgets.setAutoDelete (FALSE);
}

void MultiWidget::setWidgets (QWidget *firstWidget, 
                              QList <TalkWidget> &otherWidgets)
{
  QWidget *dummy;

  actualWidgets.clear ();
  actualWidgets.append (firstWidget);
  KASSERT ((firstWidget->parent () == this), KDEBUG_WARN, 3900,
           "MultiWidget::setWidgets (): "
           "using firstWidget with different parent!"); 
  for (dummy = otherWidgets.first (); dummy; dummy = otherWidgets.next ()) {
    actualWidgets.append (dummy);
    KASSERT ((dummy->parent () == this), KDEBUG_WARN, 3900,
             "MultiWidget::setWidgets (): "
             "using widget with different parent!");
  }
  int minW = 0, minH = 0;
  for (dummy = actualWidgets.first (); dummy; dummy = actualWidgets.next ()) {
    minW = QMAX (minW, dummy->minimumSize ().width ());
    minH += dummy->minimumSize ().height ();
  }
  setMinimumSize (minW, minH);
  arrangeWidgets ();
}

void MultiWidget::setSingleWidget (QWidget *firstWidget)
{
  QList <TalkWidget> dummy;
  setWidgets (firstWidget, dummy);
}

void MultiWidget::resizeEvent (QResizeEvent *)
{
  arrangeWidgets ();
}

void MultiWidget::arrangeWidgets ()
{
  int w = width ();
  int diffh = (height () - minimumSize ().height ()) / actualWidgets.count ();
  int y = 0;
  QWidget *dummy;
  for (dummy = actualWidgets.first (); dummy; dummy = actualWidgets.next ()) {
    int h = dummy->minimumSize ().height () + diffh;
    QRect position (0, y, w, h);
    dummy->setGeometry (position);
    y += h;
  }
}

//----------------------------------------------------------------------------

MainWidget::MainWidget (QWidget *parent, const char *name)
        : QWidget (parent, name) {

  QLabel *talkLabel = new QLabel (i18n ("I write here:"), this);
  talkLabel->setMinimumHeight (talkLabel->sizeHint ().height ());
  KEasyButton *closeAllButton = 
    new KEasyButton (i18n ("Close All"), this, 0, -1,
                     this, SIGNAL (closeAllReq ()),
                     i18n ("Disconnects all open connections and closes "
                           "the talk windows"));
  closeAllButton->setFocusPolicy (QWidget::NoFocus);
  talkText = new SimpleEdit (this);
  talkText->setMinimumHeight (3 * talkText->fontMetrics().height ());
  talkText->setFocusPolicy (QWidget::StrongFocus);
  talkText->setReadOnly (TRUE);
  talkText->setText (i18n ("no connection yet..."));
  connect (talkText, SIGNAL (keyPressed (QString)), 
           this, SIGNAL (spreadData (QString)));

  QBoxLayout *talkLayout = new QVBoxLayout (this, 5);
  QBoxLayout *buttons = new QHBoxLayout ();

  talkLayout->addLayout (buttons);
  talkLayout->addWidget (talkText, 10);

  buttons->addWidget (talkLabel, 10);
  buttons->addWidget (closeAllButton);

  talkLayout->activate ();
}

MainWidget::~MainWidget() {

}

void MainWidget::setInteractiveMode (bool mode) {
  if (mode) {
    talkText->setReadOnly (FALSE);
    talkText->reset ();
    talkText->setFocus ();
  } else {
    talkText->setReadOnly (TRUE);
    talkText->insertLine (i18n ("... all connections closed"));
    talkText->setCursorPosition (talkText->numLines () - 1, 0);
    if (options->getExitOnNoWindow ())
      kapp->quit ();
  }
}

void MainWidget::setTalkFont (const QFont &font) {
  talkText->setFont (font);
}
