#ifndef _KEASYBUTTON_H
#define _KEASYBUTTON_H

#include <qpushbt.h>

/**
 * This class implements a simple text pushbutton that automatically sets its
 * minimum size, so that the text fits into it. That's especially nice if you
 * use QBoxLayout, or if you use i18n for the button text, so that the text
 * size differs dependend on the language.
 * The class provides a default minimum width that every button has, so that
 * all the buttons have the same width as long as the text fits into it. This
 * default width can be changed, and can be overwritten for every button.
 *
 * Furthermore the constructor of this class provides arguments to connect
 * the clicked() methode to a slot, and to add a tooltip to the button. So
 * the construction of the button is in most times a one-liner. You won't need
 * a variable that you assign the button to, if you don't need to do anything
 * else with that button.
 *
 * Example:
 *
 * What you had to do before KEasyButton existed:
 * <pre> 
 * QPushButton *clear = new QPushButton (i18n ("Clear"), this);
 * clear->setMinimumHeight (clear->sizeHint().height());
 * clear->setMinimumWidth (QMAX (clear->sizeHint().width(), 80));
 * connect (clear, SIGNAL (clicked ()), multilineedit, SLOT (clear ())); 
 * QToolTip::add (clear, "Clears the text field");
 * layout->addWidget (clear);
 * </pre>
 *
 * What you can do now:
 * <pre>
 * layout->addWidget (new KEasyButton (i18n ("clear"), this, 0, 80, 
 *                                     multilineedit, SLOT (clear ()),
 *                                     "Clears the text field"));
 * </pre>
 *
 * @short Simple class for pushbuttons with minimum size set
 * @author Burkhard Lehner <b_lehner@informatik.uni-kl.de>
 * @version 1.0 (I think it is final, since it is such simple :-) )
 */

class KEasyButton : public QPushButton {

public:
/**
 * Constructor of the pushbutton. It creates a QPushButton with text, sets
 * its minimum size so that it fits the text and the button is at least 
 * "minimumWidth" pixel wide, connects the "clicked()" signal to a slot
 * and adds a tooltip to the button.
 *
 * @param text The text that should be displayed on the button
 * @param parent The parent widget of the button (or 0 if it should be a top
 *               level widget)
 * @param name The name of the object (or 0 for no name)
 * @param minimumWidth The created button has a minimum width of at least this
 *                     value (in pixel). Use a value of 0 if you want the
 *                     button to be as small as possible to fit the text. Use
 *                     a value of -1 to use the default minimumWidth, that is
 *                     initially set to a value of 60 pixel and can be changed
 *                     by calling the static methode setDefaultMinimumWidth.
 * @param receiver The object that shall receive a signal when the button is 
 *                 clicked. Use a value of 0 if you don't want to connect to
 *                 an object.
 * @param member The slot that shall be connected to the signal clicked(). Use
 *               a value of 0 if you don't want to connect to an object.
 *               (If one of the parameters receiver and member is 0, the
 *               signal clicked() is not connected.)
 * @param tooltip This text is added as a tooltip to the button. Use a value
 *                of 0 if you don't want to add a tooltip.
 */
  KEasyButton (const char *text,
               QWidget *parent = 0, const char *name = 0,
               int minimumWidth = -1,
               QObject *receiver = 0,
               const char *member = 0,
               const char *tooltip = 0);
/**
 * The destructor
 */
  ~KEasyButton () {}

/**
 * Use this static methode to change the default minimum width of the buttons.
 * The initial value is 60 pixel. Every button that is created AFTER a call
 * to this methode with parameter minimumWidth = -1 has set it's minimum
 * width to a value that is bigger or equal to this value, so that the
 * text to display fits into the button.
 *
 * @param width The new value for the default minimum width
 */
  static void setDefaultMinimumWidth (int width);

private:
  static int defaultMinimumWidth;

};

#endif   //_KEASYBUTTON_H
