// filetransfer.cpp

#include <stdio.h>
#include <stdlib.h>
#include <qdir.h>
#include <qpushbt.h>
#include <qlayout.h>
#include <kfiledialog.h>
#include <kapp.h>
#include <kmsgbox.h>
#include "daemon_socket.h"
#include "options.h"
#include "filetransfer.h"

FileTransferSend::FileTransferSend (QList <TalkWidget> &list, const char *name) 
   : QObject ()
{
  if (name != 0 && QString (name).left (5).upper () != "FILE:") {
    QString errmsg;
    errmsg.sprintf (i18n ("\"%s\" is not a local file!"), name);
    KMsgBox::message (0, i18n ("Error"), errmsg, KMsgBox::EXCLAMATION);
    delete this;
    return;
  }

  servSock = 0;
  sendSock = 0;
  QStrList destNames;
  QList <TalkWidget> destList;
  destList.setAutoDelete (FALSE);
  for (TalkWidget *help = list.first (); help; help = list.next ())
    if (help->isNewerThan (0, 2, 7)) {
      destNames.append (help->getName ());
      destList.append (help);
    }

  if (destList.count () == 0) {
    KMsgBox::message (0, i18n ("Error"),
                         i18n ("There is no connection that can "
                               "accept file transfers!"),
                      KMsgBox::EXCLAMATION);
    delete this;
    return;
  }

  // create the widgets for the dialog
  QDialog dialog (0, 0, TRUE);
  QLabel *sendLabel = new QLabel (i18n ("Send File:"), &dialog);
  sendLabel->setMinimumSize (sendLabel->sizeHint ());
  fileName = new QLineEdit (&dialog);
  fileName->setMinimumSize (fileName->sizeHint ());
  if (name == 0)
    fileName->setText (options->getFileOpenDir () + "/");
  else
    fileName->setText (name + 5);
  QPushButton *browseButton = new QPushButton (i18n ("Browse"), &dialog);
  browseButton->setMinimumSize (browseButton->sizeHint ());
  connect (browseButton, SIGNAL (clicked ()), SLOT (browse ()));
  QLabel *toLabel = new QLabel (i18n ("To:"), &dialog);
  toLabel->setMinimumSize (toLabel->sizeHint ());
  destinationCombo = new QComboBox (&dialog);
  destinationCombo->insertStrList (&destNames);
  destinationCombo->setMinimumSize (destinationCombo->sizeHint ());
  KEasyButton *sendButton = new KEasyButton (i18n ("Send"), &dialog, 0, -1,
                                             &dialog, SLOT (accept ()));
  KEasyButton *cancelButton = new KEasyButton (i18n ("Cancel"), &dialog, 0, -1,
                                               &dialog, SLOT (reject ()));

  // layout
  QVBoxLayout *topLayout = new QVBoxLayout (&dialog, 10);
  QHBoxLayout *filenameLayout = new QHBoxLayout ();
  QHBoxLayout *destLayout = new QHBoxLayout ();
  QHBoxLayout *buttonLayout = new QHBoxLayout ();

  topLayout->addWidget (sendLabel);
  topLayout->addLayout (filenameLayout);
  topLayout->addStretch (10);
  topLayout->addWidget (toLabel);
  topLayout->addLayout (destLayout);
  topLayout->addStretch (10);
  topLayout->addLayout (buttonLayout);

  filenameLayout->addSpacing (20);
  filenameLayout->addWidget (fileName, 10);
  filenameLayout->addSpacing (5);
  filenameLayout->addWidget (browseButton);

  destLayout->addSpacing (20);
  destLayout->addWidget (destinationCombo, 10);

  buttonLayout->addStretch (10);
  buttonLayout->addWidget (sendButton);
  buttonLayout->addStretch (10);
  buttonLayout->addWidget (cancelButton);
  buttonLayout->addStretch (10);

  topLayout->activate ();
  dialog.resize (400, 220);
  fileName->setFocus ();

  if (dialog.exec () == QDialog::Accepted) {
    debug ("Transfering file: %s", fileName->text());
    file.setName (fileName->text ());
    options->setFileOpenDir (QFileInfo (fileName->text ()).dirPath ());
    doSend (destList.at (destinationCombo->currentItem ()));
  } else
    delete this;
}

FileTransferSend::~FileTransferSend () {
  if (servSock) delete servSock;
  if (sendSock) delete sendSock;
}

void FileTransferSend::browse () {
  QString startDir;
  if (QString (fileName->text ()).isEmpty ())
    startDir = QDir::homeDirPath ();
  else
    startDir = QFileInfo (fileName->text ()).dirPath ();

  QString newName = KFileDialog::getOpenFileName (startDir, "*|Any File");
  if (!newName.isEmpty ())
    fileName->setText (newName);
}

void FileTransferSend::abort () {
  delete progressW;
  delete this;
}

void FileTransferSend::connected (KSocket *sock) {
  delete servSock;
  servSock = 0;
  sendSock = sock;
  connect (sendSock, SIGNAL (writeEvent (KSocket *)),
                     SLOT (readyToWrite (KSocket *)));
  sendSock->enableWrite (TRUE);
  sendSize = 0;
  writeBufferLength = 0;
}

void FileTransferSend::readyToWrite (KSocket *sock) {
  unsigned int fetchSize = 1024 - writeBufferLength;
  if (fetchSize > fileSize - sendSize)
    fetchSize = fileSize - sendSize;
  unsigned int readSize = file.readBlock (writeBuffer + writeBufferLength, 
                                          fetchSize);
  if (readSize != fetchSize) {
    abort ();
    KMsgBox::message (0, i18n ("Error"), i18n ("File Size changed!"),
                      KMsgBox::EXCLAMATION);
    return;
  }
  writeBufferLength += readSize;
  int result = send (sock->socket (), writeBuffer, writeBufferLength, 0);
  if (result <= 0) {
    abort ();
    KMsgBox::message (0, i18n ("Error"), i18n ("Connection broken!"),
                      KMsgBox::EXCLAMATION);
    return;
  }
  memmove (writeBuffer, writeBuffer + result, writeBufferLength - result);
  writeBufferLength -= result;
  sendSize += result;
  progressW->updateValue (sendSize);
  if (sendSize == fileSize)
    abort ();
}

void FileTransferSend::doSend (TalkWidget *destination) {
  if (!QFileInfo (file).isFile ()) {
    QString errmsg;
    errmsg.sprintf (i18n ("\"%s\" is not a File!"), file.name ());
    kapp->beep ();
    KMsgBox::message (0, i18n ("Error"), errmsg.data (),
                      KMsgBox::EXCLAMATION);
    delete this;
    return;
  }
  if (!file.open (IO_ReadOnly)) {
    QString errmsg;
    errmsg.sprintf (i18n ("Unable to open file \"%s\" !"), file.name ());
    kapp->beep ();
    KMsgBox::message (0, i18n ("Error"), errmsg.data (),
                      KMsgBox::EXCLAMATION);
    delete this;
    return;
  }
  servSock = new KServerSocket (0);
  if (servSock->socket () == -1) {
    KDEBUG (KDEBUG_WARN, 3900, "Cannot open Server Socket!");
    delete servSock;
    delete this;
    return;
  }
  connect (servSock, SIGNAL (accepted (KSocket *)), 
                     SLOT (connected (KSocket *)));
  struct in_addr addr = DaemonSocket::getReplyAddr (destination->
                                                    getDestIPAddr ());
  unsigned char *help = (unsigned char *) &addr;
  fileSize = file.size ();
  QString param;
  param.sprintf ("%d.%d.%d.%d;%d;%d;%s", 
                 help [0], help [1], help [2], help [3],
                 servSock->getPort (), 
                 fileSize,
                 QFileInfo (file).fileName ().data ());
  debug ("Sending command with parameter %s", param.data ());
  destination->sendCommand ("FileTransfer", param);

  progressW = new ProgressWidget (ProgressWidget::Sending, file.name (),
                                  destination->getName (), fileSize);
  connect (progressW, SIGNAL (abort ()), SLOT (abort ()));
}

// *************************************************************************

FileTransferReceive::FileTransferReceive (QString param, QString sender)
   : QObject ()
{
  receiveSock = 0;
  senderName = sender;
  host = cutUntilSemicolon (param);
  port = atoi (cutUntilSemicolon (param).data ());
  fileSize = atoi (cutUntilSemicolon (param).data ());
  receiveSize = 0;

  // create the widgets for the dialog
  dialog = new QWidget ();
  dialog->setCaption (i18n ("File Transfer Request"));
  QLabel *senderLabel = new QLabel (sender.data (), dialog);
  senderLabel->setMinimumSize (senderLabel->sizeHint ());
  QString info;
  info.sprintf (i18n ("offers a file (%d Byte)"), fileSize);
  QLabel *infoLabel = new QLabel (info, dialog);
  infoLabel->setMinimumSize (infoLabel->sizeHint ());

  fileName = new QLineEdit (dialog);
  fileName->setMinimumSize (fileName->sizeHint ());
  fileName->setText (options->getFileSaveDir () + "/" + param);
  QPushButton *browseButton = new QPushButton (i18n ("Browse"), dialog);
  browseButton->setMinimumSize (browseButton->sizeHint ());
  connect (browseButton, SIGNAL (clicked ()), SLOT (browse ()));
  KEasyButton *acceptButton = new KEasyButton (i18n ("Accept"), dialog, 0, -1,
                                               this, SLOT (accepting ()));
  KEasyButton *rejectButton = new KEasyButton (i18n ("Reject"), dialog, 0, -1,
                                               this, SLOT (rejecting ()));

  // layout
  QVBoxLayout *topLayout = new QVBoxLayout (dialog, 10);
  QHBoxLayout *filenameLayout = new QHBoxLayout ();
  QHBoxLayout *buttonLayout = new QHBoxLayout ();

  topLayout->addWidget (senderLabel);
  topLayout->addWidget (infoLabel);
  topLayout->addLayout (filenameLayout);
  topLayout->addStretch (10);
  topLayout->addLayout (buttonLayout);

  filenameLayout->addSpacing (20);
  filenameLayout->addWidget (fileName, 10);
  filenameLayout->addSpacing (5);
  filenameLayout->addWidget (browseButton);

  buttonLayout->addStretch (10);
  buttonLayout->addWidget (acceptButton);
  buttonLayout->addStretch (10);
  buttonLayout->addWidget (rejectButton);
  buttonLayout->addStretch (10);

  topLayout->activate ();
  dialog->resize (dialog->minimumSize ());
  dialog->show ();
}

FileTransferReceive::~FileTransferReceive () {
  if (receiveSock) delete receiveSock;
}

void FileTransferReceive::browse () {
  QString startDir;
  if (QString (fileName->text ()).isEmpty ())
    startDir = QDir::homeDirPath ();
  else
    startDir = QFileInfo (fileName->text ()).dirPath ();

  QString newName = KFileDialog::getSaveFileName (startDir, "*|Any File");
  if (!newName.isEmpty ())
    fileName->setText (newName);
}

void FileTransferReceive::accepting () {
  file.setName (fileName->text ());
  if (file.exists ()) {
    QString errmsg;
    errmsg.sprintf (i18n ("File \"%s\" exists!"), fileName->text ());
    int result = KMsgBox::yesNoCancel (0, i18n ("File Exists"), errmsg.data (),
                                       KMsgBox::QUESTION, i18n ("Overwrite"), 
                                       i18n ("New Name"), i18n ("Cancel"));
    if (result == 2) return;  // new name => keep dialog open
    if (result != 1) {        // cancel => delete everyting
      delete dialog;
      delete this;
      return;
    }
  }
  options->setFileSaveDir (QFileInfo (fileName->text ()).dirPath ());
  delete dialog;
  if (!file.open (IO_WriteOnly)) {
    QString errmsg;
    errmsg.sprintf (i18n ("Can't open \"%s\" for writing!"), 
                    fileName->text ());
    KMsgBox::message (0, i18n ("Error"), errmsg.data (), KMsgBox::EXCLAMATION);
    delete this;
    return;
  }
  receiveSock = new KSocket (host, port);
  if (receiveSock->socket () < 0) {
    KMsgBox::message (0, i18n ("Error"), i18n ("Connection broken!"), 
                      KMsgBox::EXCLAMATION);
    delete this;
    return;
  }
  connect (receiveSock, SIGNAL (readEvent (KSocket *)),
                        SLOT (readyToRead (KSocket *)));
  connect (receiveSock, SIGNAL (closeEvent (KSocket *)),
                        SLOT (readyToRead (KSocket *)));
  receiveSock->enableRead (TRUE);

  progressW = new ProgressWidget (ProgressWidget::Receiving, file.name (), 
                                  senderName, fileSize);
  connect (progressW, SIGNAL (abort ()), SLOT (abort ()));
}

void FileTransferReceive::rejecting () {
  delete dialog;
  delete this;
}

void FileTransferReceive::abort () {
  delete progressW;
  delete this;
}

void FileTransferReceive::readyToRead (KSocket *sock) {
  char readBuffer [1024];
  int result = recv (sock->socket (), readBuffer, 1024, 0);
  if (result <= 0 || receiveSize + result > fileSize) {
    abort ();
    KMsgBox::message (0, i18n ("Error"), i18n ("Connection broken!"),
                      KMsgBox::EXCLAMATION);
    return;
  }
  int writeSize = file.writeBlock (readBuffer, result);
  if (writeSize != result) {
    abort ();
    KMsgBox::message (0, i18n ("Error"), i18n ("Error writing to file!"),
                      KMsgBox::EXCLAMATION);
    return;
  }
  receiveSize += result;
  progressW->updateValue (receiveSize);
  if (receiveSize == fileSize) {
    file.flush ();
    abort ();
  }
}

QString FileTransferReceive::cutUntilSemicolon (QString &rest) {
  int pos = rest.find (';');
  if (pos == -1) {
    QString result = rest;
    rest = QString ("");
    return result;
  } else {
    QString result = rest.left (pos);
    rest.remove (0, pos + 1);
    return result;
  }
}

// *******************************************************************
ProgressWidget::ProgressWidget (Direction direction, QString file, 
                                QString partner, int size) :
   QWidget (0, 0)
{
  fileSize = size;
  setCaption (i18n ("File Transfer"));
  QLabel *file1Label = new QLabel (i18n ("File:"), this);
  file1Label->setMinimumSize (file1Label->sizeHint ());
  QLabel *file2Label = new QLabel (file, this);
  file2Label->setMinimumSize (file2Label->sizeHint ());
  QLabel *from1Label = new QLabel ((direction == Receiving) ? 
                                    i18n ("From:") : i18n ("To:"), this);
  from1Label->setMinimumSize (from1Label->sizeHint ());
  QLabel *from2Label = new QLabel (partner, this);
  from2Label->setMinimumSize (from2Label->sizeHint ());

  QString statusMessage;
  statusMessage.sprintf (i18n ("0 of %d Bytes received"), fileSize);
  statusLabel = new QLabel (statusMessage.data (), this);
  statusLabel->setAlignment (AlignCenter);
  statusLabel->setMinimumSize (statusLabel->sizeHint ());
  progressBar = new KProgress (0, fileSize, 0, KProgress::Horizontal, this);
  progressBar->setMinimumSize (progressBar->sizeHint ());
  KEasyButton *abortButton = new KEasyButton (i18n ("Abort"), this, 0, -1,
                                              this, SIGNAL (abort ()));
  abortButton->setFixedSize (abortButton->minimumSize ());

  // the layout stuff
  QVBoxLayout *topLayout = new QVBoxLayout (this, 10);
  QGridLayout *upperLayout = new QGridLayout (2, 2);

  topLayout->addLayout (upperLayout);
  topLayout->addWidget (statusLabel);
  topLayout->addWidget (progressBar);
  topLayout->addStretch (10);
  topLayout->addWidget (abortButton);

  upperLayout->addWidget (file1Label, 0, 0);
  upperLayout->addWidget (file2Label, 0, 1);
  upperLayout->addWidget (from1Label, 1, 0);
  upperLayout->addWidget (from2Label, 1, 1);

  topLayout->activate ();
  resize (minimumSize ());
  updateValue (0);
  show ();
}

void ProgressWidget::updateValue (int value) {
  QString newStatusString;
  newStatusString.sprintf (i18n ("%d of %d Bytes received"),
                           value, fileSize);
  statusLabel->setText (newStatusString.data ());
  progressBar->setValue (value);
}

void ProgressWidget::closeEvent (QCloseEvent *event) {
  emit abort ();
  event->accept ();
}
