// daemon_socket.cpp

#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include "daemon_socket.h"
#include "config.h"
#include <kdebug.h>

DaemonSocket::DaemonSocket (struct in_addr destination, ProtocolType protocol,
                            QObject *parent, const char *name) :
    QObject (parent, name)
{
  lastMessage.type = DELETE;  // just to set an initial value
  proto = protocol;
  notify = 0;
  ok = FALSE;

  // Init daemonAddr
  daemonAddr.sin_family = AF_INET;
  daemonAddr.sin_addr = destination;
  switch (proto) {
    case talkProtocol:  daemonAddr.sin_port = talkDaemonPort; break;
    case ntalkProtocol: daemonAddr.sin_port = ntalkDaemonPort; break;
    default:            daemonAddr.sin_port = ntalkDaemonPort;
  }

  // Init clientAddr
  clientAddr.sin_family = AF_INET;
  clientAddr.sin_addr.s_addr = htonl (INADDR_ANY);
  clientAddr.sin_port = htons (0);

  ksize_t length = sizeof (clientAddr);
  if ((sock = socket (AF_INET, SOCK_DGRAM, 0)) < 0 ||
      bind (sock, (struct sockaddr *) &clientAddr, sizeof (clientAddr)) != 0 ||
      getsockname (sock, (struct sockaddr *) &clientAddr, &length) == -1) {
    if (sock >= 0) ::close (sock);
    return;
  }
  clientAddr.sin_addr = getReplyAddr (destination);

  ok = TRUE;
  notify = new QSocketNotifier (sock, QSocketNotifier::Read, this);
  connect (notify, SIGNAL (activated (int)), SLOT (newPacketWaiting (int)));
}

DaemonSocket::~DaemonSocket () {
  if (ok) close (sock);
}

bool DaemonSocket::send (NEW_CTL_MSG msg) {
  int i, length;
  if (!ok) return FALSE;

  // fill in the correct reply address into the message
  msg.ctl_addr = clientAddr;
  msg.ctl_addr.sin_family = htons (AF_INET);

  if (proto == talkProtocol) {
    // ktalk internally only works on ntalk messages
    // copy to talk messages if using talkProtocol
    OLD_CTL_MSG old_msg;
    old_msg.type = msg.type;
    strncpy (old_msg.l_name, msg.l_name, OLD_NAME_SIZE - 1);
    old_msg.l_name [OLD_NAME_SIZE - 1] = '\0';
    strncpy (old_msg.r_name, msg.r_name, OLD_NAME_SIZE - 1);
    old_msg.r_name [OLD_NAME_SIZE - 1] = '\0';
    strncpy (old_msg.r_tty, msg.r_tty, TTY_SIZE - 1);
    old_msg.r_tty [TTY_SIZE - 1] = '\0';
    old_msg.id_num = msg.id_num;
    old_msg.pid = msg.pid;
    old_msg.addr = msg.addr;
    old_msg.ctl_addr = msg.ctl_addr;
    for (i = 0; i < 3; i++) {
      length = sendto (sock, (char *) &old_msg, sizeof (old_msg), 0,
                       (struct sockaddr *) &daemonAddr, sizeof (daemonAddr));
      if (length == sizeof (old_msg)) return TRUE;
    }
  } else {
    for (i = 0; i < 3; i++) {
      length = sendto (sock, (char *) &msg, sizeof (msg), 0,
                       (struct sockaddr *) &daemonAddr, sizeof (daemonAddr));
      if (length == sizeof (msg)) return TRUE;
    }
  }
  return FALSE;
}

bool DaemonSocket::sendReliable (NEW_CTL_MSG msg) {
  killTimers ();
  lastMessage = msg;
  if (!send (msg))
    return FALSE;
  retryCounter = 0;
  startTimer (2000);  // expect answer within two seconds
  return TRUE;
}

void DaemonSocket::timerEvent (QTimerEvent *) {
  retryCounter++;
  if (retryCounter >= 10) {    // try only 10 times
    killTimers ();
    emit timeout ();
  } else
    send (lastMessage);
}

void DaemonSocket::newPacketWaiting (int socket) {
  NEW_CTL_RESPONSE response;
  int length, expected;
  if (!ok || socket != sock) return;

  if (proto == talkProtocol) {
    OLD_CTL_RESPONSE old_resp;
    length = recv (sock, (char *) &old_resp, expected = sizeof (old_resp), 0); 
    response.vers = TALK_VERSION;
    response.type = old_resp.type;
    response.answer = old_resp.answer;
    response.id_num = old_resp.id_num;
    response.addr = old_resp.addr;
  } else
    length = recv (sock, (char *) &response, expected = sizeof (response), 0);

  if (length == expected && response.vers == TALK_VERSION) {
    // if response was okay and response was to the last sendReliable () call,
    // then disable timeout timer.
    if (response.type == lastMessage.type)
      killTimers ();
    emit receivedReply (response);
  } else
    KDEBUG (KDEBUG_WARN, 3900, "Received a bad control response from "
                               "talk daemon");
}

struct in_addr DaemonSocket::getReplyAddr (struct in_addr destination) {
// tries to find out the correct IP address that the daemon at host 
// "destination" has to respond to
  in_addr *result = replyAddrList [(long) destination.s_addr];
  unsigned char *help1;
  unsigned char *help2;
  if (result) {
    return *result;
  }
  
  int testsock, i;
  result = new (struct in_addr);
  struct sockaddr_in client, daemon;
  for (i = 0; i < 2; i++) {
    client.sin_family = daemon.sin_family = AF_INET;
    client.sin_addr.s_addr = htonl (INADDR_ANY);
    client.sin_port = htons (0);
    daemon.sin_addr = destination;
    daemon.sin_port = i ? ntalkDaemonPort : talkDaemonPort;

    // Connect to the daemon socket address
    // On some UNIXes (such as Linux) this works and sets the IP address queried
    // by getsockname to the local machine address used to reach the daemon.
    // If it doesn't work (e.g. on SunOS and Solaris), the default machine
    // address is used instead.
    ksize_t length = sizeof (daemon);
    if ((testsock = socket (AF_INET, SOCK_DGRAM, 0)) >= 0 &&
        bind (testsock, (struct sockaddr *) &client, sizeof (client)) == 0 &&
        ::connect (testsock, (struct sockaddr *) &daemon, 
                             sizeof (daemon)) == 0 &&
        getsockname (testsock, (struct sockaddr *) &client, &length) != -1 &&
        client.sin_addr.s_addr != htonl (INADDR_ANY))
    {
      *result = client.sin_addr;
      KDEBUG (KDEBUG_INFO, 3900, "Found reply address");
      ::close (testsock);
      break;
    }
    if (testsock >= 0) ::close (testsock);
  }
  if (i == 2) {
    *result = defaultReplyAddr;
    KDEBUG (KDEBUG_INFO, 3900, "Couldn't find reply address, using default");
  }
  help1 = (unsigned char *) &destination;
  help2 = (unsigned char *) result;
  KDEBUG8 (KDEBUG_INFO, 3900,
           "detected reply address for %d.%d.%d.%d: %d.%d.%d.%d",
           help1 [0], help1 [1], help1 [2], help1 [3], 
           help2 [0], help2 [1], help2 [2], help2 [3]);
  replyAddrList.insert ((long) destination.s_addr, result);
  return *result;
}

QIntDict <in_addr> DaemonSocket::replyAddrList;
