// connect.cpp

#include <netdb.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/param.h>
#include <errno.h>

#include <kapp.h>

#include "connect.h"
#include "check_protocol.h"


Connect::Connect (QObject *parent, const char *name) : 
         QObject (parent, name) {
  status = st_fresh;
  connectionSocket = -1;
  connectionSocketNotify = 0;
  user = i18n ("nobody");
  announceID = inviteID = -1;  // no invitation or announcement waiting
}

Connect::~Connect () {
  cleanup ();
}

void Connect::cleanup () {
  killTimers ();
  if (inviteID != -1) {
    daemonMessage.type = DELETE;
    daemonMessage.id_num = htonl (inviteID);
    inviteID = -1;
    localSocket->send (daemonMessage);
  }
  if (announceID != -1) {
    daemonMessage.type = DELETE;
    daemonMessage.id_num = htonl (announceID);
    announceID = -1;
    remoteSocket->send (daemonMessage);
  }

  if (connectionSocketNotify) delete connectionSocketNotify;
  if (connectionSocket != -1) ::close (connectionSocket);
  connectionSocketNotify = 0;
  connectionSocket = -1;
}

bool Connect::establish (const char *address) {
  if (status != st_fresh)
    return FALSE;

  user = address;
  emit newUserName ();
  if (!analyseUserAddress ()) return FALSE;
  emit newUserName ();

  CheckProtocol *checker = new CheckProtocol (this);
  connect (checker, SIGNAL (timeout ()), SLOT (checkProtocolTimeout ()));
  connect (checker, SIGNAL (found (ProtocolType)), 
                    SLOT (checkProtocolFound (ProtocolType)));
  status = st_check_daemons;
  if (!checker->CheckHost (remoteMachineAddr)) {
    error (i18n ("Failed looking for talk daemons"));
    return FALSE;
  }

  return TRUE;
}

const char *Connect::getUserName () {
  return user.data ();
}

void Connect::checkProtocolTimeout () {
  if (status != st_check_daemons)
    KDEBUG (KDEBUG_WARN, 3900, "CheckProtocol Timeout occured, "
                               "although daemon already found!");
  error (i18n ("No talk daemon found"));
}

void Connect::checkProtocolFound (ProtocolType protocol) {
  remoteProtocol = protocol;

  // Create two sockets for communication with local and remote daemon
  localSocket = new DaemonSocket (localhostAddr, localProtocol);
  remoteSocket = new DaemonSocket (remoteMachineAddr, protocol);
  if (!localSocket->success () || !remoteSocket->success ()) {
    error (i18n ("Failed creating sockets for daemon communication"));
    return;
  }
  connect (localSocket, SIGNAL (receivedReply (NEW_CTL_RESPONSE)), 
                        SLOT (receivedLocalControl (NEW_CTL_RESPONSE)));
  connect (localSocket, SIGNAL (timeout ()), SLOT (daemonTimeout ()));
  connect (remoteSocket, SIGNAL (receivedReply (NEW_CTL_RESPONSE)), 
                         SLOT (receivedRemoteControl (NEW_CTL_RESPONSE)));
  connect (remoteSocket, SIGNAL (timeout ()), SLOT (daemonTimeout ()));
  

  statusMessage (i18n ("Checking for invitation on remote host"));
  status = st_wait_LOOKUP_resp;
  daemonMessage.type = LOOK_UP;
  if (!remoteSocket->sendReliable (daemonMessage))
    error (i18n ("Failed sending data to remote talk daemon"));
}

void Connect::receivedLocalControl (NEW_CTL_RESPONSE rp) {

  /* This slot is called on every message the control socket receives
     from the local talk demon. If it is of the correct talk version, the
     response is analysed in respect of what type of message is expected
  */
  switch (rp.type) {
    case LEAVE_INVITE: 
      if (status != st_wait_INVITE_resp) {
	KDEBUG (KDEBUG_WARN, 3900, "Received unexpected LEAVE_INVITE "
		"message from local daemon\nMaybe daemon is a little slow");
        return;
      }
      inviteID = ntohl (rp.id_num);

      // send an announce request
      if (reringCounter == 0)
        statusMessage (i18n ("Announcing the talk request at the remote host"));
      status = st_wait_ANNOUNCE_resp;
      daemonMessage.type = ANNOUNCE;
      daemonMessage.id_num = htonl (announceID);
      remoteSocket->sendReliable (daemonMessage);
      break;

    case DELETE: 
      break;             // No reaction necessary

    default:
      KDEBUG1 (KDEBUG_WARN, 3900, "Received message of unknown type"
	                          "%d from local demon", rp.type);
  }
}


void Connect::receivedRemoteControl (NEW_CTL_RESPONSE rp) {

  /* This slot is called on every message the control socket receives
     from the remote talk demon. If it is of the correct talk version, the
     response is analysed in respect of what type of message is expected
  */
  switch (rp.type) {
    case LOOK_UP:
      if (status != st_wait_LOOKUP_resp) {
	KDEBUG (KDEBUG_WARN, 3900, "Received unexpected LOOK_UP "
		"message from remote daemon\nMaybe daemon is a little slow");
        return;
      }

      if (rp.answer == SUCCESS && rp.addr.sin_family == htons (AF_INET)) {
        statusMessage (i18n ("Invitation found, waiting to connect"));

        // delete invitation, since it is no longer needed
        daemonMessage.type = DELETE;
        daemonMessage.id_num = rp.id_num;
        if (!remoteSocket->send (daemonMessage))
          return;

        // connect to the waiting socket
        connectionSocket = openSocket (&connectionSocketAddr, SOCK_STREAM);
        if (connectionSocket == -1) {
          error (i18n ("Failed creating a communication socket"));
          return;
        }

        errno = 0;
        rp.addr.sin_family = ntohs (rp.addr.sin_family);
        do {
          if (::connect(connectionSocket, (struct sockaddr *) &(rp.addr), 
              sizeof (rp.addr)) != -1) {
            killTimers ();
            status = st_connected;
            emit succeeded (connectionSocket);
            return; 
          }
        } while (errno == EINTR);

        // connection failed
        ::close (connectionSocket);
        connectionSocket = -1;
        if (errno == ECONNREFUSED) {
            statusMessage (i18n ("Your party was no longer waiting for us. "
                                 "Inviting him."));   
        } else {
            error (i18n ("Failed connecting to your partners socket"));
            return;
        }
      } else
        statusMessage (i18n ("No invitation waiting. Inviting him."));

      // create a new socket for connection
      connectionSocket = openSocket (&connectionSocketAddr, SOCK_STREAM);
      if (connectionSocket == -1 || listen (connectionSocket, 5) != 0) {
        error (i18n ("Failed creating a communication socket"));
        return;
      }

      connectionSocketNotify = new QSocketNotifier (connectionSocket, 
                                     QSocketNotifier::Read, this);
      connect (connectionSocketNotify, SIGNAL (activated (int)), 
                                       SLOT (receivedRemoteAnswer (int)));

      // insert socket address into daemon message
      daemonMessage.addr.sin_family = htons (AF_INET);
      daemonMessage.addr.sin_addr = remoteSocket->getClientIP ();
      daemonMessage.addr.sin_port = connectionSocketAddr.sin_port;

      // leaving invitation on local daemon
      status = st_wait_INVITE_resp;
      daemonMessage.type = LEAVE_INVITE;
      daemonMessage.id_num = htonl (inviteID);
      localSocket->sendReliable (daemonMessage);           
      reringCounter = 0;
      reringTimeoutID = startTimer (RING_WAIT * 1000);
      break;

    case ANNOUNCE: 
      if (status != st_wait_ANNOUNCE_resp) {
	KDEBUG (KDEBUG_WARN, 3900, "Received unexpected ANNOUNCE "
		"message from remote daemon\nMaybe daemon is a little slow");
        return;
      }
      if (rp.answer != SUCCESS) {
        switch (rp.answer) {
          case 1: error (i18n ("Your party is not logged on"));
                  break;
          case 2: error (i18n ("Target machine too confused"));
                  break;
          case 3: error (i18n ("Target machine does not recognize us"));
                  break;
          case 4: error (i18n ("Your party is refusing messages"));
                  break;
          case 5: error (i18n ("Target machine can't handle remote talk"));
                  break;
          case 6: 
          case 7:
          case 8: error (i18n ("Target machine has protocol problems"));
                  break;
          default:
                  error (i18n ("Unknown response from remote talk demon!"));
        }
        return;
      }
      announceID  = ntohl (rp.id_num);
      if (reringCounter == 0)
        statusMessage (i18n ("Waiting for your partner to respond"));
      status = st_wait_for_connect;
      break;

    case DELETE: 
      break;             // No reaction necessary

    default:
      KDEBUG1 (KDEBUG_WARN, 3900, "Received message of unknown type"
	                          "%d from remote demon", rp.type);
  }
}

void Connect::receivedRemoteAnswer (int) {
  killTimers ();

  volatile int newSocket;
  do {
    errno = 0;
    newSocket = accept (connectionSocket, 0, 0);
  } while (newSocket < 0 && errno == EINTR);
  if (newSocket < 0) {
    error (i18n ("Failed connecting to your partner"));
    return;
  }
  status = st_connected;
  emit succeeded (newSocket);

  cleanup ();
}

void Connect::timerEvent (QTimerEvent *event) {

  if (event->timerId () == reringTimeoutID) {
    reringCounter++;
    QString s;
    s.setNum (reringCounter);
    statusMessage (i18n ("Ringing again (") + s + ")");

    announceID++;
    daemonMessage.id_num = htonl (inviteID);
    status = st_wait_INVITE_resp;
    daemonMessage.type = LEAVE_INVITE;
    localSocket->sendReliable (daemonMessage);
  }
}

void Connect::daemonTimeout () {
  error (i18n ("Timeout: talk daemon doesn't respond"));
}

void Connect::error (const char *message) {
  status = st_failed;
  cleanup ();
  emit failed (message);
}

bool Connect::analyseUserAddress () {

  QString remoteName, remoteMachineName, remoteTty;

  /* split user name into login name, host name and tty */
  int pos_hash = user.findRev ('#');
  if (pos_hash != -1) {
    remoteTty = user.right (user.length () - pos_hash - 1);
    user.truncate (pos_hash);
  } else
    remoteTty = "";

  int pos_at = user.find ('@');
  if (pos_at == -1) {
    remoteMachineName = "localhost";
    remoteName = user;
  } else {
    remoteMachineName = user.right (user.length () - pos_at - 1);
    remoteName = user.left (pos_at);
  }

  //rebuild correct user address
  user = remoteName + '@' + remoteMachineName;
  if (!remoteTty.isEmpty ())
    user = user + '#' + remoteTty;

  /* Initialize the message template */
  daemonMessage = newDefaultMsg;
  remoteName.resize (NEW_NAME_SIZE);
  strcpy (daemonMessage.r_name, remoteName.data ());
  remoteTty.resize (TTY_SIZE);
  strcpy (daemonMessage.r_tty, remoteTty.data ());

  /* setup the IP address of the remote machine */
  if (remoteMachineName != localMachineName && 
      remoteMachineName != "localhost") {
    struct hostent *hp = gethostbyname (remoteMachineName.data ());
    if (!hp) {
      error (i18n ("No route to remote host!"));
      return FALSE;
    }
    memcpy (&remoteMachineAddr, hp->h_addr, hp->h_length);
  } else
    remoteMachineAddr = localhostAddr;

  return TRUE;
}

/*
 * Open a new socket, bind it to addr, 
 * with socket type 'type' (SOCK_STREAM or SOCK_DGRAM)
 * return the socket, or -1 if an error occured
 * The address is inserted into addr. addr does not need to be initialized.
 * ATTENTION: addr is used as long as the socket is used! So don't free
 *            its memory before closing the socket!
 */

int Connect::openSocket (sockaddr_in *addr, int type) {

  addr->sin_family = AF_INET;
  addr->sin_port = htons (0);
  addr->sin_addr.s_addr = htonl (INADDR_ANY);

  int newSocket = socket (PF_INET, type, 0);

  if (newSocket <= 0) {
    return -1;
  }

  ksize_t length = sizeof (*addr);
  if (bind (newSocket, (struct sockaddr *) addr, length) != 0) {
    ::close (newSocket);
    return -1;
  }

  if (getsockname (newSocket, (struct sockaddr *) addr, &length) == -1) {
    ::close (newSocket);
    return -1;
  }

  return newSocket;
}
